/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Test WxThread Part-2: array test
      
   Build: make chk_thread2
*/
#include "wxthread.h"
#include <cstdlib>

#include <iostream>
#if WXTHREAD_VERSION!=8
#error Test code is for WXTHREAD_VERSION 8
#endif

//
// The created threads use these two errmsg's to indicate its way of terminate.
// (exit errmsg for terminate by cancel is system set)
//
static const WxErrMsg LeftByReturn= WxErrMsg::enroll("Thread left by return");
static const WxErrMsg LeftByExit= WxErrMsg::enroll("Thread left by exit");

//
// thread_enter increase count
//
static WxMutex ent_cnt_mtx;
static int ent_cnt=0;    
//
// thread_leave increase count
//
static WxMutex lft_cnt_mtx;
static int lft_cnt=0; 

//
// The thread class to run the test
//
class TestThread2 : public WxThread {
  public:
   enum ReturnType { Never,Return, Exit };
    //
    // Construct object ready to run the thread with specified exit type and code
    //
    TestThread2(ReturnType type=Return) throw(std::exception) 
           : WxThread(),_type(type) {};
    ~TestThread2() throw(std::exception);
    WxRet reset(ReturnType type=Return) throw() 
           {
            WxRet r;
            if((r=WxThread::reset())!=OK) {
              WX_RETURN(r);
            }
            _type=type;
            return(OK);
           };
     ReturnType return_type(void) const throw()
           { return _type; };
  protected:
   virtual WxRet tmain(void) throw();
   virtual void tmain_enter(void) throw() 
           {
             WxMutex::lock(&ent_cnt_mtx);
             ++ent_cnt;
             WxMutex::unlock(&ent_cnt_mtx);
           };
   virtual void tmain_leave(void) throw()
           {
             WxMutex::lock(&lft_cnt_mtx);
             ++lft_cnt;
             WxMutex::unlock(&lft_cnt_mtx);
           };
  private:
    ReturnType _type;
};

TestThread2::~TestThread2() throw(std::exception)
{
 WxThread::cancel();
 WxThread::wait_stopped();
}

WxRet TestThread2::tmain(void) throw()
{
 Wx::sleep_till(Wx::now()+WxTime(0,50000000));  // sleep 0.05 second
 switch(_type) {
   case Return:
        return(LeftByReturn);
   case Exit:
        exit(LeftByExit);
   case Never:
        while(1) {
          Wx::sleep_till(Wx::now()+WxTime(0,50000000));  // sleep 0.05 second
          WxThread::yield();
        };
   default:
     WX_ABORT();
 };
};

//
// Get a random ReturnType
//
static TestThread2::ReturnType rand_type(void)
{
 switch(std::rand()%3) {
   case 0: return TestThread2::Return;
   case 1: return TestThread2::Exit;
   default: return TestThread2::Never;
 }
};

static const WxErrMsg TEST_FAIL=WxErrMsg::enroll("Test Failed");

//
// Check default thread object properties
//
static WxRet chk_default_attr(TestThread2& thrd)
{
 WxRet r;

 if(thrd.is_default()==false) {
   WX_RETURN(TEST_FAIL);
 }

 if(thrd.thread_state()!=WxThread::Ready) {
   WX_RETURN(TEST_FAIL);
 }

 if((r=thrd.wait_stopped())!=WXM_ENOENT) {
   WX_RETURN(TEST_FAIL);
 }
 if(!thrd.exit_code().is_default()) {
   WX_RETURN(TEST_FAIL);
 }
 return(OK);
};

//
// Test in Tight Loop
//
// The tight loop searches an available entry in thrTab[200] to create a thread.
// The thread is created to return/exit/cancel
//
static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxthread.cpp, tight loop...\n";

 //
 // Initial thread enter and leave count check
 //
 {
  bool init_cnt_chk=false;
  WxMutex::lock(&ent_cnt_mtx);
  WxMutex::lock(&lft_cnt_mtx);
  if((ent_cnt==0)&&(lft_cnt==0)) {
    init_cnt_chk=true;
  }
  WxMutex::unlock(&ent_cnt_mtx);
  WxMutex::unlock(&lft_cnt_mtx);
  if(init_cnt_chk==false) {
    WX_THROW( Wx_general_error() );
  }
 }

 const int NumOfThrds=200;      // 300 would get ENOMEM on my playform
 const int LoopCount=1000000;
 TestThread2 thrTab[NumOfThrds];
 WxRet r;

 //
 // Check default thread objects in the array
 //
 for(int i=0; i<NumOfThrds; ++i) {
   if((r=chk_default_attr(thrTab[i]))!=OK) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // The tight loop searches an available entry in thrTab[200] to create a thread.
 //
 std::cout << " Loop count= " << LoopCount << " ... wait\n"; 
 for(int cnt=0; cnt<LoopCount; ++cnt) {
   const int idx=std::rand()%NumOfThrds;
   TestThread2& thrd=thrTab[idx];
   WxThread::yield();
   switch(thrd.thread_state()) {
     case WxThread::Ready:
                   //
                   // Create a thread if the entry is in Ready state
                   //
                   if((r=thrd.reset(rand_type()))!=OK) {
                     WX_HERE(r); throw(r);
                   }
                   while((r=thrd.begin())!=OK) {
                     if(r==WXM_EAGAIN) {
                       continue;
                     }
                     WX_HERE(r); throw(r);
                   }
                   break;
     case WxThread::Running:
                   //
                   // Cancel the thread entry if it never exit.
                   //
                   switch(thrd.return_type()) {
                     case TestThread2::Return:  // FALLTHROUGH
                     case TestThread2::Exit:
                                  break;
                     case TestThread2::Never:
                                  if((r=thrd.cancel())!=OK) {
                                    if(r!=WXM_ENOENT) {  // gap exists while gets here
                                      WX_HERE(r); throw(r);
                                    }
                                  }
                                  break;
                     default:
                          WX_THROW( Wx_general_error() );
                   }
                   break;
     case WxThread::Terminated:
                   //
                   // Check exit errmsg and reset
                   //
                   switch(thrd.return_type()) {
                     case TestThread2::Return:
                                  if((r=thrd.exit_code())!=LeftByReturn) {
                                    WX_HERE(r); throw(r);
                                  }
                                  break;
                     case TestThread2::Exit:
                                  if((r=thrd.exit_code())!=LeftByExit) {
                                    WX_HERE(r); throw(r);
                                  }
                                  break;
                     case TestThread2::Never:
                                  if((r=thrd.exit_code())!=WXM_THRDCAN) {
                                    WX_HERE(r); throw(r);
                                  }
                                  break;
                     default:
                          WX_THROW( Wx_general_error() );
                   }
                   if((r=thrd.reset())!=OK) {
                     WX_HERE(r); throw(r);
                   }
                   break;
     default:
          WX_THROW( Wx_general_error() );   // should not get here
   }
 }

 //
 // Count the running threads. They should be the ones configured as TestThread2::Never
 //
 // But before the checking, wait 0.1 second for all the threads that are configured 
 // TestThread2::Return or TestThread2::Exit to exit.
 //
 Wx::sleep_till(Wx::now()+WxTime(0,100000000));  // sleep 0.1 second
 int running_count=0;
 for(int i=0; i<NumOfThrds; ++i) {
   TestThread2& thrd=thrTab[i];
   switch(thrd.thread_state()) {
     case WxThread::Ready:
     case WxThread::Terminated: /* FALLTHROUGH */
          break;
     case WxThread::Running: 
          if(thrd.return_type()!=TestThread2::Never) {
            WX_THROW( Wx_general_error() );
          }
          ++running_count;
          break;
     default:
          WX_THROW( Wx_general_error() );   // should not get here
   }
 }
 
 {
   WxLock a1(ent_cnt_mtx);
   WxLock a2(lft_cnt_mtx);
   if(ent_cnt!=running_count+lft_cnt) {
     WX_THROW( Wx_general_error() );
   }
 /* 
   // As the array entries are not many, statistics showed all threads had
   // terminated, so this test is not enough.
   //  (My machine fails to have more than 255 threads)
   //
   std::cout << "ent: " << ent_cnt << std::endl;
   std::cout << "lft: " << lft_cnt << std::endl;
   std::cout << "run: " << running_count << std::endl;
 */
 }
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
