/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Test WxThread Part-I

   test_thread1 Basic public interface check

   Build: make chk_thread1
*/
#include "wxthread.h"
#include <cstdlib>

#include <iostream>
#if WXTHREAD_VERSION!=8
#error Test code is for WXTHREAD_VERSION 8
#endif

//
// The thread class to run the test
//
class TestThread1 : public WxThread {
  public:
   enum ReturnType { Never,Return, Exit };
    //
    // Construct object ready to run the thread with specified exit type and code
    //
    TestThread1(ReturnType type,WxErrMsg emsg) throw(std::exception) 
           : WxThread(),_entered(0),_left(0),_type(type),_emsg(emsg) {};
    ~TestThread1() throw(std::exception);
    WxRet reset(ReturnType type,WxErrMsg emsg) throw() 
           {
            WxRet r;
            if((r=WxThread::reset())!=OK) {
              WX_RETURN(r);
            }
            _emsg=emsg;
            _type=type;
            _entered=0;
            _left=0;
            return(OK);
           };
    int enter_count(void) const throw() { return _entered; };
    int leave_count(void) const throw() { return _left; };
  protected:
   virtual WxRet tmain(void) throw();
   virtual void tmain_enter(void) throw() { ++_entered; };
   virtual void tmain_leave(void) throw() { ++_left; };
  private:
    int _entered,_left;
    ReturnType _type;
    WxErrMsg _emsg;
};

TestThread1::~TestThread1() throw(std::exception)
{
 WxThread::cancel();
 WxThread::wait_stopped();
}

WxRet TestThread1::tmain(void) throw()
{
 if(_entered!=1) {
   WX_ABORT();
 }
/*
 {
 #if defined(_POSIX_THREAD_PRIORITY_SCHEDULING)
   int s_policy;
   struct sched_param s_param;
   int v=::pthread_getschedparam(::pthread_self(),&s_policy,&s_param);
   if(v!=0) {
     WX_RETURN(WxErrMsg(v));
   }
   std::cout << "Thread[" << ::pthread_self() << "]=";
   switch( s_policy) {
     case SCHED_FIFO:
                std::cout << " SCHED_FIFO"; 
                break;
     case SCHED_RR:
                std::cout << " SCHED_RR"; 
                break;
     case SCHED_OTHER:
                std::cout << " SCHED_OTHER"; 
                break;
     default:
                std::cout << " SCHED_(unknown)"; 
                break;
   };
   std::cout << "  priority=" << s_param.sched_priority << std::endl;
 #else
   std::cout << "_POSIX_THREAD_PRIORITY_SCHEDULING not defined\n";
 #endif
 }*/

 switch(_type) {
   case Return:
        return(_emsg);
   case Exit:
        exit(_emsg);
   case Never:
        while(1) {
          Wx::sleep_till(Wx::now()+WxTime(0,10000000));  // sleep 1/100 second
          WxThread::yield();
        };
   default:
     WX_ABORT();
 };
};

static TestThread1::ReturnType rand_type(void)
{
 switch(std::rand()%3) {
   case 0: return TestThread1::Return;
   case 1: return TestThread1::Exit;
   default: return TestThread1::Never;
 }
};

static const WxErrMsg TEST_FAIL=WxErrMsg::enroll("Test Failed");

//
// Check default thread object properties
//
static WxRet chk_default_attr(TestThread1& thrd)
{
 WxThread::ThreadState st;
 WxRet tret;
 WxRet r;

 if(thrd.is_default()==false) {
   WX_RETURN(TEST_FAIL);
 }

 st=thrd.thread_state();
 if(st!=WxThread::Ready) {
   WX_RETURN(TEST_FAIL);
 }

 if((r=thrd.wait_stopped())!=WXM_ENOENT) {
   WX_RETURN(TEST_FAIL);
 }
 tret=thrd.exit_code();
 if(!tret.is_default()) {
   WX_RETURN(TEST_FAIL);
 }
 if(thrd.enter_count()!=0) {
   WX_RETURN(TEST_FAIL);
 }
 if(thrd.leave_count()!=0) {
   WX_RETURN(TEST_FAIL);
 }
 return(OK);
};

static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxthread.cpp, basic ...\n";

 TestThread1 t1(TestThread1::Return,WxErrMsg(20));
 WxThread::ThreadState st;
 WxRet tec;      // thread exit code
 WxRet r;
 
 //
 // Check initial number of threads should be 0
 //
 if(WxThread::total_threads()!=0) {
   WX_THROW( Wx_general_error() );
 }

 //
 // Test the thread (terminate by return)
 //
 if((r=chk_default_attr(t1))!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=t1.begin())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=t1.wait_nready())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=t1.begin())!=WXM_EBADF) {
   WX_HERE(r); throw(r);
 }
 if(t1.thread_state()==WxThread::Ready) {
   WX_HERE(r); throw(r);
 }
 if((r=t1.wait_stopped())!=OK) {
   WX_HERE(r); throw(r);
 }
 st=t1.thread_state();
 if(st!=WxThread::Terminated) {
   WX_THROW( Wx_general_error() );
 }
 if(t1.enter_count()!=1) {
   WX_THROW( Wx_general_error() );
 }
 if(t1.leave_count()!=1) {
   WX_THROW( Wx_general_error() );
 }
 tec=t1.exit_code();
 if(tec!=WxErrMsg(20)) {
   WX_THROW( Wx_general_error() );
 }
 if(WxThread::total_threads()!=0) {
   WX_THROW( Wx_general_error() );
 }

 //
 // Test the thread (terminate by exit)
 //
 if((r=t1.reset(TestThread1::Exit,WxErrMsg(44)))!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=chk_default_attr(t1))!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=t1.begin())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=t1.wait_nready())!=OK) {
   WX_HERE(r); throw(r);
 }
 if(t1.thread_state()==WxThread::Ready) {
   WX_HERE(r); throw(r);
 }
 if((r=t1.wait_stopped())!=OK) {
   WX_HERE(r); throw(r);
 }
 st=t1.thread_state();
 if(st!=WxThread::Terminated) {
   WX_THROW( Wx_general_error() );
 }
 if(t1.enter_count()!=1) {
   WX_THROW( Wx_general_error() );
 }
 if(t1.leave_count()!=1) {
   WX_THROW( Wx_general_error() );
 }
 tec=t1.exit_code();
 if(tec!=WxErrMsg(44)) {
   WX_THROW( Wx_general_error() );
 }
 if(WxThread::total_threads()!=0) {
   WX_THROW( Wx_general_error() );
 }

 //
 // Test the thread in loop (terminate by cancel) 
 //
 for(int i=0; i<100; ++i) {
   TestThread1 t2(TestThread1::Never,WxErrMsg(23));
   if((r=chk_default_attr(t2))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=t2.begin())!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=t2.cancel())!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=t2.wait_stopped())!=OK) {
     WX_HERE(r); throw(r);
   }
   st=t2.thread_state();
   if(st!=WxThread::Terminated) {
     WX_THROW( Wx_general_error() );
   }
   if(t2.enter_count()!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(t2.leave_count()!=1) {
     WX_THROW( Wx_general_error() );
   }
   tec=t2.exit_code();
   if(tec!=WXM_THRDCAN) {
     WX_THROW( Wx_general_error() );
   }
   if(WxThread::total_threads()!=0) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // Test the thread in loop (random exit type) 
 //
 for(int i=0; i<100; ++i) {
   TestThread1 td(rand_type(),WxErrMsg(20));
   WxRet r;
 
   if(WxThread::total_threads()!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=chk_default_attr(td))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=td.begin())!=OK) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // Test the thread in loop (terminate by return, but may be cancelled by descructor) 
 //
 for(int i=0; i<100; i++) {
   TestThread1 t1(TestThread1::Return,WxErrMsg(20));
   WxRet r=t1.begin();
   if(r!=OK) {
    WX_THROW( Wx_general_error() );
   }
 };

 //
 // Test the thread in loop (terminate by exit, but may be cancelled by descructor) 
 //
 for(int i=0; i<100; i++) {
   TestThread1 t1(TestThread1::Exit,WxErrMsg(20));
   WxRet r=t1.begin();
   if(r!=OK) {
    WX_THROW( Wx_general_error() );
   }
 };

 //
 // Test the thread in loop (terminate by cancel, but may be cancelled by descructor) 
 //
 for(int i=0; i<100; i++) {
   TestThread1 t1(TestThread1::Never,WxErrMsg(20));
   WxRet r=t1.begin();
   if(r!=OK) {
    WX_THROW( Wx_general_error() );
   }
 };

 //
 // There should be none thread presently
 //
 if(WxThread::total_threads()!=0) {
    WX_THROW( Wx_general_error() );
 }
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
