/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Check WxTerminal public members

   Build: make chk_terminal

   Note: Test data files should exist in the working directory
         Files may be created in the working directory and unlinked
*/

#include "wxterminal.h"
#include <sys/types.h>
#include <unistd.h>
#include <sys/wait.h>

#if WXTERMINAL_VERSION!=8
#error Test code is for WXTERMINAL_VERSION 8
#endif

static const WxPathName RefFileName("/dev/tty");

static long baud_rate(speed_t sp)
{
 switch(sp) {
   case B0:  return(0);
   case B50:  return(50);
   case B75:  return(75);
   case B134: return(134); // 134.5
   case B150: return(150);
   case B200: return(200);
   case B300: return(300);
   case B600: return(600);
   case B1200: return(1200);
   case B1800: return(1800);
   case B2400: return(2400);
   case B4800: return(4800);
   case B9600: return(9600);
   case B19200: return(9200);
   case B38400: return(38400);
   case B57600: return(57600);
   case B115200: return(115200);
   case B230400: return(230400);
   default:
     return(-1);
 }
};

static void dump_termios(const WxPathName fname,const WxTermios& tio)
{
  std::cout << "WxTerminal file = " << fname.pathname() << std::endl;
  std::cout << "    ispeed= " << baud_rate(tio.getispeed()) << std::endl;
  std::cout << "    ospeed= " << baud_rate(tio.getospeed()) << std::endl;
};

//
// Assert ff is default
//
static void assert_default(const WxTerminal& ff)
{
 WxRet r;

 if(ff.is_default()==false) {
   WX_THROW( Wx_general_error() );
 }
 if(ff.fh().is_default()==false) {
   WX_THROW( Wx_general_error() );
 }

 // stat
 {
   WxFileStat stt;
   if((r=ff.stat(stt))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Setup the test 
//
static void setup_test(void)
{
};

//
// Undo setup_test
//
static void close_test(void)
{
};

//
// Check basic operations on default object
//
static void t1(void)
{
 WxRet r;

 WxTerminal NObj;     // non-default object for test as argument
 assert_default(NObj);
 if((r=NObj.open(RefFileName,O_RDONLY))!=OK) {
   WX_HERE(r); throw(r);
 }
 
 // stat
 {
   WxFileStat stt;
   if((r=NObj.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_chr()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 // constructor
 {
   WxTerminal tmp0;
   WxTerminal tmp1(tmp0);
   assert_default(tmp0);
   assert_default(tmp1);

   WxTerminal tmp2(NObj);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp2.fh()==NObj.fh()) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset
 {
   WxTerminal tmp;
   assert_default(tmp);
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   WxTerminal tmp2(NObj);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp2.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp2);
 }

 // read,read_max,write,drain
 {
   WxTerminal tmp;
   assert_default(tmp);
   char buf[2];
   if((r=tmp.read(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.max_read(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.write(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.drain())!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 //
 // assignf(WxFileHandle)
 //
 {
   WxTerminal tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxFileHandle()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(NObj.fh()))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==NObj.fh()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxFileHandle()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxTerminal)
 //
 {
   WxTerminal tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxTerminal()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(NObj))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==NObj.fh()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxTerminal()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // sendbreak,flush,flow
 //
 {
   WxTerminal tmp;
   assert_default(tmp);
   if((r=tmp.sendbreak(0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.flush(TCIFLUSH))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.flow(TCOOFF))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 //
 // getattr,setattr
 //
 {
   WxTerminal tmp;
   WxTermios tio;
   assert_default(tmp);
   if(tio.is_default()==false) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.getattr(tio))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
   if(tio.is_default()==false) {
     //WX_THROW( Wx_general_error() );
     std::cerr << "::tcgetattr modified its argument in failure? (EBADF)\n";
     tio.reset();
   }

   if((r=tmp.setattr(tio,TCSANOW))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
   if(tio.is_default()==false) {
     WX_THROW( Wx_general_error() );
   }
   assert_default(tmp);
 }

 // ttyname
 {
   WxTerminal tmp;
   WxPathName pn;
   if((r=tmp.devname(pn))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
   if(pn.is_default()==false) {
     WX_THROW( Wx_general_error() );  // pn modified
   }
 }
 
 //
 // disabled function
 //
 {
   WxTerminal t0;
   WxChrFile c0,*p=&t0;
   if((r=p->assignf(c0))!=WXM_VFDIS) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Check basic operations on non-default object
//
static void t2(void)
{
 WxRet r;
 
 // stat NObj
 WxTerminal NObj;     // non-default object for the test
 assert_default(NObj);
 if((r=NObj.open(RefFileName,O_RDWR))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(NObj.is_default()) {
   WX_THROW( Wx_general_error() );
 }
 
 // stat
 {
   WxFileStat stt;
   if((r=NObj.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_chr()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 // constructor
 {
   WxTerminal tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==NObj.fh()) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset
 {
   WxTerminal tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 // (zero byte) read,read_max,write,drain
 {
   WxTerminal tmp(NObj);
   char buf[2]={-1,-1};
   size_t n_rd=5;   // anything >1
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.max_read(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.write(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.drain())!=OK) {
     WX_HERE(r); throw(r);
   }
 }

 // open (WXM_NDEFAULT)
 {
   WxTerminal tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.open(RefFileName,O_RDONLY))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxFileHandle())
 //
 {
   WxTerminal tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxFileHandle()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxTerminal())
 //
 {
   WxTerminal tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxTerminal()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // sendbreak,flush,flow
 //
 {
   WxTerminal tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   
   if((r=tmp.sendbreak(0))!=OK) {
     WX_HERE(r); throw(r);
   }

   if((r=tmp.flush(TCOFLUSH))!=OK) {
     WX_HERE(r); throw(r);
   }

   if((r=tmp.flow(TCOON))!=OK) {
     WX_HERE(r); throw(r);
   }
 }

 // 
 // getattr,setattr
 //
 {
   WxTerminal tmp(NObj);
   WxTermios tio;
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tio.is_default()==false) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.getattr(tio))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tio.is_default()) {
     WX_THROW( Wx_general_error() );
   }

   WxTermios tio_org(tio);
   dump_termios(RefFileName,tio);

   if((r=tmp.setattr(tio_org,TCSANOW))!=OK) { // write the same back
     WX_HERE(r); throw(r);
   }
   if((r=tmp.getattr(tio))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tio!=tio_org) {
     WX_THROW( Wx_general_error() );
   }
 }

 // ttyname
 {
   WxTerminal tmp(NObj);
   WxPathName pn;
   if((r=tmp.devname(pn))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(pn.pathname()!=RefFileName.pathname()) {
     std::cerr << "devname()=" << pn.pathname()<< std::endl;
     WX_THROW( Wx_general_error() );
   }
 }
};

static void t_exception(void)
{
 //
 // Test copy constructor exception
 //
 try {
   char tbuf[sizeof(WxTerminal)];
   std::memset(tbuf,-2,sizeof(tbuf));
   //
   // as -2 is known not default fd and no such valid handle. 
   // construct from tbuf should cause Fault exception
   //
   WxTerminal tt2(reinterpret_cast<const WxTerminal&>(tbuf));
   WX_THROW( Wx_general_error() );  // no expected exception
 }
 catch(const WxTerminal::Fault& e) {
   if(e!=WXM_EBADF) {
     WxRet r=e;
     WX_HERE(r); throw(r);
   }
   // FALLTHROUGH 
 }
 catch(...) {
   throw;
 };

 //
 // Test destructor exception
 //
 {
   WxTerminal tmp;
   std::memset(&tmp,-2,sizeof(tmp));
   // destructor of tmp is now not to cause exception for this setting
 }
};

static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxterminal.h ...\n";
 if(std::strcmp(WxTerminal::class_name,"WxTerminal")!=0) {
   WX_THROW( Wx_general_error() );
 }
 try {
   setup_test();
   t1();
   t2();
   t_exception();
   close_test();
 }
 catch(...) {
   close_test();
   throw;
 }
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
