/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2004)

   Build: chk_str
*/

#include "wxstr.h"

#if WXSTR_VERSION!=8
#error Test code is for WXPATHNAME_VERSION 8
#endif

static bool chk_default(const WxStr& str)
{
 static const WxStr S0;
 if(str.is_default()==false) {
   return(false);
 }
 if(str==S0) {
 } else {
   return(false);
 }
 if(str!=S0) {
   return(false);
 }
 if(str.size()!=0) {
   return(false);
 }
 const char* cptr;
 cptr=str.data();
 if(cptr==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(*cptr!=0) {
   return(false);
 }
 cptr=str.c_str();
 if(cptr==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(*cptr!=0) {
   return(false);
 }
 if(str.std_string()!=std::string("")) {
   return(false);
 }
 return(true);
}

static bool chk_not_default(const WxStr& str)
{
 static const WxStr S0;
 if(str.is_default()) {
   return(false);
 }
 if(str!=S0) {
 } else {
   return(false);
 }
 if(str==S0) {
   return(false);
 }
 if(str.size()<=0) {
   return(false);
 }
 const char* cptr;
 cptr=str.data();
 if(cptr==0) {
   WX_THROW( Wx_general_error() ); 
 }
 cptr=str.c_str();
 if(cptr==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(str.std_string()==std::string("")) {
   return(false);
 }
 return(true);
};

static bool chk_equ(const WxStr& str1,const WxStr& str2)
{
 if(str1==str2) {
 } else {
   return(false);
 }
 if(str1!=str2) {
   return(false);
 }

 if(str1.size()!=str2.size()) {
   return(false);
 }

 const char* cptr1,*cptr2;

 cptr1=str1.data();
 cptr2=str2.data();
 if(cptr1==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(cptr2==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(::memcmp(cptr1,cptr2,str1.size())!=0) {
   return(false);
 }

 cptr1=str1.c_str();
 cptr2=str2.c_str();
 if(cptr1==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(cptr2==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(::memcmp(cptr1,cptr2,str1.size())!=0) {
   return(false);
 }

 if(str1.std_string()!=str2.std_string()) {
   return(false);
 }
 return(true);
};

static bool chk_equ(const WxStr& str1,const char* cstr)
{
 if(cstr==0) {
   return(false);
 }
 if(chk_equ(str1,WxStr(cstr))==false) {
   return(false);
 }
 if(::memcmp(str1.c_str(),cstr,str1.size()+1)!=0) {
   return(false);
 }
 return(true);
};

static bool chk_not_equ(const WxStr& str1,const WxStr& str2)
{
 if(str1==str2) {
 } else {
   return(false);
 }
 if(str1!=str2) {
   return(false);
 }

 if(str1.size()!=str2.size()) {
   return(false);
 }

 const char* cptr1,*cptr2;

 cptr1=str1.data();
 cptr2=str2.data();
 if(cptr1==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(cptr2==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(::memcmp(cptr1,cptr2,str1.size())!=0) {
   return(false);
 }

 cptr1=str1.c_str();
 cptr2=str2.c_str();
 if(cptr1==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(cptr2==0) {
   WX_THROW( Wx_general_error() ); 
 }
 if(::memcmp(cptr1,cptr2,str1.size())!=0) {
   return(false);
 }

 if(str1.std_string()!=str2.std_string()) {
   return(false);
 }

 // not distinguishible
 return(true);
};

#define CHK_EQU(expr1,expr2) if(chk_equ(expr1,expr2)==false) { WX_THROW( Wx_general_error() ); };
#define CHK_NOT_EQU(expr1,expr2) if(chk_not_equ(expr1,expr2)==false) { WX_THROW( Wx_general_error() ); };

#define CHK_DEFAULT(expr) if(chk_default(expr)==false) { WX_THROW( Wx_general_error() ); };
#define CHK_NOT_DEFAULT(expr) if(chk_not_default(expr)==false) { WX_THROW( Wx_general_error() ); };
//
// Test WxStr (basic, ops of default object)
//
static void t1(void)
{
 WxRet r;
 const WxStr a0;
 const WxStr a1(a0);

 chk_default(a0);
 chk_default(a1);

 // constructors
 {
  const WxStr Nobj("abc");

  {
   WxStr tmp0(a0);

   chk_default(tmp0);
   CHK_EQU(tmp0,a0);

   WxStr tmp1(Nobj);
   CHK_EQU(tmp1,Nobj);
   chk_not_equ(tmp1,a0);
  }

  {
   WxStr tmp1(Nobj,1,1);
   CHK_EQU(tmp1,"b");
   WxStr tmp2(Nobj,1,2);
   CHK_EQU(tmp2,"bc");
   WxStr tmp3(Nobj,1,3);
   CHK_EQU(tmp3,"bc");
   WxStr tmp4(Nobj,1,4);
   CHK_EQU(tmp4,"bc");
  }

  {
   WxStr tmp("xyz");
   CHK_EQU(tmp,"xyz");
  }

  {
   WxStr tmp("xyz",2);
   CHK_EQU(tmp,"xy");
  }

  {
   WxStr tmp(3,'a');
   CHK_EQU(tmp,"aaa");
  }
 }

 {
  const std::string str("abc");
  WxStr tmp(str);
  CHK_EQU(tmp,str);
 }

 // reset()
 {
  WxStr tmp;
  tmp.reset();
  chk_default(tmp);
 }

 // at(size_t) const version
 {
   WxStr tmp("abc");
   if(tmp.at(0)!='a') {
     WX_THROW( Wx_general_error() ); 
   }
   if(tmp.at(1)!='b') {
     WX_THROW( Wx_general_error() ); 
   }
   if(tmp.at(2)!='c') {
     WX_THROW( Wx_general_error() ); 
   }
 }

 // at(size_t) non-const version
 {
   WxStr tmp("abc");
   if((char&)tmp.at(0)!='a') {
     WX_THROW( Wx_general_error() ); 
   }
   if((char&)tmp.at(1)!='b') {
     WX_THROW( Wx_general_error() ); 
   }
   if((char&)tmp.at(2)!='c') {
     WX_THROW( Wx_general_error() ); 
   }
 }

 // data()/c_str/std_string
 {
  WxStr tmp;
  const char* ptr1=tmp.data();   // just make sure exist
  const char* ptr2=tmp.c_str();
  if(ptr1==0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(*ptr1!=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(ptr2==0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(*ptr2!=0) {
    WX_THROW( Wx_general_error() ); 
  }
 }

 // assign
 {
  const WxStr NObj("abc");
  WxStr tmp;
  
  chk_default(a0);
  chk_default(tmp);

  tmp.assign(a0);
  chk_default(tmp);

  tmp.assign(NObj);
  CHK_EQU(tmp,NObj);

  tmp.reset();
  chk_default(tmp);

  if((r=tmp.assign(NObj,0,0))!=OK) {
    WX_THROW(r);
  }
  chk_default(tmp);

  if((r=tmp.assign(NObj,0,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"a");

  if((r=tmp.assign(NObj,1,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"b");

  if((r=tmp.assign(NObj,1,2))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"bc");

  tmp.reset();
  chk_default(tmp);

  const char* CStr="xyz";
  
  if((r=tmp.assign(CStr))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,CStr);

  tmp.reset();
  chk_default(tmp);

  if((r=tmp.assign(CStr,0))!=OK) {
    WX_THROW(r);
  }
  chk_default(tmp);

  if((r=tmp.assign(CStr,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"x");

  if((r=tmp.assign(CStr,2))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"xy");

  tmp.reset();
  chk_default(tmp);

  tmp.assign(3,'x');
  CHK_EQU(tmp,"xxx");

  tmp.assign(0,'x');
  chk_default(tmp);
 }

 // swap
 {
  WxStr a("56");
  WxStr b("ijk");

  a.swap(b);
  if((a!="ijk")||(b!="56")) {
    WX_THROW( Wx_general_error() ); 
  }
  b.swap(a);
  if((a!="56")||(b!="ijk")) {
    WX_THROW( Wx_general_error() ); 
  }
 }

 // operator =
 {
  WxStr NObj("st");
  WxStr tmp;

  tmp=NObj;
  CHK_EQU(tmp,NObj);

  tmp=a0;
  CHK_EQU(tmp,a0);
  chk_default(tmp);
  
  const char* CStr="trh";
  tmp=CStr;
  CHK_EQU(tmp,CStr);

  tmp='g';
  CHK_EQU(tmp,"g");
 }

 // append(const WxStr&)
 {
  WxStr NObj("st");
  WxStr tmp;

  tmp.append(a0);
  chk_default(tmp);

  tmp.append(NObj);
  CHK_EQU(tmp,NObj);

  tmp.append(NObj);
  CHK_EQU(tmp,"stst");

  tmp.reset();
  tmp+=a0;
  chk_default(tmp);

  tmp+=NObj;
  CHK_EQU(tmp,NObj);

  tmp+=NObj;
  CHK_EQU(tmp,"stst");
  tmp.reset();
 }

 // append(const WxStr&,size_t,size_t)
 {
  WxStr NObj("rst");
  WxStr tmp;

  tmp.append(a0);
  chk_default(tmp);

  if((r=tmp.append(NObj,0,0))!=OK) {
    WX_THROW(r);
  }
  chk_default(tmp);

  if((r=tmp.append(NObj,0,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"r");

  if((r=tmp.append(NObj,1,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"rs");

  if((r=tmp.append(NObj,1,2))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"rsst");

  if((r=tmp.append(NObj,1,4))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"rsstst");

  if((r=tmp.append(NObj,NObj.size(),3))!=OK) {   // idx==NObj.size() is ok
    WX_THROW(r);
  }
  CHK_EQU(tmp,"rsstst");

  if((r=tmp.append(NObj,NObj.size()+1,3))!=WXM_EINVAL) { // idx>NObj.size() is error
    WX_THROW(r);
  }
 }

 // append(const char*)
 {
  const char* CStr="st";
  WxStr tmp;

  chk_default(tmp);

  if((r=tmp.append(CStr))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,CStr);

  if((r=tmp.append(CStr))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"stst");

  if((r=tmp.append(""))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"stst");

  tmp.reset();
  tmp+=a0;
  chk_default(tmp);

  tmp+=CStr;
  CHK_EQU(tmp,CStr);

  tmp+=CStr;
  CHK_EQU(tmp,"stst");

  if((r=tmp.append(""))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"stst");
  tmp.reset();
 }

 // append(const char*,size_t)
 {
  const char* CStr="stu";
  WxStr tmp;

  chk_default(tmp);
  if((r=tmp.append(CStr,::strlen(CStr)))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,CStr);

  if((r=tmp.append(CStr,2))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"stust");

  if((r=tmp.append(CStr,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"stusts");

  if((r=tmp.append(""))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"stusts");

  tmp.reset();
 }

 // append(size_t,char)
 {
  WxStr tmp;

  chk_default(tmp);

  tmp.append(0,'a');
  CHK_EQU(tmp,"");

  tmp.append(2,'a');
  CHK_EQU(tmp,"aa");

  tmp.append(1,'b');
  CHK_EQU(tmp,"aab");
 }

 // append(char)
 {
  WxStr tmp;

  chk_default(tmp);

  tmp.append('a');
  CHK_EQU(tmp,"a");

  tmp.append('b');
  CHK_EQU(tmp,"ab");

  tmp.reset();
  chk_default(tmp);

  tmp+='a';
  CHK_EQU(tmp,"a");

  tmp+='b';
  CHK_EQU(tmp,"ab");
 }

 // insert(size_t,const WxStr&)
 {
  WxStr NObj("ab");
  WxStr tmp;

  chk_default(tmp);
  if((r=tmp.insert(0,a0))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,a0);

  if((r=tmp.insert(0,NObj))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"ab");

  if((r=tmp.insert(1,NObj))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"aabb");

  if((r=tmp.insert(5,NObj))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // insert(size_t,const WxStr&,size_t,size_t)
 {
  WxStr NObj("ab");
  WxStr tmp;

  chk_default(tmp);
  if((r=tmp.insert(0,a0,0,5))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,a0);

  if((r=tmp.insert(0,NObj,0,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"a");

  if((r=tmp.insert(1,NObj,1,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"ab");

  if((r=tmp.insert(1,NObj,1,5))!=OK) {  // at most
    WX_THROW(r);
  }
  CHK_EQU(tmp,"abb");

  if((r=tmp.insert(4,NObj,1,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"abb");
 }

 // insert(size_t,const char*)
 {
  const char* CStr="ab";
  WxStr tmp;

  chk_default(tmp);
  if((r=tmp.insert(0,""))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,a0);

  if((r=tmp.insert(0,CStr,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"a");

  if((r=tmp.insert(0,CStr,2))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"aba");

  if((r=tmp.insert(2,CStr,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"abaa");
  if((r=tmp.insert(tmp.size()+1,CStr,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // insert(size_t,size_t,char)
 {
  WxStr tmp;

  chk_default(tmp);
  if((r=tmp.insert(0,0,'a'))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,a0);

  if((r=tmp.insert(0,2,'a'))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"aa");

  if((r=tmp.insert(2,2,'b'))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"aabb");

  if((r=tmp.insert(3,1,'x'))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"aabxb");

  if((r=tmp.insert(6,1,'x'))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // erase(size_t)
 {
  WxStr tmp("12345");

  if((r=tmp.erase(tmp.size()+1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.erase(tmp.size()))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"12345");

  if((r=tmp.erase(4))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"1234");

  if((r=tmp.erase(2))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"12");

  if((r=tmp.erase(0))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"");

 }

 // erase(size_t,size_t)
 {
  WxStr tmp("12345");

  if((r=tmp.erase(tmp.size()+1,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.erase(tmp.size(),1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"12345");

  if((r=tmp.erase(4,0))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"12345");

  if((r=tmp.erase(1,3))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"15");
 }

 // resize(size_t)
 {
  WxStr tmp;

  tmp.resize(0);
  tmp.resize(2);
  if((tmp[0]!=0)||(tmp[1]!=0)) {
    WX_THROW( Wx_general_error() );    // default filler not zero
  }
 }

 // resize(size_t,char)
 {
  WxStr tmp;

  tmp.resize(0,'a');
  if(tmp.size()!=0) {
    WX_THROW( Wx_general_error() ); 
  }

  tmp.resize(2,'b');
  CHK_EQU(tmp,"bb");

  tmp.resize(4,'a');
  CHK_EQU(tmp,"bbaa");
 }

 // replace(size_t,size_t,const WxStr&)
 {
  const WxStr NObj("xyz");
  WxStr tmp;

  if((r=tmp.replace(1,2,NObj))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.replace(0,5,NObj))!=OK) {  // default WxStr takes any len
    WX_THROW(r);
  }
  CHK_EQU(tmp,"xyz");

  if((r=tmp.replace(1,1,NObj))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"xxyzz");
 }

 // replace(size_t,size_t,const WxStr&,size_t,size_t)
 {
  const WxStr NObj("xyz");
  WxStr tmp;

  if((r=tmp.replace(2,2,NObj,0,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.replace(0,2,NObj,4,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.replace(0,5,NObj,0,4))!=OK) {  // default WxStr takes any len
    WX_THROW(r);
  }
  CHK_EQU(tmp,"xyz");
 }

 // replace(size_t,size_t,const char*)
 {
  const char* CStr="xyz";
  WxStr tmp;

  if((r=tmp.replace(1,2,CStr))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.replace(0,5,CStr))!=OK) {  // default WxStr takes any len
    WX_THROW(r);
  }
  CHK_EQU(tmp,"xyz");

  if((r=tmp.replace(1,1,CStr))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"xxyzz");
 }

 // replace(size_t,size_t,const char*,size_t)
 {
  const char* CStr="xyz";
  WxStr tmp;

  if((r=tmp.replace(2,2,CStr,0))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.replace(0,2,CStr,1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"x");

  if((r=tmp.replace(0,5,CStr,::strlen(CStr)))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,CStr);

  if((r=tmp.replace(1,1,CStr,::strlen(CStr)))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"xxyzz");
 }

 // replace(size_t,size_t,size_t,char)
 {
  WxStr tmp;

  if((r=tmp.replace(1,2,0,'a'))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  chk_default(tmp);

  if((r=tmp.replace(0,2,2,'a'))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"aa");

  if((r=tmp.replace(0,1,2,'b'))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"bba");
 }

 // find(size_t&,size_t,const WxStr&)
 {
  const WxStr NObj("ab");
  size_t res;

  if((r=a0.find(res,0,a0))!=OK) {  // note: default object contains empty string
    WX_THROW(r);
  }
  if((r=a0.find(res,0,NObj))!=WXM_ESRCH) {
    WX_THROW(r);
  }
  if((r=a0.rfind(res,0,a0))!=OK) {  // note: default object contains empty string
    WX_THROW(r);
  }
  if((r=a0.rfind(res,0,NObj))!=WXM_ESRCH) {
    WX_THROW(r);
  }
 }

 // find(size_t&,size_t,char,size_t)
 {
  size_t res;

  if((r=a0.find(res,0,'c'))!=WXM_ESRCH) {
    WX_THROW(r);
  }
  if((r=a0.find(res,1,'c'))!=WXM_ESRCH) {
    WX_THROW(r);
  }
 }

 // find(size_t&,const char*)
 {
  const char* CStr="ab";
  size_t res;

  if((r=a0.find(res,0,""))!=OK) {  // note: default object contains empty string
    WX_THROW(r);
  }
  if((r=a0.find(res,0,CStr))!=WXM_ESRCH) {
    WX_THROW(r);
  }
  if((r=a0.rfind(res,a0.size(),""))!=OK) {  // note: default object contains empty string
    WX_THROW(r);
  }
  if((r=a0.rfind(res,a0.size(),CStr))!=WXM_ESRCH) {
    WX_THROW(r);
  }
 }

 // find(size_t&,const char*,size_t)
 {
  const char* CStr="ab";
  size_t res;

  if((r=a0.find(res,0,"",0))!=OK) {  // default object contains empty string
    WX_THROW(r);
  }
  if((r=a0.find(res,0,CStr,0))!=OK) {
    WX_THROW(r);
  }
  if((r=a0.rfind(res,a0.size(),"",0))!=OK) {  // default object contains empty string
    WX_THROW(r);
  }
  if((r=a0.rfind(res,a0.size(),CStr,0))!=OK) {
    WX_THROW(r);
  }
 }

 // compare
 {
  const char* CStr="a";
  const WxStr Nobj("a");

  if(a0.compare(a0)!=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(a0.compare(Nobj)>=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(a0.compare(0,0,a0)!=0) {   // empty
    WX_THROW( Wx_general_error() ); 
  }
  if(a0.compare(0,0,Nobj)>=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(a0.compare("")!=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(a0.compare(0,0,"")!=0) {   // compare empty string
    WX_THROW( Wx_general_error() ); 
  }
  if(a0.compare(0,0,CStr)>=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(a0.compare(0,0,"",1)!=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(a0.compare(0,0,CStr,1)>=0) {
    WX_THROW( Wx_general_error() ); 
  }
 }

 // operator +
 {
  const char* CStr="";
  const char ch('x');

  CHK_EQU(a0+a0,"");

  CHK_EQU(a0+CStr,"");

  CHK_EQU(CStr+a0,"");

  CHK_EQU(a0+ch,"x");

  CHK_EQU(ch+a0,"x");
 }
};

//
// Test WxStr (ops of non-default object)
//
static void t2(void)
{
 WxRet r;
 const WxStr Nobj("abc");
 const WxStr Fobj("0123456789");

 CHK_NOT_DEFAULT(Nobj);

 // reset
 {
  WxStr tmp(Nobj);

  CHK_NOT_DEFAULT(tmp);
  tmp.reset();
  CHK_DEFAULT(tmp);
 }

 // size
 {
  WxStr tmp(Nobj);
  if(tmp.size()!=3) {
    WX_THROW( Wx_general_error() ); 
  }
 }

 // [],at(),data(),c_str(),std_string()
 {
  WxStr tmp(Nobj);

  if((tmp[0]!='a')||(tmp[1]!='b')||(tmp[2]!='c')) {
    WX_THROW( Wx_general_error() ); 
  }

  const char *ptr=tmp.data();
  if((ptr[0]!='a')||(ptr[1]!='b')||(ptr[2]!='c')) {
    WX_THROW( Wx_general_error() ); 
  }

  const char *cptr=tmp.c_str();
  if((cptr[0]!='a')||(cptr[1]!='b')||(cptr[2]!='c')||(cptr[3]!=0)) {
    WX_THROW( Wx_general_error() ); 
  }

  const char& ch=tmp.at(1);
  if(ch!='b') {
    WX_THROW( Wx_general_error() ); 
  }

  const std::string& ref=tmp.std_string();
  if(ref!=Nobj.std_string()) {
    WX_THROW( Wx_general_error() ); 
  }
 }

 // assign
 {
  const WxStr NewStr("ijk");
  WxStr tmp(Nobj);
  
  CHK_NOT_DEFAULT(Nobj);

  tmp.assign(NewStr);
  CHK_EQU(tmp,NewStr);

  if((r=tmp.assign(NewStr,1,2))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"jk");

  tmp=NewStr;
  CHK_EQU(tmp,NewStr);

  if((r=tmp.assign(NewStr,5,2))!=WXM_EINVAL) {
    WX_THROW(r);
  }

  const char* NewC="newc";
  if((r=tmp.assign(NewC))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,NewC);

  if((r=tmp.assign(NewC,2))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,"ne");

  tmp.assign(3,'h');
  CHK_EQU(tmp,"hhh");

  tmp=NewC;
  CHK_EQU(tmp,NewC);

  tmp='z';
  CHK_EQU(tmp,"z");
 }

 // swap
 {
  WxStr tmp0(Nobj),tmp1;

  CHK_NOT_DEFAULT(tmp0);
  tmp0.swap(tmp1);
  CHK_DEFAULT(tmp0);

  CHK_EQU(tmp1,Nobj);

  tmp0.swap(tmp1);
  CHK_EQU(tmp0,Nobj);

  CHK_DEFAULT(tmp1);
 }
 
 // find(size_t&, size_t,const WxStr&)
 // find(size_t&, const WxStr&,size_t)
 {
  const WxStr fstr1("456");
  const WxStr fstr2("567");
  size_t res;
  
  if((r=Fobj.find(res,0,fstr1))!=OK) {
    WX_THROW(r);
  }
  if(res!=4) {
    WX_THROW( Wx_general_error() ); 
  }

  if((r=Fobj.rfind(res,Fobj.size(),fstr2))!=OK) {
    WX_THROW(r);
  }
  if(res!=5) {
    WX_THROW( Wx_general_error() ); 
  }

 }

 // find(size_t&, size_t, char)
 {
  size_t res;
  const WxStr Nobj("aabbccdd");

  if((r=Nobj.find(res,1,'b'))!=OK) {
    WX_THROW(r);
  }
  if(res!=2) {
    WX_THROW( Wx_general_error() ); 
  }
  if((r=Nobj.find(res,2,'b'))!=OK) {
    WX_THROW(r);
  }
  if(res!=2) {
    WX_THROW( Wx_general_error() ); 
  }
  if((r=Nobj.find(res,3,'b'))!=OK) {
    WX_THROW(r);
  }
  if(res!=3) {
    WX_THROW( Wx_general_error() ); 
  }
  if((r=Nobj.find(res,4,'b'))!=WXM_ESRCH) {
    WX_THROW(r);
  }

  if((r=Nobj.rfind(res,1,'b'))!=WXM_ESRCH) {
    WX_THROW(r);
  }
  if((r=Nobj.rfind(res,2,'b'))!=OK) {
    WX_THROW(r);
  }
  if(res!=2) {
    WX_THROW( Wx_general_error() ); 
  }
  if((r=Nobj.rfind(res,3,'b'))!=OK) {
    WX_THROW(r);
  }
  if(res!=3) {
    WX_THROW( Wx_general_error() ); 
  }
  if((r=Nobj.rfind(res,4,'b'))!=OK) {
    WX_THROW(r);
  }
  if(res!=3) {
    WX_THROW( Wx_general_error() ); 
  }
 }

 // find(size_t&, cont char*)
 // find(size_t&, const char*,size_t)
 {
  const char* fstr1="23";
  const char* fstr2="45";
  size_t res;
  
  if((r=Fobj.find(res,0,fstr1))!=OK) {
    WX_THROW(r);
  }
  if(res!=2) {
    WX_THROW( Wx_general_error() ); 
  }

  if((r=Fobj.rfind(res,Fobj.size(),fstr2))!=OK) {
    WX_THROW(r);
  }
  if(res!=4) {
    WX_THROW( Wx_general_error() ); 
  }

  if((r=Fobj.find(res,0,fstr1,1))!=OK) {
    WX_THROW(r);
  }
  if(res!=2) {
    WX_THROW( Wx_general_error() ); 
  }
  if((r=Fobj.rfind(res,Fobj.size(),fstr2,1))!=OK) {
    WX_THROW(r);
  }
  if(res!=4) {
    WX_THROW( Wx_general_error() ); 
  }

 }

 // compare
 {
  const WxStr Nobj("12");
  const WxStr Lesser("01");
  const WxStr Greater("32");
  const char* CStr="21";

  if(Nobj.compare(Nobj)!=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(Nobj.compare(Lesser)<=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(Nobj.compare(Greater)>=0) {
    WX_THROW( Wx_general_error() ); 
  }

  if(Nobj.compare(1,1,Nobj)<=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(Nobj.compare(CStr)>=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(Nobj.compare(1,1,CStr)>=0) {
    WX_THROW( Wx_general_error() ); 
  }
  if(Nobj.compare(1,1,CStr,::strlen(CStr)+1)>=0) {
    WX_THROW( Wx_general_error() ); 
  }
 }
 // operator +
 {
  const WxStr Nobj("12");
  const char* CStr="ab";
  const char ch('x');

  if(Nobj+Nobj!="1212") {
    WX_THROW( Wx_general_error() ); 
  }
  if(Nobj+CStr!="12ab") {
    WX_THROW( Wx_general_error() ); 
  }
  if(CStr+Nobj!="ab12") {
    WX_THROW( Wx_general_error() ); 
  }
  if(Nobj+ch!="12x") {
    WX_THROW( Wx_general_error() ); 
  }
  if(ch+Nobj!="x12") {
    WX_THROW( Wx_general_error() ); 
  }
 }
};

//
// Test WxStr (ops failure)
//
static void t_error(void)
{
 WxRet r;
 const WxStr Nobj("abc");
 const WxStr Fobj("0123456789");
 const char* Cstr="cstr";

 // WxStr(const WxStr&,size_t,size_t)
 try {
   WxStr tmp(Nobj,Nobj.size()+1,1);
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxStr::Fault& e) {
   if(e!=WXM_EINVAL) {
     throw;
   }
   // FALLTHROUGH
 }
 catch(...) {
   WX_THROW( Wx_general_error() ); 
 };

 // WxStr(const char*)
 try {
   WxStr tmp((const char*)0);
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxStr::Fault& e) {
   if(e!=WXM_EFAULT) {
     throw;
   }
   // FALLTHROUGH
 }
 catch(...) {
   WX_THROW( Wx_general_error() ); 
 };

 // WxStr(const char*,size_t) throw WxStr::Fault(WXM_EFAULT)
 try {
   WxStr tmp((const char*)0,(size_t)1);
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxStr::Fault& e) {
   if(e!=WXM_EFAULT) {
     throw;
   }
   // FALLTHROUGH
 }
 catch(...) {
   WX_THROW( Wx_general_error() ); 
 };

 // WxStr(const char*,size_t) throw Wx_bad_alloc()
 try {
   WxStr tmp(Cstr,size_t(-1));     // assume not actually copy
   WX_THROW( Wx_general_error() ); 
 }
 catch(const Wx_bad_alloc&) {
   // FALLTHROUGH
 }
 catch(...) {
   WX_THROW( Wx_general_error() ); 
 };

 // WxStr(size_t,char) throw Wx_bad_alloc()
 try {
   WxStr tmp(size_t(-1),'x');
   WX_THROW( Wx_general_error() ); 
 }
 catch(const Wx_bad_alloc&) {
   // FALLTHROUGH
 }
 catch(...) {
   WX_THROW( Wx_general_error() ); 
 };

 // at(size_t) const
 try {
   const char& ch=Fobj.at(Fobj.size());
   const char* ptr=&ch; ptr++;     // prevent compiler warning
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxStr::Fault& e) {
   if(e!=WXM_EINVAL) {
     throw;
   }
   // FALLTHROUGH
 }
 catch(...) {
   WX_THROW( Wx_general_error() ); 
 };

 // at(size_t)
 try {
   WxStr tstr;
   char& ch=tstr.at(tstr.size()+1);
   char* ptr=&ch; ptr++;     // prevent compiler warning
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxStr::Fault& e) {
   if(e!=WXM_EINVAL) {
     throw;
   }
   // FALLTHROUGH
 }
 catch(...) {
   WX_THROW( Wx_general_error() ); 
 };

 // assign(const WxStr&,size_t,size_t)
 {
  WxStr tmp;
  if((r=tmp.assign(Nobj,Nobj.size(),1))!=OK) {
    WX_THROW(r);
  }
  if((r=tmp.assign(Nobj,Nobj.size()+1,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.assign(Nobj,0,Nobj.size()+100))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,Nobj);
 }

 // assign(const char*)
 {
  WxStr tmp;
  if((r=tmp.assign((const char*)0))!=WXM_EFAULT) {
    WX_THROW(r);
  }
 }

 // assign(const char*,size_t)
 {
  WxStr tmp;
  if((r=tmp.assign((const char*)0,(size_t)1))!=WXM_EFAULT) {
    WX_THROW(r);
  }

  try {
    tmp.assign(Cstr,size_t(-1));      // assume not actually copy
    WX_THROW( Wx_general_error() );
  }
  catch(const Wx_bad_alloc&) {
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // assign(size_t, char)
 {
  WxStr tmp;
  try {
    tmp.assign(size_t(-1),'a');
    WX_THROW( Wx_general_error() );
  }
  catch(const Wx_bad_alloc&) {
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // operator =(const char*)
 {
  WxStr tmp;
  try {
    const char* zptr=0;
    tmp=zptr;
    WX_THROW( Wx_general_error() );
  }
  catch(const WxStr::Fault& e) {
    if(e!=WXM_EFAULT) {
      throw;
    }
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // append(const WxStr&,size_t,size_t)
 {
  WxStr tmp;
  if((r=tmp.append(Nobj,0,Nobj.size()))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,Nobj);

  tmp.reset();
  if((r=tmp.append(Nobj,0,Nobj.size()+1))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,Nobj);

  if((r=tmp.append(Nobj,Nobj.size()+1,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // append(const char*)
 {
  WxStr tmp;
  const char* zptr=0;
  if((r=tmp.append(zptr))!=WXM_EFAULT) {
    WX_THROW(r);
  }
 }

 // append(const char*,size_t)
 {
  WxStr tmp;
  if((r=tmp.append((const char*)0,1))!=WXM_EFAULT) {
    WX_THROW(r);
  }
  try {
    tmp.append(Cstr,size_t(-1));
    WX_THROW( Wx_general_error() );
  }
  catch(const Wx_bad_alloc&) {
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // +=(const char*)
 try {
   WxStr tmp;
   tmp+=(const char*)0;
   WX_THROW( Wx_general_error() ); 
 }
 catch(const WxStr::Fault& e) {
   if(e!=WXM_EFAULT) {
     throw;
   }
   // FALLTHROUGH
 }
 catch(...) {
   WX_THROW( Wx_general_error() ); 
 };

 // insert(size_t,const WxStr&)
 {
  WxStr tmp;
  if((r=tmp.insert(tmp.size(),Nobj))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,Nobj);
  if((r=tmp.insert(tmp.size(),Nobj))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,Nobj+Nobj);

  if((r=tmp.insert(tmp.size()+1,Nobj))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // insert(size_t,const WxStr&,size_t,size_t)
 {
  WxStr tmp;
  if((r=tmp.insert(tmp.size(),Nobj,0,Nobj.size()))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,Nobj);
  if((r=tmp.insert(tmp.size(),Nobj,0,Nobj.size()))!=OK) {
    WX_THROW(r);
  }
  CHK_EQU(tmp,Nobj+Nobj);

  if((r=tmp.insert(tmp.size()+1,Nobj,0,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.insert(0,Nobj,Nobj.size()+1,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // insert(size_t,const char*)
 {
  WxStr tmp(Nobj);
  if((r=tmp.insert((size_t)0,(const char*)0))!=WXM_EFAULT) {
    WX_THROW(r);
  }
  if((r=tmp.insert(tmp.size()+1,Cstr))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // insert(size_t,const char*,size_t)
 {
  WxStr tmp(Nobj);
  if((r=tmp.insert((size_t)0,(const char*)0,size_t(1)))!=WXM_EFAULT) {
    WX_THROW(r);
  }
  if((r=tmp.insert(tmp.size()+1,Cstr,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  try {
    tmp.insert(size_t(0),Cstr,size_t(-1)); // assume not actually copy
    WX_THROW( Wx_general_error() );
  }
  catch(const Wx_bad_alloc&) {
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // insert(size_t,size_t,char)
 {
  WxStr tmp(Nobj);
  if((r=tmp.insert(tmp.size()+1,(size_t)1,'a'))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  try {
    tmp.insert(size_t(0),size_t(-1),'a'); // assume not actually copy
    WX_THROW( Wx_general_error() );
  }
  catch(const Wx_bad_alloc&) {
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // erase(size_t)
 {
  WxStr tmp(Nobj);
  if((r=tmp.erase(tmp.size()))!=OK) {
    WX_THROW(r);
  }
  if((r=tmp.erase(tmp.size()+1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // erase(size_t,size_t)
 {
  WxStr tmp(Nobj);
  if((r=tmp.erase(tmp.size()))!=OK) {
    WX_THROW(r);
  }
  if((r=tmp.erase(tmp.size()+1,(size_t)1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // resize(size_t)
 {
  WxStr tmp(Nobj);
  try {
    tmp.resize((size_t)-1);
    WX_THROW( Wx_general_error() );
  }
  catch(const Wx_bad_alloc&) {
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };

  try {
    tmp.resize((size_t)-1,'a');
    WX_THROW( Wx_general_error() );
  }
  catch(const Wx_bad_alloc&) {
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // replace(size_t,size_t,const WxStr&)
 {
  WxStr tmp(Nobj);
  if((r=tmp.replace(tmp.size() ,1,Nobj))!=OK) {
    WX_THROW(r);
  }
  if((r=tmp.replace(tmp.size()+1,1,Nobj))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // replace(size_t,size_t,const char*)
 {
  WxStr tmp(Nobj);
  if((r=tmp.replace(0,1,(const char*)0))!=WXM_EFAULT) {
    WX_THROW(r);
  }
  if((r=tmp.replace(tmp.size()+1,1,Cstr))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // replace(size_t,size_t,const char*,size_t)
 {
  WxStr tmp(Nobj);
  if((r=tmp.replace(0,1,(const char*)0,size_t(0)))!=WXM_EFAULT) {
    WX_THROW(r);
  }
  if((r=tmp.replace(tmp.size()+1,1,Cstr,size_t(0)))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  try {
    tmp.replace(size_t(0),size_t(1),Cstr,size_t(-1));
    WX_THROW( Wx_general_error() );
  }
  catch(const Wx_bad_alloc&) {
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // replace(size_t,size_t,size_t,char)
 {
  WxStr tmp(Nobj);
  try {
    tmp.replace(size_t(0),size_t(1),size_t(-1),'a');
    WX_THROW( Wx_general_error() );
  }
  catch(const Wx_bad_alloc&) {
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // replace(size_t,size_t,const WxStr&,size_t,size_t)
 {
  WxStr tmp(Nobj);
  if((r=tmp.replace(tmp.size(),size_t(1),Nobj,size_t(0),size_t(1)))!=OK) {
    WX_THROW(r);
  }
  if((r=tmp.replace(tmp.size()+1,size_t(1),Nobj,0,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.replace(0,1,Nobj,Nobj.size(),1))!=OK) {
    WX_THROW(r);
  }
  if((r=tmp.replace(0,1,Nobj,Nobj.size()+1,1))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // find(size_t&,size_t,const WxStr&)
 // rfind(size_t&,size_t,const WxStr&)
 {
  WxStr tmp;
  size_t res;
  if((r=tmp.find(res,size_t(0),Nobj))!=WXM_ESRCH) {
    WX_THROW(r);
  }
  if((r=tmp.rfind(res,size_t(1),Nobj))!=WXM_ESRCH) {
    WX_THROW(r);
  }
 }

 // find(size_t&,size_t,ch)
 // rfind(size_t&,size_t,ch)
 {
  WxStr tmp(Nobj);
  size_t res;
  if((r=tmp.find(res,size_t(0),'?'))!=WXM_ESRCH) {
    WX_THROW(r);
  }
  if((r=tmp.rfind(res,size_t(1),'?'))!=WXM_ESRCH) {
    WX_THROW(r);
  }
 }

 // find(size_t&,size_t,const char*)
 // rfind(size_t&,size_t,const char*)
 {
  WxStr tmp(Nobj);
  size_t res;
  if((r=tmp.find(res,size_t(0),Cstr))!=WXM_ESRCH) {
    WX_THROW(r);
  }
  if((r=tmp.rfind(res,size_t(1),Cstr))!=WXM_ESRCH) {
    WX_THROW(r);
  }
  if((r=tmp.find(res,size_t(0),(const char*)0))!=WXM_EFAULT) {
    WX_THROW(r);
  }
  if((r=tmp.rfind(res,size_t(0),(const char*)0))!=WXM_EFAULT) {
    WX_THROW(r);
  }
 }

 // find(size_t&,size_t,const char*,size_t)
 // rfind(size_t&,size_t,const char*,size_t)
 {
  WxStr tmp(Nobj);
  const size_t clen=::strlen(Cstr);
  size_t res;
  if((r=tmp.find(res,size_t(0),Cstr,clen))!=WXM_ESRCH) {
    WX_THROW(r);
  }
  if((r=tmp.rfind(res,size_t(1),Cstr,clen))!=WXM_ESRCH) {
    WX_THROW(r);
  }
  if((r=tmp.find(res,size_t(0),(const char*)0,size_t(1)))!=WXM_EFAULT) {
    WX_THROW(r);
  }
  if((r=tmp.rfind(res,size_t(0),(const char*)0,size_t(1)))!=WXM_EFAULT) {
    WX_THROW(r);
  }
 }

 // compare
 {
   try {
     WxStr tmp(Nobj);
     tmp.compare(tmp.size()+1,1,tmp);
     WX_THROW( Wx_general_error() ); 
   }
   catch(const WxStr::Fault& e) {
     if(e!=WXM_EINVAL) {
       throw;
     }
     // FALLTHROUGH
   }
   catch(...) {
     WX_THROW( Wx_general_error() ); 
   };

   try {
     WxStr tmp(Nobj);
     tmp.compare((const char*)0);
     WX_THROW( Wx_general_error() ); 
   }
   catch(const WxStr::Fault& e) {
     if(e!=WXM_EFAULT) {
       throw;
     }
     // FALLTHROUGH
   }
   catch(...) {
     WX_THROW( Wx_general_error() ); 
   };

   try {
     WxStr tmp(Nobj);
     tmp.compare(0,1,(const char*)0);
     WX_THROW( Wx_general_error() ); 
   }
   catch(const WxStr::Fault& e) {
     if(e!=WXM_EFAULT) {
       throw;
     }
     // FALLTHROUGH
   }
   catch(...) {
     WX_THROW( Wx_general_error() ); 
   };

   try {
     WxStr tmp(Nobj);
     tmp.compare(tmp.size()+1,1,Cstr);
     WX_THROW( Wx_general_error() ); 
   }
   catch(const WxStr::Fault& e) {
     if(e!=WXM_EINVAL) {
       throw;
     }
     // FALLTHROUGH
   }
   catch(...) {
     WX_THROW( Wx_general_error() ); 
   };

   try {
     WxStr tmp(Nobj);
     tmp.compare(size_t(0),size_t(1),(const char*)0,1);
     WX_THROW( Wx_general_error() ); 
   }
   catch(const WxStr::Fault& e) {
     if(e!=WXM_EFAULT) {
       throw;
     }
     // FALLTHROUGH
   }
   catch(...) {
     WX_THROW( Wx_general_error() ); 
   };

   try {
     WxStr tmp(Nobj);
     tmp.compare(tmp.size()+1,1,Cstr,1);
     WX_THROW( Wx_general_error() ); 
   }
   catch(const WxStr::Fault& e) {
     if(e!=WXM_EINVAL) {
       throw;
     }
     // FALLTHROUGH
   }
   catch(...) {
     WX_THROW( Wx_general_error() ); 
   };
 }

 // operator +
 {
   try {
     WxStr tmp(Nobj);
     tmp= Nobj+ (const char*)0;
     WX_THROW( Wx_general_error() ); 
   }
   catch(const WxStr::Fault& e) {
     if(e!=WXM_EFAULT) {
       throw;
     }
     // FALLTHROUGH
   }
   catch(...) {
     WX_THROW( Wx_general_error() ); 
   };

   try {
     WxStr tmp(Nobj);
     tmp= (const char*)0+Nobj;
     WX_THROW( Wx_general_error() ); 
   }
   catch(const WxStr::Fault& e) {
     if(e!=WXM_EFAULT) {
       throw;
     }
     // FALLTHROUGH
   }
   catch(...) {
     WX_THROW( Wx_general_error() ); 
   };
 }
};

//
// Test what_is defined in wxstr.cpp
//
static void t_whatis(void)
{
 const Wx_SrcLoc loc1(__FILE__,__LINE__); 
 
 // print "Invalid argument"
 //       "chk_str.cpp(nnn)"
 std::cout << "Wx::what_is(WXM_EINVAL)=" 
           << Wx::what_is(WXM_EINVAL) << std::endl;
 std::cout << "Wx::what_is(loc1)=" 
           << Wx::what_is(loc1) << std::endl;

 {
  WxRet r(WXM_EBADF); r.set_sli(&loc1);
  std::cout << "Wx::what_is(r(WXM_EBADF))=" 
            << Wx::what_is(r) << std::endl;
  if(r!=WXM_EBADF) {
    WX_THROW( Wx_general_error() );
  }
  if(r.sli_ptr()!=&loc1) {
    WX_THROW( Wx_general_error() );
  }
 }

 {
  Wx_bad_alloc e1; e1.set_sli(&loc1);
  std::cout << "Wx::what_is(Wx_bad_alloc())="
            << Wx::what_is(e1) << std::endl;
  if(e1.sli_ptr()!=&loc1) {
    WX_THROW( Wx_general_error() );
  }
 }
 
 {
  Wx_except e1; e1.set_sli(&loc1);
  std::cout << "Wx::what_is(Wx_except())="
            << Wx::what_is(e1) << std::endl;
  if(e1.sli_ptr()!=&loc1) {
    WX_THROW( Wx_general_error() );
  }
 }

 {
  Wx_general_error e1; e1.set_sli(&loc1);
  std::cout << "Wx::what_is(Wx_general_error())="
            << Wx::what_is(e1) << std::endl;
  if(e1.sli_ptr()!=&loc1) {
    WX_THROW( Wx_general_error() );
  }
 }

 {
  Wx_bad_errno e1(5); e1.set_sli(&loc1);
  std::cout << "Wx::what_is(Wx_bad_errno(5))="
            << Wx::what_is(e1) << std::endl;
  if(e1.sli_ptr()!=&loc1) {
    WX_THROW( Wx_general_error() );
  }
  if(e1.value()!=5) {
    WX_THROW( Wx_general_error() );
  }
 }
};

static const std::string chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxstr.h ...\n";
 if(std::string(WxStr::class_name)!="WxStr") {
   WX_THROW( Wx_general_error() );
 }
 t1(); 
 t2();
 t_whatis();
 t_error();
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknow exception" << std::endl;
 throw;
};
