/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

  Test WxSockFile to communicate with PF_LOCAL,  PF_INET

*/

#include "wxsockfile.h"
#include "wxsockaddrlocal.h"
#include "wxsockaddrinet.h"
#include <sys/types.h>
#include <unistd.h>
#include <sys/wait.h>

#if WXSOCKFILE_VERSION!=8
#error Test code is for WXSOCKFILE_VERSION 8
#endif

#define WX__TEST_SOCK_INET__   // we are going to test INET

//
// Socket address (pathname for AF_LOCAL file) for test
//
static const WxPathName LocalSockFileName("tmp.chk_sockfile.local.tmp");

//
// Message to send to the socket for the tests
//
static const WxStr SvrMsg("Messages sent from the server");
static const WxStr CliMsg("Messages sent from the client");

#ifdef WX__TEST_SOCK_INET__
const char* TestINetIP="127.0.0.1";
const ushort TestINetPort=8080;
#endif

//
// Assert ff is default
//
static void assert_default(const WxSockFile& ff)
{
 WxRet r;

 if(ff.is_default()==false) {
   WX_THROW( Wx_general_error() );
 }
 if(ff.fh().is_default()==false) {
   WX_THROW( Wx_general_error() );
 }

 // stat
 {
   WxFileStat stt;
   if((r=ff.stat(stt))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
 }
 
 if(ff.family()!=PF_UNSPEC) {
   WX_THROW( Wx_general_error() );
 }
};

//
// Setup the test 
//
static void setup_test(void)
{
};

//
// Undo setup_test
//
static void close_test(void)
{
};

//
// Check basic operations on default object
//
static void t1(void)
{
 WxRet r;

 WxSockFile LocalObj;     // non-default object for test as argument
 assert_default(LocalObj);
 if((r=LocalObj.open(PF_LOCAL,SOCK_STREAM))!=OK) {
   WX_HERE(r); throw(r);
 }
 
 // stat
 {
   WxFileStat stt;
   if((r=LocalObj.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_sock()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 // constructor
 {
   WxSockFile tmp0;
   WxSockFile tmp1(tmp0);
   assert_default(tmp0);
   assert_default(tmp1);

   WxSockFile tmp2(LocalObj);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp2.fh()==LocalObj.fh()) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset
 {
   WxSockFile tmp;
   assert_default(tmp);
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   WxSockFile tmp2(LocalObj);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp2.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp2);
 }

 // read,read_max,write,drain,bind
 {
   WxSockFile tmp;
   assert_default(tmp);
   char buf[2];
   if((r=tmp.read(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.max_read(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.write(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.drain())!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if(tmp.family()!=PF_UNSPEC) {
     WX_THROW( Wx_general_error() );
   }
   assert_default(tmp);
 }

 //
 // assignf(WxFileHandle)
 //
 {
   WxSockFile tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxFileHandle()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(LocalObj.fh()))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==LocalObj.fh()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxFileHandle()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxSockFile)
 //
 {
   WxSockFile tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxSockFile()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(LocalObj))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==LocalObj.fh()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxSockFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // shutdown
 //
 {
   WxSockFile tmp;
   assert_default(tmp);
   if((r=tmp.shutdown(SHUT_RD))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // bind,listen,accept,connect,get_addr,get_peer_addr
 //
 {
   WxSockFile tmp;
   WxSockAddrLocal addr;
   assert_default(tmp);
   
   if((r=tmp.bind(addr))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.listen(0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }

   WxSockFile tmp1;
   assert_default(tmp1);
   if((r=tmp.accept(tmp1,addr))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.connect(addr))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.get_addr(addr))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.get_peer_addr(addr))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }

   assert_default(tmp);
   assert_default(tmp1);
 }

};

//
// Check basic operations on non-default object
//
static void t2(void)
{
 WxRet r;
 
 // stat LocalObj
 WxSockFile LocalObj;     // non-default object for the test
 assert_default(LocalObj);
 if((r=LocalObj.open(PF_LOCAL,SOCK_STREAM))!=OK) {
   WX_HERE(r); throw(r);
 }
 
 // stat
 {
   WxFileStat stt;
   if((r=LocalObj.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_sock()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset
 {
   WxSockFile tmp(LocalObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 // (zero byte) read,read_max,write,drain
 {
   WxSockFile tmp(LocalObj);
   char buf[2]={-1,-1};
   size_t n_rd=5;   // anything >1
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.max_read(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.write(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.drain())!=WXM_EINVAL) {
     WX_HERE(r); throw(r); // ref fsync(2)
   }
 }

 // open (WXM_NDEFAULT)
 {
   WxSockFile tmp(LocalObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.open(PF_LOCAL,SOCK_STREAM))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxFileHandle())
 //
 {
   WxSockFile tmp(LocalObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxFileHandle()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxSockFile())
 //
 {
   WxSockFile tmp(LocalObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxSockFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // shutdown,bind,listen,accept,connect,get_addr,get_peer_addr
 //
 {
   WxSockFile tmp(LocalObj);
   const WxSockAddrLocal addr;
   WxSockAddrLocal addr2;
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(addr2!=addr) {
     WX_THROW( Wx_general_error() );
   }
     
   // 
   // I am not sure what shoud be returned for the following functions
   // Coding takes whatever tested
   //
   if((r=tmp.shutdown(SHUT_RD))!=OK) {  // test showed errno==0
     WX_HERE(r); throw(r);
   }

   if((r=tmp.bind(addr))!=OK) {         // test showed errno==0
     WX_HERE(r); throw(r);
   }
   std::cout << "WxSockFile (PF_LOCAL) bind a default address\n";

   if((r=tmp.listen(0))!=OK) {         // test showed errno==0
     WX_HERE(r); throw(r);
   }

   WxSockFile tmp1;
   assert_default(tmp1);
   if((r=tmp.accept(tmp1,addr2))!=WXM_EINVAL) { // test showed errno=EINVAL
     WX_HERE(r); throw(r);                      // and addr2 is modified
   }
   assert_default(tmp1);
   if(addr2.is_default()==false) {
     WX_THROW( Wx_general_error() );
   }
   if(addr2.family()!=AF_LOCAL) {
     WX_THROW( Wx_general_error() );
   }
   std::cout << "WxSockFile (PF_LOCAL) accepted a default address\n";

   if((r=tmp.connect(addr))!=WXM_EINVAL) { // test showed errno==EINVAL
     WX_HERE(r); throw(r);
   }

   if((r=tmp.get_addr(addr2))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.get_peer_addr(addr2))!=WXM_ENOTCONN) { // test showed errno==ENOTCONN
     WX_HERE(r); throw(r);
   }
 }

};

static void t_exception(void)
{
 //
 // Test copy constructor exception
 //
 try {
   char tbuf[sizeof(WxSockFile)];
   std::memset(tbuf,-2,sizeof(tbuf));
   //
   // as -2 is known not default fd and no such valid handle. 
   // construct from tbuf should cause Fault exception
   //
   WxSockFile tt2(reinterpret_cast<const WxSockFile&>(tbuf));
   WX_THROW( Wx_general_error() );  // no expected exception
 }
 catch(const WxSockFile::Fault& e) {
   if(e!=WXM_EBADF) {
     WxRet r=e;
     WX_HERE(r); throw(r);
   }
   // FALLTHROUGH 
 }
 catch(...) {
   throw;
 };

 //
 // Test destructor exception
 //
 {
   WxSockFile tmp;
   std::memset(&tmp,-2,sizeof(tmp));
   // destructor of tmp is now not to cause exception for this setting
 }
};

//
// Test WxSockFile communication (AF_LOCAL)
//
// Connect to LocalSockFileName and receieve SvrMsg, then send CliMsg 
// and exit(0).
//
static int process_client_local(void) throw()
try {
 WxRet r;
 WxSockFile cli;
 WxSockAddrLocal saddr(LocalSockFileName);
 
 //
 // Open socket (PF_LOCAL) and connect to the server
 //
 if((r=cli.open(PF_LOCAL,SOCK_STREAM))!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=cli.connect(saddr))!=OK) {
   WX_HERE(r); throw(r);
 }

 // stat
 {
   WxFileStat stt;
   if((r=cli.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_sock()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 // Dump addr
 {
  WxStr str;
  if((r=saddr.get_notation(str))!=OK) {
     WX_THROW( Wx_general_error() ); 
  }
  std::cout << "process_client_local connected to: " << str << std::endl;
 }

 //
 // Receive message from the server (SvrMsg expected)
 //
 size_t n_rd;
 char rbuf[64];
 if(sizeof(rbuf)<SvrMsg.size()+1) {
   WX_THROW( Wx_general_error() );   // rbuf not long enough, reedit rbuf size.
 }
 if((r=cli.read(rbuf,sizeof(rbuf),&n_rd))!=OK) {  // note: max_read may block
   WX_HERE(r); throw(r);
 }
 if(n_rd!=SvrMsg.size()) {
   WX_THROW( Wx_general_error() ); 
 }
 if(SvrMsg!=WxStr(rbuf,n_rd)) {
   WX_THROW( Wx_general_error() );   // not expected server message
 }
 
 //
 // Send message CliMsg to the server
 //
 size_t n_wr;
 if((r=cli.write(CliMsg.c_str(),CliMsg.size(),&n_wr))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(n_wr!=CliMsg.size()) {
   WX_THROW( Wx_general_error() ); 
 }

 return(0);
}
catch(const WxRet& e) {
 std::cerr << "process_client_local: "  << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << "process_client_local :" << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << "process_client_local :" << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << "process_client_local : unknown exception" << std::endl;
 return(-1);
};

//
// Test WxSockFile communication (AF_LOCAL)
// t_comm_local fork process_client_local and 
//
static void t_comm_local(void)
try {
 WxRet r;
 pid_t child_pid;
 int child_status;

 //
 // Create server socket (PF_LOCAL) LocalSockFileName
 //
 ::unlink(LocalSockFileName.pathname().c_str());
 WxSockAddrLocal saddr(LocalSockFileName);
 WxSockFile svr;

 if((r=svr.open(PF_LOCAL,SOCK_STREAM))!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=svr.bind(saddr))!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=svr.listen(1))!=OK) {
   WX_HERE(r); throw(r);
 }

 // stat
 {
   WxFileStat stt;
   if((r=svr.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_sock()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // Fork child process
 //
 child_pid=::fork();
 if(child_pid<0) {
   WX_THROW( Wx_bad_errno(errno) );
 }
 if(child_pid==0) {
   ::exit(process_client_local());
 }

 //
 // Server accepts connection
 //
 WxSockFile psock;
 WxSockAddrLocal paddr;
 if((r=svr.accept(psock,paddr))!=OK) {
   WX_HERE(r); throw(r);
 } 
 //
 // test showed default address returned (probable right for PF_LOCAL?)
 //
 if(paddr.is_default()==false) {
   // Dump the accept addr
   {
    WxStr str;
    if((r=paddr.get_notation(str))!=OK) {
       WX_THROW( Wx_general_error() ); 
    }
    std::cout << "server(PF_LOCAL) accept() address: " << str << std::endl;
   }
   WX_THROW( Wx_general_error() );
 }


 //
 // Send SvrMsg to the client
 //
 size_t n_wr;
 if((r=psock.write(SvrMsg.c_str(),SvrMsg.size(),&n_wr))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(n_wr!=SvrMsg.size()) {
   WX_THROW( Wx_general_error() ); 
 }

 //
 // Receive message from the client (CliMsg expected)
 //
 size_t n_rd;
 char rbuf[64];
 if(sizeof(rbuf)<CliMsg.size()+1) {
   WX_THROW( Wx_general_error() );   // rbuf not long enough
 }
 if((r=psock.max_read(rbuf,sizeof(rbuf),&n_rd))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(n_rd!=CliMsg.size()) {
   WX_THROW( Wx_general_error() ); 
 }
 if(CliMsg!=WxStr(rbuf,n_rd)) {
   WX_THROW( Wx_general_error() ); 
 }
 
 if(::wait(&child_status)!=child_pid) {
   WX_THROW( Wx_bad_errno(errno) );
 }
 if(WIFEXITED(child_status)==0) {
   WX_THROW( Wx_general_error() );  // child process was not terminated voluntarily
 }
 if(WEXITSTATUS(child_status)!=0) {
   WX_THROW( Wx_general_error() );  // child process not exist(0)
 }
 ::unlink(LocalSockFileName.pathname().c_str());
}
catch(...) {
 ::unlink(LocalSockFileName.pathname().c_str());
 throw;
};

#ifdef WX__TEST_SOCK_INET__
//
// Test WxSockFile communication (AF_INET)
//
// Connect to TestINetIP and receieve SvrMsg, then send CliMsg 
// and exit(0).
//
static int process_client_inet4(void) throw()
try {
 WxRet r;
 WxSockFile cli;
 WxSockAddrINet saddr(TestINetIP,TestINetPort);

 //
 // Open socket (PF_INET) and connect to the server
 //
 if((r=cli.open(PF_INET,SOCK_STREAM))!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=cli.connect(saddr))!=OK) {
   WX_HERE(r); throw(r);
 }

 // stat
 {
   WxFileStat stt;
   if((r=cli.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_sock()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 // Dump addr
 {
  WxStr str;
  if((r=saddr.get_notation(str))!=OK) {
     WX_THROW( Wx_general_error() ); 
  }
  std::cout << "process_client_inet4 connected to: " << str << std::endl;
 }

 //
 // Receive message from the server (SvrMsg expected)
 //
 size_t n_rd;
 char rbuf[64];
 if(sizeof(rbuf)<SvrMsg.size()+1) {
   WX_THROW( Wx_general_error() );   // rbuf not long enough, reedit rbuf size.
 }
 if((r=cli.read(rbuf,sizeof(rbuf),&n_rd))!=OK) {  // note: using max_read may block
   WX_HERE(r); throw(r);
 }
 if(n_rd!=SvrMsg.size()) {
   WX_THROW( Wx_general_error() ); 
 }
 if(SvrMsg!=WxStr(rbuf,n_rd)) {
   WX_THROW( Wx_general_error() ); 
 }
 
 //
 // Send message CliMsg to the server
 //
 size_t n_wr;
 if((r=cli.write(CliMsg.c_str(),CliMsg.size(),&n_wr))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(n_wr!=CliMsg.size()) {
   WX_THROW( Wx_general_error() ); 
 }

 return(0);
}
catch(const WxRet& e) {
 std::cerr << "process_client_inet4: "  << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << "process_client_inet4 :" << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << "process_client_inet4 :" << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << "process_client_inet4 : unknown exception" << std::endl;
 return(-1);
};

//
// Test WxSockFile communication (AF_INET)
// t_comm_inet4 fork process_client_inet4 and runs as the server
// process_client_inet4 will run as the client
//
static void t_comm_inet4(void)
{
 WxRet r;
 pid_t child_pid;
 int child_status;

 //
 // Create server socket
 //
 WxSockFile svr;
 WxSockAddrINet saddr(TestINetIP,TestINetPort);

 if((r=svr.open(PF_INET,SOCK_STREAM))!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=svr.bind(saddr))!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=svr.listen(1))!=OK) {
   WX_HERE(r); throw(r);
 }

 // stat
 {
   WxFileStat stt;
   if((r=svr.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_sock()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // Fork child process
 //
 child_pid=::fork();
 if(child_pid<0) {
   WX_THROW( Wx_bad_errno(errno) );
 }
 if(child_pid==0) {
   ::exit(process_client_inet4());
 }

 //
 // Server accepts connection
 //
 WxSockFile psock;
 WxSockAddrINet paddr;
 if((r=svr.accept(psock,paddr))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(paddr.is_default()) {
   WX_THROW( Wx_general_error() );
 }

 // Dump the accept addr
 {
   WxStr str;
   if((r=paddr.get_notation(str))!=OK) {
     WX_HERE(r); throw(r);
   }
   std::cout << "server(PF_INET) accept() address: " << str << std::endl;

   if((r=psock.get_addr(paddr))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=paddr.get_notation(str))!=OK) {
     WX_HERE(r); throw(r);
   }
   std::cout << " accepted.sock.get_addr()= " << str << std::endl;

   if((r=psock.get_peer_addr(paddr))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=paddr.get_notation(str))!=OK) {
     WX_HERE(r); throw(r);
   }
   std::cout << " accepted.sock.get_peer_addr()= " << str << std::endl;
 }

 //
 // Send SvrMsg to the client
 //
 size_t n_wr;
 if((r=psock.write(SvrMsg.c_str(),SvrMsg.size(),&n_wr))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(n_wr!=SvrMsg.size()) {
   WX_THROW( Wx_general_error() ); 
 }

 //
 // Receive message from the client (CliMsg expected)
 //
 size_t n_rd;
 char rbuf[64];
 if(sizeof(rbuf)<CliMsg.size()+1) {
   WX_THROW( Wx_general_error() );   // rbuf not long enough
 }
 if((r=psock.max_read(rbuf,sizeof(rbuf),&n_rd))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(n_rd!=CliMsg.size()) {
   WX_THROW( Wx_general_error() ); 
 }
 if(CliMsg!=WxStr(rbuf,n_rd)) {
   WX_THROW( Wx_general_error() ); 
 }
 
 if(::wait(&child_status)!=child_pid) {
   WX_THROW( Wx_bad_errno(errno) );
 }
 if(WIFEXITED(child_status)==0) {
   WX_THROW( Wx_general_error() );  // child process was not terminated voluntarily
 }
 if(WEXITSTATUS(child_status)!=0) {
   WX_THROW( Wx_general_error() );  // child process not exist(0)
 }
}
#endif

static WxStr chdr="+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n";
int main(void) throw()
try {
 std::cout << "Checking wxsockfile.h ...\n";
 if(std::strcmp(WxSockFile::class_name,"WxSockFile")!=0) {
   WX_THROW( Wx_general_error() );
 }
 try {
   setup_test();
   t1();
   t2();
   t_exception();
   t_comm_local();
#ifdef WX__TEST_SOCK_INET__
   t_comm_inet4();
#endif
   close_test();
 }
 catch(...) {
   close_test();
   throw;
 }
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
