/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Check WxRegFile public members

   Build: make chk_regfile

   Note: Test data files should exist in the working directory
         Files may be created in the working directory and unlinked
*/

#include "wxregfile.h"
#include "chk_data.h"
#include <memory>
#include <unistd.h>

#if WXREGFILE_VERSION!=8
#error Test code is for WXREGFILE_VERSION 8
#endif

static const WxStr Tmp_Name="tmp.chk_tobjile-open_tmpfile.tmp";

//
// Assert regf is default
//
static void assert_default(const WxRegFile& regf)
{
 WxRet r;

 if(regf.is_default()==false) {
   WX_THROW( Wx_general_error() );
 }
 if(regf.fh().is_default()==false) {
   WX_THROW( Wx_general_error() );
 }

 // stat
 {
   WxFileStat stt;
   if((r=regf.stat(stt))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
 }

 // check pos(0)
 if(regf.get_pos()!=0) {
   WX_THROW( Wx_general_error() );
 }
};

//
// Check basic operations on default object
//
static void t1(void)
{
 WxRegFile EmptyFile;

 WxRet r;

 assert_default(EmptyFile);

 //
 // Prepare
 //
 if((r=EmptyFile.open(TestFile_Empty,O_RDONLY))!=OK) {
   WX_HERE(r); throw(r); // Test file (TestFile_Empty) open failure
 }
 if(EmptyFile.is_default()) {
   WX_THROW( Wx_general_error() );
 }
 if(EmptyFile.get_pos()!=0) {
   WX_THROW( Wx_general_error() );
 }

 // reset(1)
 {
   WxRegFile tmp;
   assert_default(tmp);
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 // constructor
 {
   WxRegFile tobj;
   WxRegFile tmp1(tobj);
   assert_default(tobj);
   assert_default(tmp1);

   WxRegFile tmp2(EmptyFile);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp2.fh()==EmptyFile.fh()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp2.get_pos()!=EmptyFile.get_pos()) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset(2)
 {
   WxRegFile tmp2(EmptyFile);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp2.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp2);
 }

 // read
 {
   WxRegFile tobj;
   char buf[2]={-1,-1};
   size_t n_rd=5;   // anything not zero
   assert_default(tobj);
   if((r=tobj.read(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=EmptyFile.read(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if(EmptyFile.get_pos()!=0) {
     WX_THROW( Wx_general_error() );
   }
   assert_default(tobj);
 }

 // max_read
 {
   WxRegFile tobj;
   char buf[2]={-1,-1};
   size_t n_rd=5;   // anything not zero
   assert_default(tobj);
   if((r=tobj.max_read(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=EmptyFile.max_read(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if(EmptyFile.get_pos()!=0) {
     WX_THROW( Wx_general_error() );
   }
   assert_default(tobj);
 }

 // write
 {
   WxRegFile tobj;
   char buf[2]={-1,-1};
   size_t n_wr=5;   // anything not zero
   assert_default(tobj);
   if((r=tobj.write(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tobj);
   if((r=tobj.write(buf,sizeof(buf),&n_wr))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if(n_wr!=0) {
     WX_THROW( Wx_general_error() );
   }
   assert_default(tobj);
 }

 // drain
 {
   WxRegFile tobj;
   assert_default(tobj);
   if((r=tobj.drain())!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tobj);
 }

 // get_pos
 {
   WxRegFile tobj;
   assert_default(tobj);
   if(tobj.get_pos()!=0) {
     WX_THROW( Wx_general_error() );
   }
   assert_default(tobj);
 }

 // set_pos_end
 {
   WxRegFile tobj;
   assert_default(tobj);
   if((r=tobj.seek_pos_end())!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tobj);
 }

 // read_line(1)
 {
   WxRegFile tobj;
   char buf[2]={-1,-1};
   size_t n_rd=5;   // anything not zero
   assert_default(tobj);
   if((r=tobj.read_line(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=EmptyFile.read_line(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if(EmptyFile.get_pos()!=0) {
     WX_THROW( Wx_general_error() );
   }
   assert_default(tobj);
 }

 // read_line(2)
 {
   WxRegFile tobj;
   WxStr buf;
   size_t n_rd=5;   // anything not zero
   assert_default(tobj);
   if((r=tobj.read_line(buf,0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=EmptyFile.read_line(buf,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if(EmptyFile.get_pos()!=0) {
     WX_THROW( Wx_general_error() );
   }
   assert_default(tobj);
 }

 // resize
 {
   WxRegFile tobj;
   assert_default(tobj);
   if((r=tobj.resize(1))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tobj);
 }

 // open
 {
   WxRegFile tobj;
   assert_default(tobj);
   if((r=tobj.open(WxPathName("tmp.damn.shit.fuck.84jdnjk.tmp"),0))!=WXM_ENOENT) {
     WX_HERE(r); throw(r);  // well, what can i say
   }
   assert_default(tobj);
 }

 //
 // open_tmpfile
 //
 {
   WxRegFile tobj;
   WxPathName tmp_name(Tmp_Name);
   assert_default(tobj);
   if((r=tobj.open_tmpfile(tmp_name))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tobj.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   try {
     if(tmp_name.pathname().size()!=Tmp_Name.size()+6) {
       WX_THROW( Wx_general_error() );
     }
     if(::access(tmp_name.pathname().c_str(),F_OK)!=0) {
       r=WxErrMsg(errno); WX_HERE(r); throw(r);
     }
     if((r=tobj.open_tmpfile(tmp_name))!=WXM_NDEFAULT) {
       WX_HERE(r); throw(r);
     }
     if((r=tobj.reset())!=OK) {
       WX_HERE(r); throw(r);
     }
     assert_default(tobj);
     ::unlink(tmp_name.pathname().c_str());
   }
   catch(...) {
     tobj.reset();
     ::unlink(tmp_name.pathname().c_str());
     throw;
   }
 }

 //
 // assignf(WxFileHandle)
 //
 {
   WxRegFile tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxFileHandle()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(EmptyFile.fh()))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==EmptyFile.fh()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxFileHandle()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assign(WxRegFile)
 //
 {
   WxRegFile tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxRegFile()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(EmptyFile))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==EmptyFile.fh()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxRegFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Check basic operations on non-default object
//
static void t2(void)
{
 WxRegFile EmptyFile;
 WxRegFile TFile1A;
 WxRegFile TFile2A;
 WxRet r;

 assert_default(EmptyFile);
 assert_default(TFile1A);
 assert_default(TFile2A);

 //
 // Prepare EmptyFile
 //
 if((r=EmptyFile.open(TestFile_Empty,O_RDONLY))!=OK) {
   WX_HERE(r); throw(r); // Test file (TestFile_Empty) open failure
 }
 if(EmptyFile.is_default()) {
   WX_THROW( Wx_general_error() );
 }
 if(EmptyFile.get_pos()!=0) {
   WX_THROW( Wx_general_error() );
 }
 if(EmptyFile.fh().is_default()) {
   WX_THROW( Wx_general_error() );
 }

 // stat EmptyFile
 {
   WxFileStat stt;
   if((r=EmptyFile.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(stt.is_reg()==false) {
     WX_THROW( Wx_general_error() );
   }
   if(stt.size()!=0) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // Prepare TFile1A
 //
 if((r=TFile1A.open(TestFile_1A,O_RDONLY))!=OK) {
   WX_HERE(r); throw(r); // Test file open failure
 }
 if(TFile1A.is_default()) {
   WX_THROW( Wx_general_error() );
 }
 if(TFile1A.get_pos()!=0) {
   WX_THROW( Wx_general_error() );
 }
 if(TFile1A.fh().is_default()) {
   WX_THROW( Wx_general_error() );
 }

 // stat TFile1A
 {
   WxFileStat stt;
   if((r=TFile1A.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(stt.is_reg()==false) {
     WX_THROW( Wx_general_error() );
   }
   if(stt.size()!=1) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // Prepare TFile2A
 //
 if((r=TFile2A.open(TestFile_2A,O_RDONLY))!=OK) {
   WX_HERE(r); throw(r); // Test file open failure
 }
 if(TFile2A.is_default()) {
   WX_THROW( Wx_general_error() );
 }
 if(TFile2A.get_pos()!=0) {
   WX_THROW( Wx_general_error() );
 }
 if(TFile2A.fh().is_default()) {
   WX_THROW( Wx_general_error() );
 }

 // stat TFile2A
 {
   WxFileStat stt;
   if((r=TFile2A.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(stt.is_reg()==false) {
     WX_THROW( Wx_general_error() );
   }
   if(stt.size()!=2) {
     WX_THROW( Wx_general_error() );
   }
 }

 // constructor
 {
   WxRegFile tmp1(TFile1A);

   if(tmp1.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp1.fh()==TFile1A.fh()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp1.get_pos()!=TFile1A.get_pos()) {
     WX_THROW( Wx_general_error() );
   }
   tmp1.set_pos(5);

   WxRegFile tmp2(tmp1);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp2.fh()==tmp1.fh()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp2.get_pos()!=tmp1.get_pos()) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset
 {
   WxRegFile tmp(EmptyFile);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 // read
 {
   WxRegFile tmp(TFile1A);
   char buf[2]={-1,-1};
   size_t n_rd=5;   // anything >1
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(buf[0]!='\n') {
     WX_THROW( Wx_general_error() );
   }
   if(buf[1]!=char(-1)) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.get_pos()!=1) {
     WX_THROW( Wx_general_error() );
   }
   // eof
   if((r=tmp.read(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if(buf[0]!='\n') {
     WX_THROW( Wx_general_error() );
   }
   if(buf[1]!=char(-1)) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.get_pos()!=1) {
     WX_THROW( Wx_general_error() );
   }
 }

 // max_read
 {
   WxRegFile tmp(TFile1A);
   char buf[2]={-1,-1};
   size_t n_rd=5;   // anything >1
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.max_read(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(buf[0]!='\n') {
     WX_THROW( Wx_general_error() );
   }
   if(buf[1]!=char(-1)) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.get_pos()!=1) {
     WX_THROW( Wx_general_error() );
   }
   // eof
   if((r=tmp.max_read(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if(buf[0]!='\n') {
     WX_THROW( Wx_general_error() );
   }
   if(buf[1]!=char(-1)) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.get_pos()!=1) {
     WX_THROW( Wx_general_error() );
   }
 }

 // write
 {
   WxRegFile tmp;
   const char buf[2]={-1,-1};
   size_t n_wr=5;   // anything not zero
   assert_default(tmp);
   if((r=tmp.open(WxPathName(Tmp_Name),O_WRONLY|O_CREAT))!=OK) {
     WX_HERE(r); throw(r);
   }
   try {
     if(tmp.is_default()) {
       WX_THROW( Wx_general_error() );
     }
     if((r=tmp.write(buf,sizeof(buf),&n_wr))!=OK) {
       WX_HERE(r); throw(r);
     }
     if(n_wr!=sizeof(buf)) {
       WX_THROW( Wx_general_error() );
     }
     if(tmp.get_pos()!=sizeof(buf)) {
       WX_THROW( Wx_general_error() );
     }
     if((tmp.drain())!=OK) {
       WX_THROW( Wx_general_error() );
     }
     tmp.set_pos(1);
     if(tmp.get_pos()!=1) {
       WX_THROW( Wx_general_error() );
     }
     ::unlink(Tmp_Name.c_str());
   }
   catch(...) {
     ::unlink(Tmp_Name.c_str());
     throw;
   }
 }

 // seek_pos_end
 {
   WxRegFile tmp(TFile1A);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.seek_pos_end())!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.get_pos()!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(TFile1A.get_pos()!=0) {
     WX_THROW( Wx_general_error() ); // should not be modified
   }
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 // read_line(1), TFile1A
 {
   WxRegFile tmp(TFile1A);
   char buf[2]={-1,-1};
   size_t n_rd=5;   // anything not zero
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read_line(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(buf[0]!='\n') {
     WX_THROW( Wx_general_error() );
   }
   if(buf[1]!=0) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.get_pos()!=1) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read_line(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 // read_line(2), TFile1A
 {
   WxRegFile tmp(TFile1A);
   WxStr buf;
   size_t n_rd=5;   // anything not zero
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read_line(buf,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(buf.size()!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(n_rd!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(buf[0]!='\n') {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.get_pos()!=1) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read_line(buf,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
 }

 // read_line(1), TFile2A
 {
   WxRegFile tmp(TFile2A);
   char buf[3]={-1,-1,-1};
   size_t n_rd=5;   // anything not zero
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read_line(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(buf[0]!='\n') {
     WX_THROW( Wx_general_error() );
   }
   if(buf[1]!=0) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.get_pos()!=1) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read_line(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(buf[0]!='\n') {
     WX_THROW( Wx_general_error() );
   }
   if(buf[1]!=0) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.get_pos()!=2) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.read_line(buf,sizeof(buf),&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 // read_line(2), TFile2A
 {
   WxRegFile tmp(TFile2A);
   WxStr buf;
   size_t n_rd=5;   // anything not zero
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read_line(buf,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(buf.size()!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(n_rd!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(buf[0]!='\n') {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.get_pos()!=1) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.read_line(buf,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(buf.size()!=2) {
     WX_THROW( Wx_general_error() );  // not appended
   }
   if(n_rd!=1) {
     WX_THROW( Wx_general_error() );
   }
   if(buf[1]!='\n') {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.get_pos()!=2) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.read_line(buf,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if(buf.size()!=2) {
     WX_THROW( Wx_general_error() );  // modified
   }
 }

 // resize
 {
   WxRegFile tmp;
   const char buf[2]={-1,-1};
   size_t n_wr=5;   // anything not zero
   assert_default(tmp);
   if((r=tmp.open(WxPathName(Tmp_Name),O_WRONLY|O_CREAT))!=OK) {
     WX_HERE(r); throw(r);
   }
   try {
     if(tmp.is_default()) {
       WX_THROW( Wx_general_error() );
     }
     if((r=tmp.write(buf,sizeof(buf),&n_wr))!=OK) {
       WX_HERE(r); throw(r);
     }
     if(n_wr!=sizeof(buf)) {
       WX_THROW( Wx_general_error() );
     }
     if(tmp.get_pos()!=sizeof(buf)) {
       WX_THROW( Wx_general_error() );
     }
     if((tmp.drain())!=OK) {
       WX_THROW( Wx_general_error() );
     }

     // resize to 1
     if((r=tmp.resize(1))!=OK) {
       WX_HERE(r); throw(r);
     }
     if((r=tmp.seek_pos_end())!=OK) {
       WX_HERE(r); throw(r);
     }
     if(tmp.get_pos()!=1) {
       WX_THROW( Wx_general_error() );
     }

     // resize to 10
     if((r=tmp.resize(10))!=OK) {
       WX_HERE(r); throw(r);
     }
     if((r=tmp.seek_pos_end())!=OK) {
       WX_HERE(r); throw(r);
     }
     if(tmp.get_pos()!=10) {
       WX_THROW( Wx_general_error() );
     }

     ::unlink(Tmp_Name.c_str());
   }
   catch(...) {
     ::unlink(Tmp_Name.c_str());
     throw;
   }
 }

 // open (WXM_NDEFAULT)
 {
   WxRegFile tmp(TFile1A);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.open(WxPathName(Tmp_Name),O_RDONLY))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxFileHandle())
 //
 {
   WxRegFile tmp(TFile1A);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxFileHandle()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxRegFile())
 //
 {
   WxRegFile tmp(TFile1A);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxRegFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Test read operation (interleave read)
//
// Two objects are designated to read tfile alternatively.
// tfile points to any file that can be read.
//
static void interleaved_read(const WxPathName& tfile)
{
 WxRet r;
 WxRegFile f0;   // tfile

 if((r=f0.open(tfile,O_RDONLY))!=OK) {
   WX_HERE(r); throw(r);
 }
 WxRegFile f1(f0);   // f1 will read with the former half part of tfile
 WxRegFile f2(f0);   // f2 will read with the latter half part of tfile

 if((r=f0.seek_pos_end())!=OK) {
   WX_HERE(r); throw(r);
 }

 //
 // file size of f0 and the start position of f2, also the defined file size of f1
 //
 const off_t FSize=f0.get_pos(); 
 const  off_t MidPos=FSize>>1; 
 f1.set_pos(0);                 // set f1 to begin from 0
 f2.set_pos(MidPos);            // set f2 to begin from MidPos

 //
 // Set str0 with the whole file of f0
 //
 WxStr str0;
 {
   size_t n_rd;
   std::auto_ptr<char> text0(new char[FSize]);
   f0.set_pos(0);
   if((r=f0.max_read(text0.get(),FSize,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((off_t)n_rd!=FSize) {
     WX_THROW( Wx_general_error() );
   }
   if(f0.get_pos()!=FSize) {
     WX_THROW( Wx_general_error() ); // value of seek_pos_end not identical to read EOF
   }
   str0.assign(text0.get(),FSize); // transfer text0 to str0
 }

 //
 // Read f1 to str1 and f2 to str2 in a loop, each loop read one byte
 //
 WxStr str1,str2;
 for(;;) {
   char ch;
   size_t n_rd;
   bool all_eof=true;
   if(str1.size()<(size_t)MidPos) {  // f1 read up to MidPos characters
     if((r=f1.read(&ch,1,&n_rd))!=OK) {
       WX_HERE(r); throw(r);
     }
     if(n_rd==1) {
       str1+=ch;         // append the read
       all_eof=false;
     } else {
       if(n_rd!=0) {
         WX_THROW( Wx_general_error() );  // not expected value of n_rd
       }
       WX_THROW( Wx_general_error() );  // not expected value of n_rd(0)
     }
   }

   if((r=f2.read(&ch,1,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd==1) {
     str2+=ch;           // append the read
     all_eof=false;
   } else {
     if(n_rd!=0) {
       WX_THROW( Wx_general_error() );  // not expected value
     }
     if(f2.get_pos()!=FSize) {
       WX_THROW( Wx_general_error() );  // not expected value
     }
     // FALLTHROUGH
   }

   if(all_eof==true) {
     break;
   }
 }

 if(f1.get_pos()!=MidPos) {
   WX_THROW( Wx_general_error() );  // not expected value
 }
 if(f2.get_pos()!=FSize) {
   WX_THROW( Wx_general_error() );  // not expected value
 }
 str1+=str2;   // append str2 to str1
 if(str0!=str1) {
   WX_THROW( Wx_general_error() );  // assertion failed
 }

};

static void t_exception(void)
{
 //
 // Test copy constructor exception
 //
 try {
   char tbuf[sizeof(WxRegFile)];
   std::memset(tbuf,-2,sizeof(tbuf));
   //
   // as -2 is known not default fd and no such valid handle. 
   // construct from tbuf should cause Fault exception
   //
   WxRegFile tt2(reinterpret_cast<const WxRegFile&>(tbuf));
   WX_THROW( Wx_general_error() );  // no expected exception
 }
 catch(const WxRegFile::Fault& e) {
   if(e!=WXM_EBADF) {
     WxRet r=e;
     WX_HERE(r); throw(r);
   }
   // FALLTHROUGH 
 }
 catch(...) {
   throw;
 };

 //
 // Test destructor exception
 //
 {
   WxRegFile tmp;
   std::memset(&tmp,-2,sizeof(tmp));
   // destructor of tmp is now not to cause exception
 }

};

//
// Test read tfile (read_line against read)
//
static void t_read_line(const WxPathName& tfile)
{
 WxRet r;

 //
 // Set str0 with the whole file of tfile
 //
 WxStr str0;
 {
   WxRegFile f0;
   size_t n_rd;

   if((r=f0.open(tfile,O_RDONLY))!=OK) {
     WX_HERE(r); throw(r); // file open failure
   }
   if((r=f0.seek_pos_end())!=OK) {
     WX_HERE(r); throw(r);
   }
   const size_t FSize=f0.get_pos();
   std::auto_ptr<char> text0(new char[FSize]);
   f0.set_pos(0);

   if((r=f0.max_read(text0.get(),FSize,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=FSize) {
     WX_THROW( Wx_general_error() );
   }
   if(f0.get_pos()!=(off_t)FSize) {
     WX_THROW( Wx_general_error() ); // value of seek_pos_end not identical to read EOF
   }
   str0.assign(text0.get(),FSize); // transfer text0 to str0
 }

 //
 // Read tfile line by line(1), and append to str1, check against str0
 //
 {
   WxRegFile f0;
   const size_t MaxLineLen=str0.size();
   size_t n_rd;
   std::auto_ptr<char> buf(new char[MaxLineLen+1]);
   WxStr str1;

   if((r=f0.open(tfile,O_RDONLY))!=OK) {
     WX_HERE(r); throw(r); // file open failure
   }
   for(;;) {
     if((r=f0.read_line(buf.get(),MaxLineLen+1,&n_rd))!=OK) {
       WX_HERE(r); throw(r);
     }
     if(n_rd==0) {  // eof
       if(str1!=str0) {
         WX_THROW( Wx_general_error() );
       }
       break;
     }
     str1.append(buf.get(),n_rd);  // append the line read
   }
 }

 //
 // Read tfile line by line(2), and append to str1, check against str0
 //
 {
   WxRegFile f0;
   size_t n_rd;
   WxStr str1;

   if((r=f0.open(tfile,O_RDONLY))!=OK) {
     WX_HERE(r); throw(r); // file open failure
   }
   for(;;) {
     if((r=f0.read_line(str1,&n_rd))!=OK) {
       WX_HERE(r); throw(r);
     }
     if(n_rd==0) {  // eof
       if(str1!=str0) {
         WX_THROW( Wx_general_error() );
       }
       break;
     }
   }
 }
};

static void t_writr(void)
{
  WxRet r;
  WxRegFile tmp;
  const char buf[2]={-2,-2};
  size_t count=5;   // anything not zero
  assert_default(tmp);
  if((r=tmp.open(WxPathName(Tmp_Name),O_RDWR|O_CREAT))!=OK) {
    WX_HERE(r); throw(r);
  }
  try {
    if(tmp.is_default()) {
      WX_THROW( Wx_general_error() );
    }
    if((r=tmp.write(buf,0,&count))!=OK) {
      WX_HERE(r); throw(r);
    }
    if(count!=0) {
      WX_THROW( Wx_general_error() );
    }
    if(tmp.get_pos()!=0) {
      WX_THROW( Wx_general_error() );
    }

    if((r=tmp.write(buf,1,&count))!=OK) {
      WX_HERE(r); throw(r);
    }
    if(count!=1) {
      WX_THROW( Wx_general_error() );
    }
    if(tmp.get_pos()!=1) {
      WX_THROW( Wx_general_error() );
    }

    if((r=tmp.write(buf,2,&count))!=OK) {
      WX_HERE(r); throw(r);
    }
    if(count!=2) {
      WX_THROW( Wx_general_error() );
    }
    if(tmp.get_pos()!=3) {
      WX_THROW( Wx_general_error() );
    }

    if((tmp.drain())!=OK) {
      WX_THROW( Wx_general_error() );
    }

    // read back
    tmp.set_pos(0);
    if(tmp.get_pos()!=0) {
      WX_THROW( Wx_general_error() );
    }
    char rbuf[4];
    if((r=tmp.max_read(rbuf,sizeof(rbuf),&count))!=OK) {
      WX_HERE(r); throw(r);
    }
    if(count!=3) {
      WX_THROW( Wx_general_error() );
    }
    if(tmp.get_pos()!=3) {
      WX_THROW( Wx_general_error() );
    }
    if(rbuf[0]!=buf[0]) {
      WX_THROW( Wx_general_error() );
    }
    if((rbuf[1]!=buf[0])||(rbuf[2]!=buf[1])) {
      WX_THROW( Wx_general_error() );
    }

    ::unlink(Tmp_Name.c_str());
  }
  catch(...) {
    ::unlink(Tmp_Name.c_str());
    throw;
  };
};

static void t3(void) 
{
 interleaved_read(TestFile_Empty);
 interleaved_read(TestFile_1A);
 interleaved_read(TestFile_2A);
 interleaved_read(WxPathName("chk_regfile.cpp"));
 interleaved_read(WxPathName("chk_regfile"));

 t_read_line(TestFile_1A);
 t_read_line(TestFile_2A);
 t_read_line(WxPathName("chk_regfile.cpp"));
 t_read_line(WxPathName("chk_regfile"));
 t_writr();
};

static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxregfile.h ...\n";
 if(std::strcmp(WxRegFile::class_name,"WxRegFile")!=0) {
   WX_THROW( Wx_general_error() );
 }
 t1();
 t2();
 t_exception();
 t3();
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
