/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Build: chk_pathname
*/

#include "wxpathname.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#if WXPATHNAME_VERSION!=8
#error Test code is for WXPATHNAME_VERSION 8
#endif

static bool chk_default(const WxPathName& p)
{
 static const WxPathName P0;
 if(p.is_default()==false) {
   return(false);
 }
 if(p==P0) {
 } else {
   return(false);
 }
 if(p!=P0) {
   return(false);
 }
 if(p.size()!=0) {
   return(false);
 }
 if(p.filename()!="") {
   return(false);
 }
 if(p.pathname()!="") {
   return(false);
 }
 return(true);
};

static bool chk_not_default(const WxPathName& p)
{
 return !chk_default(p);
};

static bool chk_equ(const WxPathName& p1,const WxPathName& p2)
{
 if(p1==p2) {
 } else {
   return(false);
 }
 if(p1!=p2) {
   WX_THROW( Wx_general_error() );
 }

 if(p1.size()!=p2.size()) {
   WX_THROW( Wx_general_error() );
 }

 if(p1.filename()!=p2.filename()) {
   WX_THROW( Wx_general_error() );
 }

 if(p1.pathname()!=p2.pathname()) {
   WX_THROW( Wx_general_error() );
 }
 return(true);
};

static bool chk_equ(const WxPathName& p1,const char* pstr)
{
 if(pstr==0) {
   WX_THROW( Wx_general_error() );
 }
 const size_t plen=::strlen(pstr);

 if(p1.size()!=plen) {
   return(false);
 }

 if(::memcmp(p1.pathname().c_str(),pstr,plen+1)==0) {
 } else {
   return(false);
 }
 if(p1.pathname()!=WxStr(pstr)) {
   WX_THROW( Wx_general_error() );
 }

 return(true);
};

#define CHK_EQU(expr1,expr2) if(chk_equ(expr1,expr2)==false) { WX_THROW( Wx_general_error() ); };
#define CHK_NOT_EQU(expr1,expr2) if(chk_not_equ(expr1,expr2)==false) { WX_THROW( Wx_general_error() ); };

#define CHK_DEFAULT(expr) if(chk_default(expr)==false) { WX_THROW( Wx_general_error() ); };
#define CHK_NOT_DEFAULT(expr) if(chk_not_default(expr)==false) { WX_THROW( Wx_general_error() ); };

//
// Test WxPathName (basic)
//
static void t1(void)
{
 WxRet r;
 WxPathName a0;
 WxPathName a1(a0);

 CHK_DEFAULT(a0);

 CHK_EQU(a1,a0);

 //
 // Test other constructors with relative name1
 //
 {
   const char t_str[]="relative-name1";
   std::string t_str1(t_str);
   WxStr t_str2(t_str);

   WxPathName a2(t_str);
   WxPathName a3(t_str1);
   WxPathName a4(t_str2);
   WxPathName a5(a4);

   CHK_NOT_DEFAULT(a2);
   CHK_NOT_DEFAULT(a3);
   CHK_NOT_DEFAULT(a4);
   CHK_NOT_DEFAULT(a5);

   CHK_EQU(a2,t_str);
   CHK_EQU(a3,t_str);
   CHK_EQU(a4,t_str);
   CHK_EQU(a5,t_str);

   if(a2.filename()!=t_str) {
     WX_THROW( Wx_general_error() );
   }
   if(a2.pathname()!=t_str) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // Test other constructors with relative name2
 //
 {
   const char t_str[]="dir/relative-name2";
   std::string t_str1(t_str);
   WxStr t_str2(t_str);

   WxPathName a2(t_str);
   WxPathName a3(t_str1);
   WxPathName a4(t_str2);
   WxPathName a5(a4);

   CHK_NOT_DEFAULT(a2);
   CHK_NOT_DEFAULT(a3);
   CHK_NOT_DEFAULT(a4);
   CHK_NOT_DEFAULT(a5);

   CHK_EQU(a2,t_str);
   CHK_EQU(a3,t_str);
   CHK_EQU(a4,t_str);
   CHK_EQU(a5,t_str);

   if(a2.filename()!="relative-name2") {
     WX_THROW( Wx_general_error() );
   }
   if(a2.pathname()!=t_str) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // Test other constructors with relative name3
 //
 {
   const char t_str[]="dir/relative-name3";
   const char t_str_t[]="dir/relative-name3/";
   std::string t_str1(t_str_t);
   WxStr t_str2(t_str_t);

   WxPathName a2(t_str_t);
   WxPathName a3(t_str1);
   WxPathName a4(t_str2);
   WxPathName a5(a4);

   CHK_NOT_DEFAULT(a2);
   CHK_NOT_DEFAULT(a3);
   CHK_NOT_DEFAULT(a4);
   CHK_NOT_DEFAULT(a5);

   CHK_EQU(a2,t_str);
   CHK_EQU(a3,t_str);
   CHK_EQU(a4,t_str);
   CHK_EQU(a5,t_str);

   if(a2.filename()!="relative-name3") {
     WX_THROW( Wx_general_error() );
   }
   if(a2.pathname()!=t_str) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // Test other constructors with pathname '/'
 //
 {
   const char t_str[]="/";
   std::string t_str1(t_str);
   WxStr t_str2(t_str);

   WxPathName a2(t_str);
   WxPathName a3(t_str1);
   WxPathName a4(t_str2);
   WxPathName a5(a4);

   CHK_NOT_DEFAULT(a2);
   CHK_NOT_DEFAULT(a3);
   CHK_NOT_DEFAULT(a4);
   CHK_NOT_DEFAULT(a5);

   CHK_EQU(a2,t_str);
   CHK_EQU(a3,t_str);
   CHK_EQU(a4,t_str);
   CHK_EQU(a5,t_str);

   if(a2.filename()!="") {
     WX_THROW( Wx_general_error() );
   }
   if(a2.pathname()!=t_str) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // Test other constructors with absolute pathname
 //
 {
   const char t_str[]="/dir/abs-name";
   const char t_str_t[]="/dir/abs-name/";
   std::string t_str1(t_str_t);
   WxStr t_str2(t_str_t);

   WxPathName a2(t_str_t);
   WxPathName a3(t_str1);
   WxPathName a4(t_str2);
   WxPathName a5(a4);

   CHK_NOT_DEFAULT(a2);
   CHK_NOT_DEFAULT(a3);
   CHK_NOT_DEFAULT(a4);
   CHK_NOT_DEFAULT(a5);

   CHK_EQU(a2,t_str);
   CHK_EQU(a3,t_str);
   CHK_EQU(a4,t_str);
   CHK_EQU(a5,t_str);

   if(a2.filename()!="abs-name") {
     WX_THROW( Wx_general_error() );
   }
   if(a2.pathname()!=t_str) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // Test reset
 //
 {
   WxPathName a4("abc");

   CHK_NOT_DEFAULT(a4);
   a4.reset();
   CHK_DEFAULT(a4);
 }

 //
 // Test assign
 //
 {
   const char t_cstr1[]="/dir/cstr";
   const char t_cstr2[]="/dir/cstr/";
   WxPathName a2,a3,a4;

   CHK_DEFAULT(a2);
   if((r=a2.assign(t_cstr1))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=a3.assign(t_cstr2))!=OK) {
     WX_HERE(r); throw(r);
   }

   CHK_EQU(a2,a3);
   if(a2.filename()!="cstr") {
     WX_THROW( Wx_general_error() ); 
   }
   if(a2.pathname()!=t_cstr1) {
     WX_THROW( Wx_general_error() ); 
   }

   a4.assign(a3);
   CHK_EQU(a4,a3);

   if((r=a4.assign(""))!=OK) {
     WX_HERE(r); throw(r);
   }
   CHK_DEFAULT(a4);

   const char t_cstr3[]="/";
   if((r=a2.assign(t_cstr3))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=a3.assign(std::string(t_cstr3)))!=OK) {
     WX_HERE(r); throw(r);
   }
   CHK_EQU(a2,a3);

   if(a2.filename()!="") {
     WX_THROW( Wx_general_error() ); 
   }
   if(a2.pathname()!=t_cstr3) {
     WX_THROW( Wx_general_error() ); 
   }

   if((r=a2.assign(t_cstr1,0))!=OK) {
     WX_HERE(r); throw(r);
   }
   CHK_DEFAULT(a2);

   if((r=a2.assign(t_cstr1,1))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(a2.pathname()!="/") {
     WX_THROW( Wx_general_error() ); 
   }
   if((r=a2.assign(t_cstr1,2))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(a2.pathname()!="/d") {
     WX_THROW( Wx_general_error() ); 
   }
   if((r=a2.assign(t_cstr1,6))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(a2.pathname()!="/dir/c") {
     WX_THROW( Wx_general_error() ); 
   }

   WxStr t_wxstr(t_cstr1);
   if((r=a2.assign(t_wxstr))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(a2.pathname()!=t_wxstr) {
     WX_THROW( Wx_general_error() ); 
   }
 }

 //
 // Test operator=
 //
 {
   const char t_cstr1[]="/dir/cstr";
   const char t_cstr2[]="/dir/cstr/";
   WxPathName a2,a3,a4;

   CHK_DEFAULT(a2);
   a2=t_cstr1;
   a3=t_cstr2;
   CHK_EQU(a2,a3);

   if(a2.filename()!="cstr") {
     WX_THROW( Wx_general_error() ); 
   }
   if(a2.pathname()!=t_cstr1) {
     WX_THROW( Wx_general_error() ); 
   }
   a4=a3;
   CHK_EQU(a4,a3);

   a4="";
   CHK_DEFAULT(a4);

   const char t_cstr3[]="/";
   a2=t_cstr3;
   a3=std::string(t_cstr3);
   CHK_EQU(a2,a3);

   if(a2.filename()!="") {
     WX_THROW( Wx_general_error() ); 
   }
   if(a2.pathname()!=t_cstr3) {
     WX_THROW( Wx_general_error() ); 
   }

   WxStr t_wxstr(t_cstr1);
   a2=t_wxstr;
   if(a2.pathname()!=t_wxstr) {
     WX_THROW( Wx_general_error() ); 
   }
 }

};

//
// Test WxPathName (error report)
//
static void t2(void)
{
 WxRet r;

 // WxPathName(const WxStr&)
 {
  try {
    WxPathName tmp(WxStr("//"));
    WX_THROW( Wx_general_error() );
  }
  catch(const WxPathName::Fault& e) {
    if(e!=WXM_EINVAL) {
      throw;
    }
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // WxPathName(const std::string&)
 {
  try {
    WxPathName tmp(std::string("//"));
    WX_THROW( Wx_general_error() );
  }
  catch(const WxPathName::Fault& e) {
    if(e!=WXM_EINVAL) {
      throw;
    }
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // WxPathName(const char*)
 {
  try {
    WxPathName tmp("//");
    WX_THROW( Wx_general_error() );
  }
  catch(const WxPathName::Fault& e) {
    if(e!=WXM_EINVAL) {
      throw;
    }
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };

  try {
    WxPathName tmp((const char*)0);
    WX_THROW( Wx_general_error() );
  }
  catch(const WxPathName::Fault& e) {
    if(e!=WXM_EFAULT) {
      throw;
    }
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // assign(const WxStr&)
 {
  WxPathName tmp;
  if((r=tmp.assign(WxStr("//")))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // assign(const std::string&)
 {
  WxPathName tmp;
  if((r=tmp.assign(std::string("//")))!=WXM_EINVAL) {
    WX_THROW(r);
  }
 }

 // assign(const char*)
 {
  WxPathName tmp;
  if((r=tmp.assign("//"))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.assign((const char*)0))!=WXM_EFAULT) {
    WX_THROW(r);
  }
 }

 // assign(const char*)
 {
  WxPathName tmp;
  if((r=tmp.assign("//",size_t(2)))!=WXM_EINVAL) {
    WX_THROW(r);
  }
  if((r=tmp.assign((const char*)0))!=WXM_EFAULT) {
    WX_THROW(r);
  }
  if((r=tmp.assign("",size_t(-1)))!=WXM_ENAMETOOLONG) {
    WX_THROW(r);
  }
 }

 // operator =(const WxStr&)
 {
  try {
    WxPathName tmp;
    tmp=WxStr("//");
    WX_THROW( Wx_general_error() );
  }
  catch(const WxPathName::Fault& e) {
    if(e!=WXM_EINVAL) {
      throw;
    }
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }

 // operator =(const char*)
 {
  try {
    WxPathName tmp;
    tmp="//";
    WX_THROW( Wx_general_error() );
  }
  catch(const WxPathName::Fault& e) {
    if(e!=WXM_EINVAL) {
      throw;
    }
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };

  try {
    WxPathName tmp;
    tmp=(const char*)0;
    WX_THROW( Wx_general_error() );
  }
  catch(const WxPathName::Fault& e) {
    if(e!=WXM_EFAULT) {
      throw;
    }
    // FALLTHROUGH
  }
  catch(...) {
    WX_THROW( Wx_general_error() );
  };
 }
};

//
// Test Wx functions
//
static void t_wxf(void)
{
 const char* TmpName="tmp.tmp.tmp";
 WxRet r;

 ::unlink(TmpName);
 if((r=Wx::access(WxPathName(TmpName),F_OK))==OK) {
   WX_HERE(r); throw(r);
 }
 
 // create TmpName file
 const int fd=::open(TmpName,O_RDONLY|O_CREAT,S_IRUSR|S_IWUSR);
 if(fd<0) {
   WX_THROW( Wx_bad_errno(errno) );
 } else {
   ::close(fd);
 }

 try {
   if((r=Wx::access(WxPathName(TmpName),F_OK))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=Wx::access(WxPathName(TmpName),R_OK))!=OK) {
     if(::umask(0777)&S_IRUSR) {
       std::cerr << TmpName << " is ::created(O_RDONLY) but ::access(R_OK) failed?\n";
       WX_HERE(r); throw(r);
     }
   }
   if((r=Wx::access(WxPathName(TmpName),W_OK))!=OK) {
     if(::umask(0777)&S_IWUSR) {
       WX_HERE(r); throw(r);
     }
   }
   if((r=Wx::chmod(WxPathName(TmpName),S_IWUSR))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=Wx::access(WxPathName(TmpName),W_OK))!=OK) {
     WX_HERE(r); throw(r);
   }

   // chown
   if((r=Wx::chown(WxPathName(TmpName),0,0))!=WXM_EPERM) {
     WX_HERE(r); throw(r);
   }
   WxPathName cwd;
   if((r=Wx::getcwd(cwd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=Wx::chdir(cwd))!=OK) {
     WX_HERE(r); throw(r);
   }

   // unlink
   if((r=Wx::unlink(WxPathName(TmpName)))!=OK) {
     WX_HERE(r); throw(r);
   }
 }
 catch(...) {
   ::unlink(TmpName);
   throw;
 }
};

static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxpathname.cpp ...\n";
 if(WxStr(WxPathName::class_name)!="WxPathName") {
   WX_THROW( Wx_general_error() );
 }
 t1(); 
 t2();
 t_wxf();
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
