/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Test functions in wxnum.h

   Build: chk_num
*/

#include "wxret.h"
#include "wxnum.h"
#include <limits>
#include <cstdlib>

#include <iostream>
#if WXNUM_VERSION!=8
#error Test code is for WXNUM_VERSION 8
#endif

//
// Test Wx::append_number(WxStr,int,int)   random number test
//
static void t1_long(void)
{
 WxRet r;
 for(int i=0; i<100000; ++i) {
   long num=std::rand();
   int radix=(std::rand()%35)+2;
   WxStr str;
   if((r=Wx::append_number(str,num,radix))!=OK) {
     WX_THROW( Wx_general_error() );
   }

   char* endptr=(char*)str.c_str();
   long v=strtol(str.c_str(),&endptr,radix);
   switch(v) {
     case LONG_MIN:    // underflow
          std::cerr << "underflow= " << str << std::endl;
          break;
     case LONG_MAX:    // overflow
          std::cerr << "overflow= " << str << std::endl;
          break;
     default:
          if(*endptr!=0) {
            std::cerr << "not all converted back= " << str << std::endl;
          }
          if(v!=num) {
            std::cerr << "conversion failed= " << num << "!=" << v << std::endl;
          }
   }
 }
}

//
// Test Wx::append_number(WxStr,int,int)   random number test
//
static void t1_ulong(void)
{
 WxRet r;
 for(int i=0; i<100000; ++i) {
   ulong num=std::rand();
   int radix=(std::rand()%35)+2;
   WxStr str;
   if((r=Wx::append_number(str,num,radix))!=OK) {
     WX_THROW( Wx_general_error() );
   }

   char* endptr=(char*)str.c_str();
   ulong v=strtoll(str.c_str(),&endptr,radix);
   if(v!=num) {
      std::cerr << "conversion failed= " << num << "!=" << v << std::endl;
   }
 }
}

//
// Test Wx::append_number(WxStr,int,int)
// Test value long_max and long_min in radix 2-36
//
static void t2_long(void)
{
 WxRet r;
 long num;
 WxStr str;
 char* endptr;
 long v;

 for(int radix=2; radix<=36; ++radix) {
   str.reset();
   num=std::numeric_limits<long>::max();
   if((r=Wx::append_number(str,num,radix))!=OK) {
     WX_THROW( Wx_general_error() );
   }
   endptr=(char*)str.c_str();
   v=strtol(str.c_str(),&endptr,radix);
   if(v!=num) {
     WX_THROW( Wx_general_error() );
   }
 
   str.reset();
   num=std::numeric_limits<long>::min();
   if((r=Wx::append_number(str,num,radix))!=OK) {
     WX_THROW( Wx_general_error() );
   }
   endptr=(char*)str.c_str();
   v=strtol(str.c_str(),&endptr,radix);
   if(v!=num) {
     WX_THROW( Wx_general_error() );
   }
 }
}

//
// Test Wx::append_number(WxStr,int,int)
// Test value ulong_max and ulong_min in radix 2-36
//
static void t2_ulong(void)
{
 WxRet r;
 ulong num;
 WxStr str;
 char* endptr;
 ulong v;

 for(int radix=2; radix<=36; ++radix) {
   str.reset();
   num=std::numeric_limits<ulong>::max();
   if((r=Wx::append_number(str,num,radix))!=OK) {
     WX_THROW( Wx_general_error() );
   }
   endptr=(char*)str.c_str();
   v=strtoll(str.c_str(),&endptr,radix);
   if(v!=num) {
     WX_THROW( Wx_general_error() );
   }
 
   str.reset();
   num=std::numeric_limits<ulong>::min();
   if((r=Wx::append_number(str,num,radix))!=OK) {
     WX_THROW( Wx_general_error() );
   }
   endptr=(char*)str.c_str();
   v=strtoll(str.c_str(),&endptr,radix);
   if(v!=num) {
     WX_THROW( Wx_general_error() );
   }
 }
};

//
// Check abs
//
static void t_abs(void)
{
  // test int
  {
   typedef int TestType;
   const int TSIZE=9;
   static const TestType test_value[]=
          {0,1,2,-1,-2,INT_MAX,INT_MAX-1,INT_MIN,INT_MIN+1};
   for(int i=0; i<TSIZE; i++) {
     const TestType n=test_value[i];
     try {
       if(std::abs(n)!=WxNum::abs(n)) {
         WX_THROW( Wx_general_error() );
       }
     }
     catch(const WxNum::Fault& e) {
       if(e!=WXM_MATHNEG) {
         WX_THROW( Wx_general_error() );
       }
     }
   }
  }

  // test long
  {
   typedef long int TestType;
   const int TSIZE=9;
   static const TestType test_value[]=
          {0,1,2,-1,-2,INT_MAX,INT_MAX-1,INT_MIN,INT_MIN+1};
   for(int i=0; i<TSIZE; i++) {
     const TestType n=test_value[i];
     try {
       if(std::labs(n)!=WxNum::abs(n)) {
         WX_THROW( Wx_general_error() );
       }
     }
     catch(const WxNum::Fault& e) {
       if(e!=WXM_MATHNEG) {
         WX_THROW( Wx_general_error() );
       }
     }
   }
  }

  // test long long
  {
   typedef long long int TestType;
   const int TSIZE=9;
   static const TestType test_value[]=
          {0,1,2,-1,-2,INT_MAX,INT_MAX-1,INT_MIN,INT_MIN+1};
   for(int i=0; i<TSIZE; i++) {
     const TestType n=test_value[i];
     try {
       if(std::llabs(n)!=WxNum::abs(n)) {
         WX_THROW( Wx_general_error() );
       }
     }
     catch(const WxNum::Fault& e) {
       if(e!=WXM_MATHNEG) {
         WX_THROW( Wx_general_error() );
       }
     }
   }
  }
};

//
// Check div
//
static void t_div(void)
{
  // test int div
  {
   typedef int TestType;
   TestType dent,dvsr;
   std::div_t a1,a2;

   dent=0,dvsr=4;
   a1=std::div(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=INT_MAX,dvsr=1;
   a1=std::div(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=INT_MIN,dvsr=1;
   a1=std::div(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=INT_MAX-1,dvsr=37;
   a1=std::div(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=INT_MIN,dvsr=-1;
   try { 
     a2=WxNum::div(dent,dvsr);
     WX_THROW( Wx_general_error() );
   }
   catch(const WxNum::Fault& e) {
     if(e!=WXM_MATHNEG) {
       WX_THROW( Wx_general_error() );
     }
   }
  
   dent=1,dvsr=0;
   try {
     a2=WxNum::div(dent,dvsr);
     WX_THROW( Wx_general_error() );
   }
   catch(const WxNum::Fault& e) {
     if(e!=WXM_DIVZERO) {
       WX_THROW( Wx_general_error() );
     }
   }
  }

  // test long int div
  {
   typedef long TestType;
   TestType dent,dvsr;
   std::ldiv_t a1,a2;

   dent=0,dvsr=4;
   a1=std::ldiv(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=LONG_MAX,dvsr=1;
   a1=std::ldiv(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=LONG_MIN,dvsr=1;
   a1=std::ldiv(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=LONG_MAX-1,dvsr=37;
   a1=std::ldiv(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=LONG_MIN,dvsr=-1;
   try { 
     a2=WxNum::div(dent,dvsr);
     WX_THROW( Wx_general_error() );
   }
   catch(const WxNum::Fault& e) {
     if(e!=WXM_MATHNEG) {
       WX_THROW( Wx_general_error() );
     }
   }
  
   dent=1,dvsr=0;
   try {
     a2=WxNum::div(dent,dvsr);
     WX_THROW( Wx_general_error() );
   }
   catch(const WxNum::Fault& e) {
     if(e!=WXM_DIVZERO) {
       WX_THROW( Wx_general_error() );
     }
   }
  }

  // test long long div
  {
   typedef long long int TestType;
   TestType dent,dvsr;
   std::lldiv_t a1,a2;

   dent=0,dvsr=4;
   a1=std::lldiv(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=LONG_LONG_MAX,dvsr=1;
   a1=std::lldiv(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=LONG_LONG_MIN,dvsr=1;
   a1=std::lldiv(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=LONG_LONG_MAX-1,dvsr=37;
   a1=std::lldiv(dent,dvsr);
   a2=WxNum::div(dent,dvsr);
   if((a1.quot!=a2.quot)||(a1.rem!=a2.rem)) {
     WX_THROW( Wx_general_error() );
   }

   dent=LONG_LONG_MIN,dvsr=-1;
   try { 
     a2=WxNum::div(dent,dvsr);
     WX_THROW( Wx_general_error() );
   }
   catch(const WxNum::Fault& e) {
     if(e!=WXM_MATHNEG) {
       WX_THROW( Wx_general_error() );
     }
   }
  
   dent=1,dvsr=0;
   try {
     a2=WxNum::div(dent,dvsr);
     WX_THROW( Wx_general_error() );
   }
   catch(const WxNum::Fault& e) {
     if(e!=WXM_DIVZERO) {
       WX_THROW( Wx_general_error() );
     }
   }
  }
};

//
// Check sqrt
//
static void t_sqrt(void)
{
 {
   unsigned int v=1000;
   if(WxNum::sqrt(v)!=31) {
     WX_THROW( Wx_general_error() );
   }
   v=0x7fffffff;
   if(WxNum::sqrt(v)!=46340) {
     WX_THROW( Wx_general_error() );
   }
   v=0xffffffff;
   if(WxNum::sqrt(v)!=65535) {
     WX_THROW( Wx_general_error() );
   }
 }

 {
   unsigned long long v=1000;
   if(WxNum::sqrt(v)!=31) {
     WX_THROW( Wx_general_error() );
   }
   v=0x7fffffff;
   if(WxNum::sqrt(v)!=46340) {
     WX_THROW( Wx_general_error() );
   }
   v=0xffffffff;
   if(WxNum::sqrt(v)!=65535) {
     WX_THROW( Wx_general_error() );
   }
 }
};

static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxnum.cpp...\n";
 t1_long();
 t1_ulong();
 t2_long();
 t2_ulong();
 t_abs();
 t_div();
 t_sqrt();
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
