/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Note: No thread involved. This is basic check.

   Build: make chk_mutex
*/

#include "wxmutex.h"
#include <iostream>
#include <cassert>
#include <cstring>
#include "wxstr.h"

#if WXMUTEX_VERSION!=8
#error Test code is for WXMUTEX_VERSION 8
#endif

//
// Test mutex throws Wx_general_error as specificd
// This test is common for all type of mutexes.
//
static void test_throw_general_error(void)
{
 //
 // Test constructor throw Wx_general_error
 //
 try {
   WxMutex bad_mtx(WxMutex::MutexType(100));
 }
 catch(const Wx_general_error&) {
 }

 //
 // Test destructor throw Wx_general_error
 //
 try {
   WxMutex bad_mtx;
   std::memset(&bad_mtx,0,sizeof(bad_mtx)); 
 }
 catch(const Wx_general_error&) {
 }

 //
 // Test lock() throw Wx_general_error
 //
 try {
   char zbuf[sizeof(WxMutex)];
   WxMutex& bad_mtx=reinterpret_cast<WxMutex&>(zbuf);
   bad_mtx.lock();
   assert("");
 }
 catch(const Wx_general_error&) {
 }

 //
 // Test trylock() throw Wx_general_error
 //
 try {
   char zbuf[sizeof(WxMutex)];
   WxMutex& bad_mtx=reinterpret_cast<WxMutex&>(zbuf);
   bad_mtx.trylock();
   assert("");
 }
 catch(const Wx_general_error&) {
 }

 //
 // Test unlock() throw Wx_general_error
 //
 try {
   char zbuf[sizeof(WxMutex)];
   WxMutex& bad_mtx=reinterpret_cast<WxMutex&>(zbuf);
   bad_mtx.unlock();
   assert("");
 }
 catch(const Wx_general_error&) {
 }

//
// Test Fast Mutex Features
//
};

//
// Test mutex throws WxMutex::Fault as specificd (static version of functions)
// This test is common for all type of mutexes.
//
static void test_throw_class_fault(void)
{
 //
 // test lock(&mtx) throw Fault(WXM_EDEADLK)
 //
 try {
   WxMutex mtx(WxMutex::ErrorCheck);
   WxRet r=mtx.lock();
   if(r!=OK) {
     WX_THROW( Wx_general_error() );
   }
   WxMutex::lock(&mtx);   // should throw Fault(WXM_EDEADLK)
   assert("");
 }
 catch(const WxMutex::Fault& e) {
   if(e!=WXM_EDEADLK) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // test trylock(&mtx) throw Fault(WXM_EDEADLK)
 //
 try {
   WxMutex mtx;
   WxRet r=mtx.lock();
   if(r!=OK) {
     WX_THROW( Wx_general_error() );
   }
   WxMutex::trylock(&mtx);   // should throw Fault(WXM_EBUSY)
   assert("");
 }
 catch(const WxMutex::Fault& e) {
   if(e!=WXM_EBUSY) {
     WX_THROW( Wx_general_error() );
   }
 }

 //
 // test unlock(&mtx) throw Fault(WXM_EDEADLK)
 // 
 try {
   WxMutex mtx(WxMutex::ErrorCheck);
   WxMutex::unlock(&mtx);   // should throw Fault(WXM_EPERM)
   assert("");
 }
 catch(const WxMutex::Fault& e) {
   if(e!=WXM_EPERM) {
     WX_THROW( Wx_general_error() );
   }
 }
}

static void test_fast_mutex(void)
{
 WxRet r;
 WxMutex mtx(WxMutex::Fast);

 if((r=mtx.lock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.trylock())!=WXM_EBUSY) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.unlock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.trylock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.unlock())!=OK) {
   WX_HERE(r); throw(r);
 }
 //
 // Unlock mutex twice was tested ok(fast mutex), keep this feature.
 // man. says 'always returns to the unlocked state'
 //
 if((r=mtx.unlock())!=OK) {
   WX_HERE(r); throw(r);
 }

 // static functions
 WxMutex::lock(&mtx);
 WxMutex::unlock(&mtx);
 WxMutex::trylock(&mtx);
 WxMutex::unlock(&mtx);
 WxMutex::unlock(&mtx);  // see above unlock twice
};

static void test_recursive_mutex(void)
{
 WxRet r;
 WxMutex mtx(WxMutex::Recursive);

 if((r=mtx.lock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.trylock())!=OK) {  // lock twice
   WX_HERE(r); throw(r);
 }
 if((r=mtx.lock())!=OK) {     // lock thrice
   WX_HERE(r); throw(r);
 }
 if((r=mtx.unlock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.unlock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.unlock())!=OK) {
   WX_HERE(r); throw(r);
 }

 //
 // Unlock mutex twice was tested to report WXM_EPERM(recursive mutex),
 // keep this feature for the test.
 // man. says 'always returns to the unlocked state'
 //
 if((r=mtx.unlock())!=WXM_EPERM) {
   WX_HERE(r); throw(r);
 }

 // static functions
 WxMutex::lock(&mtx);
 WxMutex::trylock(&mtx);
 WxMutex::lock(&mtx);
 WxMutex::unlock(&mtx);
 WxMutex::unlock(&mtx);  // see above unlock twice
 WxMutex::unlock(&mtx);
};

static void test_errcheck_mutex(void)
{
 WxRet r;
 WxMutex mtx(WxMutex::ErrorCheck);

 if((r=mtx.lock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.trylock())!=WXM_EBUSY) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.lock())!=WXM_EDEADLK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.unlock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.trylock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.unlock())!=OK) {
   WX_HERE(r); throw(r);
 }
 //
 // Unlock mutex twice was tested to report WXM_EPERM(errorcheck mutex),
 // keep this feature for the test.
 // man. says 'always returns to the unlocked state'
 //
 if((r=mtx.unlock())!=WXM_EPERM) {
   WX_HERE(r); throw(r);
 }

 // static functions
 WxMutex::lock(&mtx);
 WxMutex::unlock(&mtx);
 WxMutex::trylock(&mtx);
 WxMutex::unlock(&mtx);
};

//
// Test WxLock
//
static void test_WxLock(void)
{
 WxRet r;
 WxMutex mtx; 

 {
  WxMutex mtx(WxMutex::ErrorCheck);
  if((r=mtx.lock())!=OK) {
    WX_HERE(r); throw(r);
  }
  try {
    WxLock aa(mtx);   // should throw WXM_EDEADLK
    assert("");
  }
  catch(const WxLock::Fault& e) {
    if(e!=WXM_EDEADLK) {
      WX_HERE(r); throw(r);
    }
  }
 }

 {
  // test destructor unlock
  WxLock aa(mtx);
  if((r=mtx.trylock())!=WXM_EBUSY) {
    WX_HERE(r); throw(r);
  }
 }  // aa should unlock mtx leaving the scope
 if((r=mtx.trylock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.unlock())!=OK) {
   WX_HERE(r); throw(r);
 }

 { // Test WxLock::unlock()
  WxLock aa(mtx);
  if((r=mtx.trylock())!=WXM_EBUSY) {
    WX_HERE(r); throw(r);
  }
  if((r=aa.unlock())!=OK) {       // aa.unlock should unlock mtx
    WX_HERE(r); throw(r);
  }
  if((r=mtx.trylock())!=OK) {
    WX_HERE(r); throw(r);   // WxLock::unlock failed
  }
  if((r=mtx.unlock())!=OK) {
    WX_HERE(r); throw(r);
  }
  if((r=aa.unlock())!=WXM_ENOENT) {
    WX_HERE(r); throw(r);
  }
 }

 //
 // Test WxLock in a series of construct/and destruct
 //
 for(int i=0; i<10; i++) {
   WxLock aa(mtx);
   if((r=mtx.trylock())!=WXM_EBUSY) {
     WX_HERE(r); throw(r);
   }
   if((r=aa.unlock())!=OK) {
     WX_HERE(r); throw(r);
   }
 }

 if((r=mtx.trylock())!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=mtx.unlock())!=OK) {
   WX_HERE(r); throw(r);
 }
};

static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking WxMutex class...\n";
 if(WxStr(WxMutex::class_name)!="WxMutex") {
   WX_THROW( Wx_general_error() );
 }
 if(WxStr(WxLock::class_name)!="WxLock") {
   WX_THROW( Wx_general_error() );
 }
 test_fast_mutex();
 test_recursive_mutex();
 test_errcheck_mutex();
 test_throw_general_error();
 test_throw_class_fault();
 test_WxLock();
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
