/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Check WxLnkFile public members

   Build: make chk_regfile

   Note: Test data files should exist in the working directory
         Files may be created in the working directory and unlinked
*/

#include "wxlnkfile.h"
#include <unistd.h>

#if WXLNKFILE_VERSION!=8
#error Test code is for WXLNKFILE_VERSION 8
#endif

static const WxPathName RefLnkName("testfile.lnkf.L1"); // points to "nofile"
static const WxPathName RefLnkNameValue("nofile");

static const WxPathName TmpLnkName("tmp.lnkfile.tmp");

//
// Assert ff is default
//
static void assert_default(const WxLnkFile& ff)
{
 WxRet r;

 if(ff.is_default()==false) {
   WX_THROW( Wx_general_error() );
 }

 // stat
 {
   WxFileStat stt;
   if((r=ff.stat(stt))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Setup the test 
//
static void setup_test(void)
{
 ::unlink(RefLnkName.pathname().c_str());
 if(::symlink(RefLnkNameValue.pathname().c_str(),
              RefLnkName     .pathname().c_str())!=0) {
   WX_THROW( Wx_bad_errno(errno) );  // cannot create the test file
 }
};

//
// Undo setup_test
//
static void close_test(void)
{
 ::unlink(RefLnkName.pathname().c_str());
};

//
// Check basic operations on default object
//
static void t1(void)
{
 WxRet r;

 WxLnkFile NObj;     // non-default object for test as argument
 assert_default(NObj);
 if((r=NObj.open(RefLnkName))!=OK) {
   WX_HERE(r); throw(r);
 }
 
 // stat
 {
   WxFileStat stt;
   if((r=NObj.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_lnk()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 // constructor
 {
   WxLnkFile tmp0;
   WxLnkFile tmp1(tmp0);
   assert_default(tmp0);
   assert_default(tmp1);

   WxLnkFile tmp2(NObj);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   WxPathName pn;
   if((r=tmp2.read(pn))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(pn!=RefLnkNameValue) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset
 {
   WxLnkFile tmp;
   assert_default(tmp);
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   WxLnkFile tmp2(NObj);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp2.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp2);
 }

 // read
 {
   WxLnkFile tmp;
   assert_default(tmp);
   WxPathName pn;
   if((r=tmp.read(pn))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 //
 // assignf(WxLnkFile)
 //
 {
   WxLnkFile tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxLnkFile()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(NObj))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxLnkFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Check basic operations on non-default object
//
static void t2(void)
{
 WxRet r;
 
 WxLnkFile NObj;     // non-default object for the test
 assert_default(NObj);
 if((r=NObj.open(RefLnkName))!=OK) {
   WX_HERE(r); throw(r);
 }
 
 // stat
 {
   WxFileStat stt;
   if((r=NObj.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_lnk()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 // constructor
 {
 }

 // reset
 {
   WxLnkFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 // read
 {
   WxLnkFile tmp(NObj);
   WxPathName pn;
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read(pn))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(pn!=RefLnkNameValue) {
     WX_THROW( Wx_general_error() );
   }
 }

 // open (WXM_NDEFAULT)
 {
   WxLnkFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.open(RefLnkName))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxLnkFile())
 //
 {
   WxLnkFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxLnkFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Check basic operations for static functions
//
static void t3(void)
try {
 WxRet r;
 //
 // Create TmpLnkName and unlink it
 //
 ::unlink(TmpLnkName.pathname().c_str());
 if((r=WxLnkFile::create(WxPathName("tmp.tmp.tmp"),TmpLnkName))!=OK) {
   WX_HERE(r); throw(r);
 }

 // check if it's there
 {
   WxLnkFile tmp;
   if((r=tmp.open(TmpLnkName))!=OK) {
     WX_HERE(r); throw(r);
   }
 }

 ::unlink(TmpLnkName.pathname().c_str());

 // check if it's unlinked
 {
   WxLnkFile tmp;
   if((r=tmp.open(TmpLnkName))!=WXM_ENOENT) {
     WX_HERE(r); throw(r);
   }
 }
}
catch(...) {
 ::unlink(TmpLnkName.pathname().c_str());
 throw;
};

static void t_exception(void)
{
 //
 // Test copy constructor exception
 //
 try {
 }
 catch(const WxLnkFile::Fault& e) {
   // FALLTHROUGH 
 }
 catch(...) {
   throw;
 };

 //
 // Test destructor exception
 //
 {
 }
};

static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxlnkfile.h ...\n";
 if(std::strcmp(WxLnkFile::class_name,"WxLnkFile")!=0) {
   WX_THROW( Wx_general_error() );
 }
 try {
   setup_test();
   t1();
   t2();
   t3();
   t_exception();
   close_test();
 }
 catch(...) {
   close_test();
   throw;
 }
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
