/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Check WxFifoFile public members

   Build: make chk_fifofile

   Note: Test data files should exist in the working directory
         Files may be created in the working directory and unlinked
*/

#include "wxfifofile.h"
#include <memory>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#if WXFIFOFILE_VERSION!=8
#error Test code is for WXFIFOFILE_VERSION 8
#endif

static const WxPathName FileName_Empty("testfile.fifo.empty"); // size=0
static const WxStr FileName_Tmp="tmp.chk_fifofile.tmp";
static const char FifoMsg[]="Fifo test message";
static const size_t FifoMsgLen=std::strlen(FifoMsg);

static WxFifoFile TFile__Empty_;
static const WxFifoFile& TFile_Empty(TFile__Empty_);

//
// Assert ff is default
//
static void assert_default(const WxFifoFile& ff)
{
 WxRet r;

 if(ff.is_default()==false) {
   WX_THROW( Wx_general_error() );
 }
 if(ff.fh().is_default()==false) {
   WX_THROW( Wx_general_error() );
 }

 // stat
 {
   WxFileStat stt;
   if((r=ff.stat(stt))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
 }

};

//
// Setup the test 
//
static void setup_test(void)
{
 WxRet r;

 //
 // create and open TFile_Empty for FileName_Empty
 //
 assert_default(TFile_Empty);
 ::unlink(FileName_Empty.pathname().c_str());

 if((r=WxFifoFile::create(FileName_Empty,S_IRUSR|S_IWUSR))!=OK) {
   WX_HERE(r); throw(r);
 }
 if((r=TFile__Empty_.open(FileName_Empty,O_RDWR))!=OK) {
   WX_HERE(r); throw(r); // Test file (FileName_Empty) open failure
 }
 if(TFile__Empty_.is_default()) {
   WX_THROW( Wx_general_error() );
 }

 // stat TFile_Empty
 {
   WxFileStat stt;
   if((r=TFile_Empty.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(stt.is_fifo()==false) {
     WX_THROW( Wx_general_error() );
   }
 }
};

//
// Undo setup_test
//
static void close_test(void)
{
 ::unlink(FileName_Empty.pathname().c_str());
};

//
// Check basic operations on default object
//
static void t1(void)
{
 WxRet r;

 // constructor
 {
   WxFifoFile tobj;
   WxFifoFile tmp1(tobj);
   assert_default(tobj);
   assert_default(tmp1);

   WxFifoFile tmp2(TFile_Empty);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp2.fh()==TFile_Empty.fh()) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset
 {
   WxFifoFile tmp;
   assert_default(tmp);
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   WxFifoFile tmp2(TFile_Empty);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp2.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp2);
 }

 // read,read_max,write,drain,open(ENOENT)
 {
   WxFifoFile tmp;
   assert_default(tmp);
   char buf[2];
   if((r=tmp.read(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.max_read(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.write(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.drain())!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.open(WxPathName("tmp.a,b,1,0,z,tmp"),O_RDONLY))!=WXM_ENOENT) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 //
 // assignf(WxFileHandle)
 //
 {
   WxFifoFile tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxFileHandle()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(TFile_Empty.fh()))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==TFile_Empty.fh()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxFileHandle()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxFifoFile)
 //
 {
   WxFifoFile tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxFifoFile()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(TFile_Empty))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==TFile_Empty.fh()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxFifoFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Check basic operations on non-default object
//
static void t2(void)
{
 WxRet r;
 
 // stat EmptyFile
 {
   WxFileStat stt;
   if((r=TFile_Empty.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(stt.is_fifo()==false) {
     WX_THROW( Wx_general_error() );
   }
   if(stt.size()!=0) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset
 {
   WxFifoFile tmp(TFile_Empty);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 // (zero byte) read,read_max,write,drain
 {
   WxFifoFile tmp(TFile_Empty);
   char buf[2]={-1,-1};
   size_t n_rd=5;   // anything >1
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.max_read(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.write(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.drain())!=WXM_EINVAL) {
     WX_HERE(r); throw(r); // ref fsync(2)
   }
 }

 // open (WXM_NDEFAULT)
 {
   WxFifoFile tmp(TFile_Empty);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.open(WxPathName("abc"),O_RDONLY))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxFileHandle())
 //
 {
   WxFifoFile tmp(TFile_Empty);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxFileHandle()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxFifoFile())
 //
 {
   WxFifoFile tmp(TFile_Empty);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxFifoFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 // mkpipe
 {
   WxFifoFile p0(TFile_Empty),p1;
   if(p0.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   assert_default(p1);

   if((r=WxFifoFile::mkpipe(p0,p1))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
};

static void t_exception(void)
{
 //
 // Test copy constructor exception
 //
 try {
   char tbuf[sizeof(WxFifoFile)];
   std::memset(tbuf,-2,sizeof(tbuf));
   //
   // as -2 is known not default fd and no such valid handle. 
   // construct from tbuf should cause Fault exception
   //
   WxFifoFile tt2(reinterpret_cast<const WxFifoFile&>(tbuf));
   WX_THROW( Wx_general_error() );  // no expected exception
 }
 catch(const WxFifoFile::Fault& e) {
   if(e!=WXM_EBADF) {
     WxRet r=e;
     WX_HERE(r); throw(r);
   }
   // FALLTHROUGH 
 }
 catch(...) {
   throw;
 };

 //
 // Test destructor exception
 //
 {
   WxFifoFile tmp;
   std::memset(&tmp,-2,sizeof(tmp));
   // destructor of tmp is now not to cause exception
 }

};

//
// Test FIFO communication (read)
// Open fname FIFO and read
//
static int porcess_read_fifo(const WxPathName& fname) throw()
try {
 WxRet r;
 WxFifoFile ff;
 //
 // Open FIFO (fname) to read
 //
 if((r=ff.open(fname,O_RDONLY))!=OK) {
   WX_HERE(r); throw(r);
 }

 //
 // Receive message from the FIFO (FifoMsg expected)
 //
 size_t n_rd;
 char rbuf[64];
 if(sizeof(rbuf)<FifoMsgLen) {
   WX_THROW( Wx_general_error() );   // rbuf not long enough, reedit rbuf size.
 }
 if((r=ff.read(rbuf,sizeof(rbuf),&n_rd))!=OK) {  // note: using max_read may block
   WX_HERE(r); throw(r);
 }
 if(n_rd!=FifoMsgLen) {
   WX_THROW( Wx_general_error() ); 
 }
 if(std::memcmp(rbuf,FifoMsg,n_rd)!=0) {
   WX_THROW( Wx_general_error() );   // not expected server message
 }
 
 // expect to read eof and return(0);
 if((r=ff.read(rbuf,sizeof(rbuf),&n_rd))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(n_rd!=0) {
   WX_THROW( Wx_general_error() ); 
 }
 return(0);
}
catch(const WxRet& e) {
 std::cerr << "process_read_fifo: "  << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << "process_read_fifo :" << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << "process_read_fifo :" << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << "process_read_fifo : unknown exception" << std::endl;
 return(-1);
};


//
// Test read/write (in different process)
//
static void t_fifo_rw(void)
try {
 WxRet r;
 WxFifoFile ff;
 pid_t child_pid;
 int   child_status;

 //
 // Create FIFO (FileName_Tmp)
 //
 ::unlink(FileName_Tmp.c_str());
 if((r=WxFifoFile::create(WxPathName(FileName_Tmp),S_IRUSR|S_IWUSR))!=OK) {
   WX_HERE(r); throw(r);
 }

 //
 // Fork to run process_read_fifo(FileName_Tmp)
 //
 child_pid=::fork();
 if(child_pid<0) {
   WX_THROW( Wx_bad_errno(errno) );
 }
 if(child_pid==0) {
   // In child process
   ::exit( porcess_read_fifo(WxPathName(FileName_Tmp)) );
 }

 //
 // Open FIFO (FileName_Tmp) to write
 //
 if((r=ff.open(WxPathName(FileName_Tmp),O_WRONLY))!=OK) {
   WX_HERE(r); throw(r);
 }

 //
 // Send FifoMsg into the FIFO ff
 //
 size_t n_wr;
 if((r=ff.write(FifoMsg,FifoMsgLen,&n_wr))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(n_wr!=FifoMsgLen) {
   WX_THROW( Wx_general_error() );
 }
 if((r=ff.reset())!=OK) {
   WX_HERE(r); throw(r);
 }

 if(::wait(&child_status)!=child_pid) {
   WX_THROW( Wx_bad_errno(errno) );
 }
 if(WIFEXITED(child_status)==0) {
   WX_THROW( Wx_general_error() );  // child process was not terminated voluntarily
 }
 if(WEXITSTATUS(child_status)!=0) {
   WX_THROW( Wx_general_error() );  // child process not exist(0)
 }
 ::unlink(FileName_Tmp.c_str());
}
catch(...) {
 ::unlink(FileName_Tmp.c_str());
 throw;
};

//
// Test pipe
//
static void t_pipe(void)
{
  WxRet r;
  WxFifoFile pin,pout;
  assert_default(pin);
  assert_default(pout);

  if((r=WxFifoFile::mkpipe(pin,pout))!=OK) {
    WX_HERE(r); throw(r);
  }
  if(pin.is_default()) {
    WX_THROW( Wx_general_error() ); 
  }
  if(pout.is_default()) {
    WX_THROW( Wx_general_error() ); 
  }

  // stat
  {
    WxFileStat stt;
    if((r=pin.stat(stt))!=OK) {
      WX_HERE(r); throw(r);
    }
    if(stt.is_fifo()==false) {
      WX_THROW( Wx_general_error() ); 
    }

    if((r=pout.stat(stt))!=OK) {
      WX_HERE(r); throw(r);
    }
    if(stt.is_fifo()==false) {
      WX_THROW( Wx_general_error() ); 
    }
  }
};

static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxfifofile.h ...\n";
 if(std::strcmp(WxFifoFile::class_name,"WxFifoFile")!=0) {
   WX_THROW( Wx_general_error() );
 }
 try {
   setup_test();
   t1();
   t2();
   t_exception();
   t_fifo_rw();
   t_pipe();
   close_test();
 }
 catch(...) {
   close_test();
   throw;
 }
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
