/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Check WxDirFile public members

   Build: make chk_dirfile

   Note: Test data files should exist in the working directory
         Files may be created in the working directory and unlinked
*/

#include "wxdirfile.h"
#include <sys/types.h>
#include <unistd.h>
#include <sys/wait.h>

#if WXDIRFILE_VERSION!=8
#error Test code is for WXDIRFILE_VERSION 8
#endif

static const WxPathName RefDirName(".");
static const WxPathName TmpDirName("tmp.dirfile.tmp");

//
// Assert ff is default
//
static void assert_default(const WxDirFile& ff)
{
 WxRet r;

 if(ff.is_default()==false) {
   WX_THROW( Wx_general_error() );
 }
 if(ff.wx_ref_count()!=0) {
   WX_THROW( Wx_general_error() );
 }

 // stat
 {
   WxFileStat stt;
   if((r=ff.stat(stt))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Setup the test 
//
static void setup_test(void)
{
};

//
// Undo setup_test
//
static void close_test(void)
{
};

//
// Check basic operations on default object
//
static void t1(void)
{
 WxRet r;

 WxDirFile NObj;     // non-default object for test as argument
 assert_default(NObj);
 if((r=NObj.open(RefDirName))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }
 
 // stat
 {
   WxFileStat stt;
   if((r=NObj.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_dir()==false) {
     WX_THROW( Wx_general_error() );
   }
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }

 // constructor
 {
   WxDirFile tmp0;
   WxDirFile tmp1(tmp0);
   assert_default(tmp0);
   assert_default(tmp1);

   WxDirFile tmp2(NObj);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp2.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }
   if(NObj.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }

 // ref count
 {
   if(NObj.wx_ref_count()!=1) {
     WX_THROW( Wx_general_error() );
   }
   WxDirFile tmp(NObj);
   if(NObj.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }

   {
     WxDirFile tmp_tt(tmp);
     if(NObj.wx_ref_count()!=3) {
       WX_THROW( Wx_general_error() );
     }
     if(tmp.wx_ref_count()!=3) {
       WX_THROW( Wx_general_error() );
     }
     if(tmp_tt.wx_ref_count()!=3) {
       WX_THROW( Wx_general_error() );
     }
   }
   if(NObj.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }

 // reset
 {
   WxDirFile tmp;
   assert_default(tmp);
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   WxDirFile tmp2(NObj);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp2.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }
   if(NObj.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp2.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp2);
 }
 if(NObj.wx_ref_count()!=1) {
   //std::cerr << "NObj(" << NObj.wx_ref_count() << std::endl;
   WX_THROW( Wx_general_error() );
 }

 // read,rewind
 {
   WxDirFile tmp;
   assert_default(tmp);
   WxDirEnt dent;
   if((r=tmp.read(dent))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.rewind())!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }

 //
 // assignf(WxDirFile)
 //
 {
   WxDirFile tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxDirFile()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(NObj))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(NObj.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxDirFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }
};

//
// Check basic operations on non-default object
//
static void t2(void)
{
 WxRet r;
 
 WxDirFile NObj;     // non-default object for the test
 assert_default(NObj);
 if((r=NObj.open(RefDirName))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }
 
 // stat
 {
   WxFileStat stt;
   if((r=NObj.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_dir()==false) {
     WX_THROW( Wx_general_error() );
   }
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }

 // constructor
 {
 }

 // reset
 {
   WxDirFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }
   if(NObj.wx_ref_count()!=2) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
   if(NObj.wx_ref_count()!=1) {
     WX_THROW( Wx_general_error() );
   }
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }

 // read,rewind
 {
   WxDirFile tmp(NObj);
   WxDirEnt dent1;
   WxDirEnt dent2;
   WxDirEnt dent3;
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read(dent1))!=OK) { // should be at least . .. thisfile (3 files)
     WX_HERE(r); throw(r);
   }
   if((r=tmp.read(dent2))!=OK) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.read(dent3))!=OK) {
     WX_HERE(r); throw(r);
   }
   // rewind and read again to check
   if((r=tmp.rewind())!=OK) {
     WX_HERE(r); throw(r);
   }
   WxDirEnt dent_tt;
   if((r=tmp.read(dent_tt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(dent_tt.filename()!=dent1.filename()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read(dent_tt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(dent_tt.filename()!=dent2.filename()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read(dent_tt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(dent_tt.filename()!=dent3.filename()) {
     WX_THROW( Wx_general_error() );
   }
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }

 // open (WXM_NDEFAULT)
 {
   WxDirFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.open(RefDirName))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }

 //
 // assignf(WxDirFile())
 //
 {
   WxDirFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxDirFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
 if(NObj.wx_ref_count()!=1) {
   WX_THROW( Wx_general_error() );
 }
};

//
// Check basic operations for static functions
//
static void t3(void)
try {
 WxRet r;
 //
 // Create TmpDirName and unlink it
 //
 if((r=WxDirFile::create(TmpDirName,S_IRUSR|S_IWUSR))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(::access(TmpDirName.pathname().c_str(),F_OK)!=0) {
   WX_THROW( Wx_bad_errno(errno) );
 }

 if((r=WxDirFile::unlink(TmpDirName))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(::access(TmpDirName.pathname().c_str(),F_OK)==ENOENT) {
   WX_THROW( Wx_general_error() );
 } else {
   if(errno!=ENOENT) {
     WX_THROW( Wx_bad_errno(errno) );
   }
 }
}
catch(...) {
 ::rmdir(TmpDirName.pathname().c_str());
 throw;
};

static void t_exception(void)
{
 //
 // Test copy constructor exception
 //
 try {
 }
 catch(const WxDirFile::Fault& e) {
   // FALLTHROUGH 
 }
 catch(...) {
   throw;
 };

 //
 // Test destructor exception
 //
 {
 }
};

static WxStr chdr="+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n";
int main(void) throw()
try {
 std::cout << "Checking wxdirfile.h ...\n";
 if(std::strcmp(WxDirFile::class_name,"WxDirFile")!=0) {
   WX_THROW( Wx_general_error() );
 }
 try {
   setup_test();
   t1();
   t2();
   t3();
   t_exception();
   close_test();
 }
 catch(...) {
   close_test();
   throw;
 }
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
