/* Copyright is licensed under GNU LGPL.                (I.J. Wang, 2003)

   Check WxChrFile public members

   Build: make chk_chrfile

   Note: Test data files should exist in the working directory
         Files may be created in the working directory and unlinked
*/

#include "wxchrfile.h"
#include <sys/types.h>
#include <unistd.h>

#if WXCHRFILE_VERSION!=8
#error Test code is for WXCHRFILE_VERSION 8
#endif

static const WxPathName RefFileName("/dev/tty");

//
// Assert ff is default
//
static void assert_default(const WxChrFile& ff)
{
 WxRet r;

 if(ff.is_default()==false) {
   WX_THROW( Wx_general_error() );
 }
 if(ff.fh().is_default()==false) {
   WX_THROW( Wx_general_error() );
 }

 // stat
 {
   WxFileStat stt;
   if((r=ff.stat(stt))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Setup the test 
//
static void setup_test(void)
{
};

//
// Undo setup_test
//
static void close_test(void)
{
};

//
// Check basic operations on default object
//
static void t1(void)
{
 WxRet r;

 WxChrFile NObj;     // non-default object for test as argument
 assert_default(NObj);
 if((r=NObj.open(RefFileName,O_RDONLY))!=OK) {
   WX_HERE(r); throw(r);
 }
 
 // stat
 {
   WxFileStat stt;
   if((r=NObj.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_chr()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 // constructor
 {
   WxChrFile tmp0;
   WxChrFile tmp1(tmp0);
   assert_default(tmp0);
   assert_default(tmp1);

   WxChrFile tmp2(NObj);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp2.fh()==NObj.fh()) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset
 {
   WxChrFile tmp;
   assert_default(tmp);
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   WxChrFile tmp2(NObj);
   if(tmp2.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp2.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp2);
 }

 // read,read_max,write,drain,bind
 {
   WxChrFile tmp;
   assert_default(tmp);
   char buf[2];
   if((r=tmp.read(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.max_read(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.write(buf,sizeof(buf),0))!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   if((r=tmp.drain())!=WXM_EBADF) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 //
 // assignf(WxFileHandle)
 //
 {
   WxChrFile tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxFileHandle()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(NObj.fh()))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==NObj.fh()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxFileHandle()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxChrFile)
 //
 {
   WxChrFile tmp;
   assert_default(tmp);
   if((r=tmp.assignf(WxChrFile()))!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);

   if((r=tmp.assignf(NObj))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==NObj.fh()) {
     WX_THROW( Wx_general_error() );
   }

   if((r=tmp.assignf(WxChrFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
};

//
// Check basic operations on non-default object
//
static void t2(void)
{
 WxRet r;
 
 // stat NObj
 WxChrFile NObj;     // non-default object for the test
 assert_default(NObj);
 if((r=NObj.open(RefFileName,O_RDWR))!=OK) {
   WX_HERE(r); throw(r);
 }
 if(NObj.is_default()) {
   WX_THROW( Wx_general_error() );
 }
 
 // stat
 {
   WxFileStat stt;
   if((r=NObj.stat(stt))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(stt.is_chr()==false) {
     WX_THROW( Wx_general_error() );
   }
 }

 // constructor
 {
   WxChrFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if(tmp.fh()==NObj.fh()) {
     WX_THROW( Wx_general_error() );
   }
 }

 // reset
 {
   WxChrFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.reset())!=OK) {
     WX_HERE(r); throw(r);
   }
   assert_default(tmp);
 }

 // (zero byte) read,read_max,write,drain
 {
   WxChrFile tmp(NObj);
   char buf[2]={-1,-1};
   size_t n_rd=5;   // anything >1
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.read(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.max_read(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.write(buf,0,&n_rd))!=OK) {
     WX_HERE(r); throw(r);
   }
   if(n_rd!=0) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.drain())!=OK) {
     if(r!=WXM_EINVAL) {
       WX_HERE(r); throw(r); // ref fsync(2)
     }
   }
 }

 // open (WXM_NDEFAULT)
 {
   WxChrFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.open(RefFileName,O_RDONLY))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxFileHandle())
 //
 {
   WxChrFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxFileHandle()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }

 //
 // assignf(WxChrFile())
 //
 {
   WxChrFile tmp(NObj);
   if(tmp.is_default()) {
     WX_THROW( Wx_general_error() );
   }
   if((r=tmp.assignf(WxChrFile()))!=WXM_NDEFAULT) {
     WX_HERE(r); throw(r);
   }
 }
};

static void t_exception(void)
{
 //
 // Test copy constructor exception
 //
 try {
   char tbuf[sizeof(WxChrFile)];
   std::memset(tbuf,-2,sizeof(tbuf));
   //
   // as -2 is known not default fd and no such valid handle. 
   // construct from tbuf should cause Fault exception
   //
   WxChrFile tt2(reinterpret_cast<const WxChrFile&>(tbuf));
   WX_THROW( Wx_general_error() );  // no expected exception
 }
 catch(const WxChrFile::Fault& e) {
   if(e!=WXM_EBADF) {
     WxRet r=e;
     WX_HERE(r); throw(r);
   }
   // FALLTHROUGH 
 }
 catch(...) {
   throw;
 };

 //
 // Test destructor exception
 //
 {
   WxChrFile tmp;
   std::memset(&tmp,-2,sizeof(tmp));
   // destructor of tmp is now not to cause exception for this setting
 }
};

static const WxStr chdr(
                  "+-------------------------+\n"
                  "| main() caught exception:|\n"
                  "+-------------------------+\n");
int main(void) throw()
try {
 std::cout << "Checking wxchrfile.h ...\n";
 if(std::strcmp(WxChrFile::class_name,"WxChrFile")!=0) {
   WX_THROW( Wx_general_error() );
 }
 try {
   setup_test();
   t1();
   t2();
   t_exception();
   close_test();
 }
 catch(...) {
   close_test();
   throw;
 }
 std::cout << "Checked OK\n";
 return(0);
}
catch(const WxRet& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_general_error& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_errno& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_except& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const Wx_bad_alloc& e) {
 std::cerr << chdr << Wx::what_is(e) << std::endl;
 return(-1);
}
catch(const std::exception& e) {
 std::cerr << chdr << "std::exception" << std::endl;
 return(-1);
}
catch(...) {
 std::cerr << chdr << "unknown exception" << std::endl;
 return(-1);
};
