/*
   wav.c  --  Functions for the manipulation of WAV files and data
   
   SNDCAP Library
   
   Copyright (C) 2007 Laszlo Menczel
   
   This is free software with NO WARRANTY. Distributed under
   the GNU Library General Public Licence (LGPL) version 2.1.
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "sndcap.h"
#include "sndcaplib.h"

//========================================================================

// Macros to read and write values using big-endian byte order.
// Assume that the host byte order is big-endian while file data
// is little-endian.

#define READ_WORD_BIGENDIAN(fp, p) \
{					\
  if (fread((p) + 1, 1, 1, (fp)) != 1)	\
    return 0;				\
  if (fread((p), 1, 1, (fp)) != 1)	\
    return 0;				\
}

#define WRITE_WORD_BIGENDIAN(fp, p) \
{					\
  if (fwrite((p) + 1, 1, 1, (fp)) != 1)	\
    return 0;				\
  if (fwrite((p), 1, 1, (fp)) != 1)	\
    return 0;				\
}

#define READ_DWORD_BIGENDIAN(fp, p) \
{					\
  if (fread((p) + 3, 1, 1, (fp)) != 1)	\
    return 0;				\
  if (fread((p) + 2, 1, 1, (fp)) != 1)	\
    return 0;				\
  if (fread((p) + 1, 1, 1, (fp)) != 1)	\
    return 0;				\
  if (fread((p), 1, 1, (fp)) != 1)	\
    return 0;				\
}

#define WRITE_DWORD_BIGENDIAN(fp, p) \
{					\
  if (fwrite((p) + 3, 1, 1, (fp)) != 1)	\
    return 0;				\
  if (fwrite((p) + 2, 1, 1, (fp)) != 1)	\
    return 0;				\
  if (fwrite((p) + 1, 1, 1, (fp)) != 1)	\
    return 0;				\
  if (fwrite((p), 1, 1, (fp)) != 1)	\
    return 0;				\
}

//========================================================================

#define FSIZE_OFS		4
#define MAX_STR_SCANLEN		1024

enum { UNDEF_ENDIAN, ENDIAN_LITTLE, ENDIAN_BIG };

static char sbuf[MAX_STR_SCANLEN];

static int host_byte_order = UNDEF_ENDIAN;

// chunk tag strings

static char riff_str[4] = { 'R', 'I', 'F', 'F' };
static char wave_str[4] = { 'W', 'A', 'V', 'E' };
static char fmt_str[4]  = { 'f', 'm', 't', ' ' };
static char data_str[4] = { 'd', 'a', 't', 'a' };

//========================================================================
// Local functions
//========================================================================

static char *strend(char *s)
{
  char *p = s;
  int i = 0;

  while (i < MAX_STR_SCANLEN && *p != 0)
    p++;
    
  if (i == MAX_STR_SCANLEN)
    return s;
  else
    return p;
}

//========================================================================

static void set_byte_order(void)
{
  int x = 1;
  char *p = (char *) &x;
  
  if (*p == 1)
    host_byte_order = ENDIAN_LITTLE;
  else
    host_byte_order = ENDIAN_BIG;
}

//=============================================================================

static int write_word(FILE *fp, word val)
{
  byte *p = (byte *) &val;
  
  if (host_byte_order == ENDIAN_BIG)
    WRITE_WORD_BIGENDIAN(fp, p)
  else
  {
    if (fwrite(p, 1, 2, fp) != 2)
      return 0;
  }
  
  return 1;
}

//=============================================================================

static int read_word(FILE *fp, word *val)
{
  byte *p = (byte *) val;
        
  if (host_byte_order == ENDIAN_BIG)
    READ_WORD_BIGENDIAN(fp, p)
  else
  {
    if (fread(p, 1, 2, fp) != 2)
      return 0;
  }
  
  return 1;
}

//=============================================================================

static int write_dword(FILE *fp, dword val)
{
  byte *p = (byte *) &val;
  
  if (host_byte_order == ENDIAN_BIG)
    WRITE_DWORD_BIGENDIAN(fp, p)
  else
  {
    if (fwrite(p, 1, 4, fp) != 4)
      return 0;
  }
  
  return 1;
}

//=============================================================================

static int read_dword(FILE *fp, dword *val)
{
  byte *p = (byte *) val;
        
  if (host_byte_order == ENDIAN_BIG)
    READ_DWORD_BIGENDIAN(fp, p)
  else
  {
    if (fread(p, 1, 4, fp) != 4)
      return 0;
  }
  
  return 1;
}

//========================================================================

static int read_audio_data(void *buf, wavinfo_t *info, FILE *fp)
{
  int i, count;
  byte *p;
  
  count = info->frames * info->numchan;

  if (info->numbits == 16)	// we have to use the correct byte order here
  {
    if (host_byte_order == ENDIAN_BIG)
    {
      p = (byte *) buf;
      for (i = 0; i < count; i++, p += 2)
        READ_WORD_BIGENDIAN(fp, p)
    }
    else
    {
      count *= 2;
      if (fread(buf, 1, count, fp) != count)
        return 0;
    }
  }
  else				// just bytes, read the whole block
  {
    if (fread(buf, 1, count, fp) != count)
      return 0;
  }

  return 1;
}

//========================================================================

static int write_audio_data(void *buf, wavinfo_t *info, FILE *fp)
{
  int i, count;
  byte *p;
  
  count = info->frames * info->numchan;

  if (info->numbits == 16)	// we have to use the correct byte order here
  {
    if (host_byte_order == ENDIAN_BIG)
    {
      p = (byte *) buf;
      for (i = 0; i < count; i++, p += 2)
        WRITE_WORD_BIGENDIAN(fp, p)
    }
    else
    {
      count *= 2;
      if (fwrite(buf, 1, count, fp) != count)
        return 0;
    }
  }
  else				// just bytes, read the whole block
  {
    if (fwrite(buf, 1, count, fp) != count)
      return 0;
  }

  return 1;
}

//========================================================================

static int check_wav_info(wavinfo_t *info)
{
  if (info->name[0] == 0)
  {
    __scap_err = WAVERR_FILE_NAME;
    return 0;
  }

  if (info->numchan < 1 || info->numchan > 2)	// only MONO or STEREO allowed
  {
    __scap_err = WAVERR_NUM_CHAN;
    return 0;
  }
    
  switch (info->freq)
  {
    case 8000:
    case 11025:
    case 22050:
    case 44100:
      break;
      
    default:
      __scap_err = WAVERR_FREQ;
      return 0;
  }

  if (info->numbits != 8 && info->numbits != 16)
  {
    __scap_err = WAVERR_BITS;
    return 0;
  }

  if (info->frames < 1)		// there must be at least one sound frame
  {
    __scap_err = WAVERR_SAMPLES;
    return 0;
  }
    
  __scap_err = WAVERR_NONE;
  return 1;
}

//========================================================================

static int load_format(FILE *fp, wavinfo_t *info)
{
  char str[5];
  dword dw;
  word ww;
  int fmt_len, fmt_tag;

  str[4] = 0;					// string termination

  //======== read 'RIFF' tag and file size

  if (fread(str, 1, 4, fp) != 4)
    ERR(WAVERR_FILE_READ)

  if (strcmp(str, "RIFF") != 0)
    ERR(WAVERR_RIFF)

  if (! read_dword(fp, &dw))			// file size
    ERR(WAVERR_FILE_READ)

  //======== read 'WAVE' tag

  if (fread(str, 1, 4, fp) != 4)
    ERR(WAVERR_FILE_READ)

  if (strcmp(str, "WAVE") != 0)
    ERR(WAVERR_WAVE)

  //======== read 'fmt' tag

  if (fread(str, 1, 4, fp) != 4)
    ERR(WAVERR_FILE_READ)

  if (strcmp(str, "fmt ") != 0)
    ERR(WAVERR_FMT_CHUNK)

  //======== read and check format specifiers

  if (! read_dword(fp, &dw))
    ERR(WAVERR_FILE_READ)
  else
    fmt_len = (int) dw;

  if (! read_word(fp, &ww))
    ERR(WAVERR_FILE_READ)
  else
    fmt_tag = (int) ww;

  if (fmt_len != 16 || fmt_tag != 1)
    ERR(WAVERR_TAG)

  //======== read format parameters

  if (! read_word(fp, &ww))
    ERR(WAVERR_FILE_READ)
  else
    info->numchan = (int) ww;
    
  if (! read_dword(fp, &dw))
    ERR(WAVERR_FILE_READ)
  else
    info->freq = (int) dw;

  if (! read_dword(fp, &dw))	// average bytes per second
    ERR(WAVERR_FILE_READ)

  if (! read_word(fp, &ww))	// block align
    ERR(WAVERR_FILE_READ)

  if (! read_word(fp, &ww))
    ERR(WAVERR_FILE_READ)
  else
    info->numbits = (int) ww;

  //======== read 'data' tag and data byte count

  if (fread(str, 1, 4, fp) != 4)
    ERR(WAVERR_FILE_READ)

  if (strcmp(str, "data") != 0)
    ERR(WAVERR_DATA_CHUNK)

  if (! read_dword(fp, &dw))	// data count in bytes
    ERR(WAVERR_FILE_READ)

  info->frames = (int) dw;

  if (info->numbits == 16)
    info->frames /= 2;

  info->frames /= info->numchan;
  
  RETURN(1)
}
   
//========================================================================
// Public functions
//========================================================================

int wav_load_format(char *name, wavinfo_t *info)
{
  FILE *fp;
  
  if (host_byte_order == UNDEF_ENDIAN)
    set_byte_order();

  if (name == NULL || info == NULL)
    ERR(WAVERR_NULL_PTR)
    
  if (name[0] == 0 || strlen(name) > WAV_MAX_PATHLEN)
    ERR(WAVERR_FILE_NAME)

  fp = fopen(name, "rb");
  if (fp == NULL)
    ERR(WAVERR_FILE_OPEN)
    
  strcpy(info->name, name);
  return load_format(fp, info);
}

//========================================================================

int wav_load(char *name, void *buf, wavinfo_t *info)
{
  FILE *fp;
  int res;
  
  if (host_byte_order == UNDEF_ENDIAN)
    set_byte_order();

  if (name == NULL || buf == NULL || info == NULL)
    ERR(WAVERR_NULL_PTR)
    
  if (name[0] == 0 || strlen(name) > WAV_MAX_PATHLEN)
    ERR(WAVERR_FILE_NAME)

  fp = fopen(name, "rb");
  if (fp == NULL)
    ERR(WAVERR_FILE_OPEN)
    
  strcpy(info->name, name);

  if (! load_format(fp, info))
  {
    fclose(fp);
    return 0;
  }
  
  if (! check_wav_info(info))
  {
    fclose(fp);
    return 0;
  }
      
  res = read_audio_data(buf, info, fp);
  fclose(fp);
  
  if (! res)
    ERR(WAVERR_FILE_READ)

  RETURN(1)
}

//========================================================================

int wav_save(void *buf, wavinfo_t *info)
{
  FILE *fp;
  int file_size, data_size;
  word ww;
  dword dw;
  
  if (host_byte_order == UNDEF_ENDIAN)
    set_byte_order();

  if (buf == NULL || info == NULL)
    ERR(WAVERR_NULL_PTR)
    
  if (! check_wav_info(info))
    return 0;

  fp = fopen(info->name, "wb");
  if (fp == NULL)
    ERR(WAVERR_FILE_OPEN)
  
  if (fwrite(riff_str, 1, 4, fp) != 4)
    goto err_exit;
  
  dw = 0;			// dummy file size
  if (! write_dword(fp, dw))
    goto err_exit;
  
  if (fwrite(wave_str, 1, 4, fp) != 4)
    goto err_exit;

  if (fwrite(fmt_str, 1, 4, fp) != 4)
    goto err_exit;

  dw = 16;
  if (! write_dword(fp, dw))
    goto err_exit;

  ww = 1;
  if (! write_word(fp, ww))
    goto err_exit;

  ww = (word) info->numchan;
  if (! write_word(fp, ww))
    goto err_exit;

  dw = (dword) info->freq;
  if (! write_dword(fp, dw))
    goto err_exit;

  dw = (dword) (info->freq * info->numchan * (info->numbits / 8));	// avg. bytes
  if (! write_dword(fp, dw))
    goto err_exit;

  ww = (word) (info->numchan * (info->numbits / 8));			// block align
  if (! write_word(fp, ww))
    goto err_exit;

  ww = (word) info->numbits;
  if (! write_word(fp, ww))
    goto err_exit;

  if (fwrite(data_str, 1, 4, fp) != 4)
    goto err_exit;

  data_size = info->frames * (info->numbits / 8) * info->numchan;
  dw = (dword) data_size;
  if (! write_dword(fp, dw))
    goto err_exit;

  file_size = ftell(fp) + data_size;

  if (! write_audio_data(buf, info, fp))
    goto err_exit;

  fseek(fp, FSIZE_OFS, SEEK_SET);
  if (! write_dword(fp, dw))
    goto err_exit;
  
  fclose(fp);
  RETURN(1)

err_exit:

  fclose(fp);
  unlink(info->name);
  ERR(WAVERR_FILE_WRITE)
}

//========================================================================

char *wav_info_str(wavinfo_t *info)
{
  if (info == NULL)
    ERR_P(WAVERR_NULL_PTR)

  sbuf[0] = 0;
      
  sprintf(strend(sbuf), "name:         %s\n", info->name);
  sprintf(strend(sbuf), "channels:     %d\n", info->numchan);
  sprintf(strend(sbuf), "frequency:    %d\n", info->freq);
  sprintf(strend(sbuf), "sample size:  %d\n", info->numbits);
  sprintf(strend(sbuf), "frame count:  %d\n", info->frames);

  RETURN(sbuf)
}

//========================================================================

int wav_compare_format(wavinfo_t *a, wavinfo_t *b)
{
  if (a == NULL || b == NULL)
    ERR(WAVERR_NULL_PTR)
  else
    __scap_err = WAVERR_NONE;
    
  if (
       a->numchan != b->numchan ||
       a->numbits != b->numbits ||
       a->freq != b->freq
     )
     return 0;
  
  return 1;
}
 //========================================================================

void *wav_convert(void *buf, wavinfo_t *sinfo, wavinfo_t *dinfo)
{
  if (buf == NULL || sinfo == NULL || dinfo == NULL)
    ERR_P(WAVERR_NULL_PTR)

  printf("*** waw_convert() IS NOT IMPLEMENTED IN THIS VERSION ***\n");
  ERR_P(WAVERR_NOT_YET)
}
