/*
   sndcap.h -- Public header

   SNDCAP Voice Capture Library
   
   Copyright (C) 2007 Laszlo Menczel
   
   This is free software with NO WARRANTY. Distributed under
   the GNU Library General Public Licence (LGPL) version 2.1.
*/

#if ! defined _SNDCAP_H
#define _SNDCAP_H

#if defined __cplusplus
extern "C" {
#endif

//======== Unsigned types

#if ! defined byte
  typedef unsigned char byte;
#endif

#if ! defined word
  typedef unsigned short word;
#endif

#if ! defined dword
  typedef unsigned long dword;
#endif

//======== 'Sleep()' for Linux

#if defined _LINUX
  void Sleep(int msec);
#endif

//======== Error handling

enum
{
  CAPERR_NONE,

  // capture
  CAPERR_INIT,
  CAPERR_BAD_ARG,
  CAPERR_ALLOC,
  CAPERR_NULL_PTR,
  CAPERR_BAD_BUF,
  CAPERR_DEV_OPEN,
  CAPERR_IN_RECORD,
  CAPERR_NOT_RECORD,
  CAPERR_START,
  CAPERR_BUF_ADD,
  CAPERR_THREAD,

  // ring buffer
  RBERR_BAD_ARG,
  RBERR_BAD_BUF,
  RBERR_BUF_STATIC,
  RBERR_EMPTY,
  
  // WAV
  WAVERR_FILE_NAME,
  WAVERR_FILE_OPEN,
  WAVERR_FILE_READ,
  WAVERR_FILE_WRITE,
  WAVERR_RIFF,
  WAVERR_WAVE,
  WAVERR_FMT_CHUNK,
  WAVERR_TAG,
  WAVERR_NUM_CHAN,
  WAVERR_DATA_CHUNK,
  WAVERR_BITS,
  WAVERR_FREQ,
  WAVERR_SAMPLES,
  WAVERR_SAME_FORMAT,
  WAVERR_NOT_YET,

  CAPERR_ERR_NUM
};
  
#define RBERR_NONE	CAPERR_NONE
#define RBERR_ALLOC	CAPERR_ALLOC
#define RBERR_NULL_PTR	CAPERR_NULL_PTR
#define WAVERR_NONE	CAPERR_NONE
#define WAVERR_NULL_PTR	CAPERR_NULL_PTR

char *scap_errmsg(int code);
int  scap_last_error(void);

//======== Sound capture

typedef struct ringbuf_s ringbuf_t;

enum
{
  SCAP_8K = 8000,
  SCAP_11K = 11025,
  SCAP_22K = 22050,
  SCAP_44K = 44100
};

enum { SCAP_BYTE = 8, SCAP_WORD = 16 };
enum { SCAP_MONO = 1, SCAP_STEREO = 2 };

int  scap_init(int chan, int freq, int bits, ringbuf_t *buf);
int  scap_start(void);
int  scap_stop(void);
void scap_exit(void);

//======== Ringbuffer management

enum { RBUF_TAIL, RBUF_HEAD };
enum { RBUF_BYTE, RBUF_WORD, RBUF_DWORD };

struct ringbuf_s
{
  byte  *data;		// buffer containing data
  int   dynamic;	// 1 = dynamically allocated
  int   incr;		// size of data items
  int   size;		// number of bytes in buffer
  int   count;		// current number of data items in buffer
  int   tail;		// pointer to first data item
  int   head;		// pointer to first empty location
  int   overflow;	// overflow flag
};

ringbuf_t *rbuf_create(int count, int size);

int   rbuf_destroy(ringbuf_t *buf);
int   rbuf_reset(ringbuf_t *buf);
int   rbuf_incr(ringbuf_t *buf, int which);
int   rbuf_empty(ringbuf_t *buf);
int   rbuf_full(ringbuf_t *buf);
int   rbuf_insert(ringbuf_t *buf, void *data, int data_type);
int   rbuf_remove(ringbuf_t *buf, void *data, int data_type);

//======== WAV manipulation

#define WAV_MAX_PATHLEN		255

typedef struct
{
  char name[WAV_MAX_PATHLEN + 1];
  int  freq;
  int  numbits;
  int  numchan;
  int  frames;
} wavinfo_t; 

int   wav_load(char *name, void *buf, wavinfo_t *info);
int   wav_load_format(char *name, wavinfo_t *info);
int   wav_compare_format(wavinfo_t *a, wavinfo_t *b);
int   wav_save(void *buf, wavinfo_t *info);
char *wav_info_str(wavinfo_t *info);
void *wav_convert(void *buf, wavinfo_t *sinfo, wavinfo_t *dinfo);

//======== Error handling

int  scap_last_error(void);
char *scap_errmsg(int code);

#if defined __cplusplus
}
#endif

#endif  /* _SNDCAP_H */
