/*
   capture_alsa.c  --  Linux specific functions using the ALSA library
   
   SCAP Voice Capture Library
   
   Copyright (C) 2007 Laszlo Menczel
   
   This is free software with NO WARRANTY. Distributed under
   the GNU Library General Public Licence (LGPL) version 2.1.
*/

#include <stdio.h>
#include <string.h>
#include <alsa/asoundlib.h> 

#include "sndcap.h"
#include "sndcaplib.h"

//============================================================================

static snd_pcm_t *dev = NULL;
static snd_async_handler_t *cb_handle = NULL;
static snd_pcm_uframes_t period_size;

static int processing = 0;

static char *dev_name = "plughw:0,0";

static void boo(void)
{
  printf("Boo!\n");
}

//============================================================================

void cb_func(snd_async_handler_t *handle)
{
  int ret;
       
  if (! processing)
    return;

  ret = snd_pcm_readi(dev, __scap_buf->data + __scap_buf->head, period_size);

  if (ret < 0)
  {
    if (ret == -EPIPE)
      snd_pcm_prepare(dev);
  }
  else
    rbuf_incr(__scap_buf, RBUF_HEAD);
}
 
//============================================================================

int __scap_open_device(void)
{
  snd_pcm_hw_params_t *hw_param = NULL;
  snd_pcm_sw_params_t *sw_param = NULL;
  snd_pcm_uframes_t buf_size;
  int ret, dir, periods;
  unsigned int freq;
  enum _snd_pcm_format sample_format;

  //======== open the sound device

  ret = snd_pcm_open(&dev, dev_name, SND_PCM_STREAM_CAPTURE, 0);
  if (ret < 0)
    ERR(CAPERR_DEV_OPEN)

  /*******************************/
  /* Set the hardware parameters */
  /*******************************/

  //======== allocate & initialize parameter structure

  ret = snd_pcm_hw_params_malloc(&hw_param);
  if (ret < 0)
    goto err_exit;

  ret = snd_pcm_hw_params_any(dev, hw_param);
  if (ret < 0)
    goto err_exit;

  //======== set access type

  ret = snd_pcm_hw_params_set_access(dev, hw_param, SND_PCM_ACCESS_RW_INTERLEAVED);
  if (ret < 0)
    goto err_exit;

  //======== set sample format

  if (__scap_dat_size == SCAP_BYTE)
    sample_format = SND_PCM_FORMAT_U8;
  else
    sample_format = SND_PCM_FORMAT_S16;

  ret = snd_pcm_hw_params_set_format(dev, hw_param, sample_format);
  if (ret < 0)
    goto err_exit;

  //======== set number of channels

  ret = snd_pcm_hw_params_set_channels(dev, hw_param, __scap_num_chan);
  if (ret < 0)
  {
    printf("%s\n", snd_strerror(ret));
    goto err_exit;
  }

  //======== set sampling rate

  dir = 0;
  freq = (unsigned int) __scap_cap_freq;
  ret = snd_pcm_hw_params_set_rate_near(dev, hw_param, &freq, &dir);
  if (ret < 0 || (int) freq != __scap_cap_freq)
    goto err_exit;

  //======== calculate & set period size (number of frames)

  period_size = __scap_buf->incr / __scap_num_chan;
  if (__scap_dat_size == SCAP_WORD)
    period_size /= 2;

  ret = snd_pcm_hw_params_set_period_size(dev, hw_param, period_size, 0);
  if (ret < 0)
    goto err_exit;

  periods = 2;
  ret = snd_pcm_hw_params_set_periods(dev, hw_param, periods, 0);
  if (ret < 0)
    goto err_exit;

  //======== calculate & set buffer size (number of frames)

  buf_size = period_size * periods;

  ret = snd_pcm_hw_params_set_buffer_size(dev, hw_param, buf_size);
  if (ret < 0)
    goto err_exit;

  //======== send parameters to the device

  ret = snd_pcm_hw_params(dev, hw_param);
  if (ret < 0)
    goto err_exit;

  /*******************************/
  /* Set the software parameters */
  /*******************************/

  //======== allocate & initialize parameter structure

  ret = snd_pcm_sw_params_malloc(&sw_param);
  if (ret < 0)
    goto err_exit;

  ret = snd_pcm_sw_params_current(dev, sw_param);
  if (ret < 0)
    goto err_exit;

  //======== set minimum sample number for notification

  ret = snd_pcm_sw_params_set_avail_min(dev, sw_param, period_size);
  if (ret < 0)
    goto err_exit;

  //======== send parameters to the device

  ret = snd_pcm_sw_params(dev, sw_param);
  if (ret < 0)
    goto err_exit;

  /*********************************/
  /* Prepare the device & clean up */
  /*********************************/

  ret = snd_async_add_pcm_handler(&cb_handle, dev, cb_func, NULL);
  if (ret < 0)
    goto err_exit;

  ret = snd_pcm_prepare(dev);
  if (ret < 0)
    goto err_exit;

  snd_pcm_hw_params_free(hw_param);  
  snd_pcm_sw_params_free(sw_param);  

  RETURN(1)

err_exit:

  if (hw_param != NULL)
    snd_pcm_hw_params_free(hw_param);  

  if (sw_param != NULL)
    snd_pcm_sw_params_free(sw_param);  

  if (dev != NULL)
    snd_pcm_close(dev);

  ERR(CAPERR_DEV_OPEN)
}

//============================================================================

void __scap_close_device(void)
{
  processing = 0;
  snd_async_del_handler(cb_handle);
  snd_pcm_close(dev);
}

//============================================================================

int __scap_start(void)
{
  if (processing)
    return 0;

  snd_pcm_start(dev);
  processing = 1;  
  return 1;
}

//============================================================================

int __scap_stop(void)
{
  if (! processing)
    return 0;

  processing = 0;  
  snd_pcm_drop(dev);
  return 1;
}
