/*
   captest.c  --  sound capture test
   
   SCAP Voice Capture Library
   
   Copyright (C) 2007 Laszlo Menczel
   
   This is free software with NO WARRANTY. Distributed under
   the GNU Library General Public Licence (LGPL) version 2.1.
*/

#if defined _WIN32
  #include <windows.h>
#elif defined _LINUX
  #include <sys/time.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "sndcap.h"

#define SOUND_FREQ	8000
#define SAMPLE_COUNT	(SOUND_FREQ / 10)
#define FRAME_COUNT	20
#define FRAME_SIZE	(2 * SAMPLE_COUNT)
#define BUF_SIZE	(FRAME_COUNT * FRAME_SIZE)
#define CAP_TIMEOUT	(100 * FRAME_COUNT + 500)

static ringbuf_t *buf = NULL;
static char *save_name = "captest.wav";
static dword cap_start = 0;

#if defined _WIN32  /**********************************************************/

static void time_reset_period(void)
{
  timeEndPeriod(1);
}

//=============================================================================

static void time_init(void)
{
  if (timeBeginPeriod(1) == TIMERR_NOERROR)
    atexit(time_reset_period);
}

//=============================================================================

static dword get_system_time(void)
{
  return timeGetTime();
}

#elif defined _LINUX  /*******************************************************/

static dword get_system_time(void)
{
  struct timeval time_val;
  double curr_time;

  gettimeofday(&time_val, NULL);
  curr_time = 1000.0 * (double) time_val.tv_sec + (double) time_val.tv_usec / 1000.0;
  return (dword) curr_time;
}

#endif  /*********************************************************************/

static void pause(void)
{
  printf("Press ENTER...\n");
  getchar();
}

//=============================================================================

int main(int argc, char *argv[])
{
  wavinfo_t info;

  printf("SNDCAP capture test\n");

  buf = rbuf_create(FRAME_COUNT, FRAME_SIZE);

  if (buf == NULL)
  {
    printf("Error: %s.\n", scap_errmsg(scap_last_error()));
    pause();
    return 1;
  }
  
  if (! scap_init(SCAP_MONO, SOUND_FREQ, SCAP_WORD, buf))
  {
    printf("Error: %s.\n", scap_errmsg(scap_last_error()));
    pause();
    return 1;
  }

#if defined _WIN32
  time_init();
#endif

  printf("Press ENTER to record for 2 seconds...\n");
  getchar();

  cap_start = get_system_time();
  scap_start();
  printf("capturing...\n");

  while (buf->count < FRAME_COUNT)
  {
    Sleep(20);
    putchar('.');
    if (get_system_time() - cap_start >= CAP_TIMEOUT)
      break;
  }

  scap_stop();
  printf("\n\n");
  
  if (buf->count == 0)
  {
    printf("Error: no captured samples to save.\n");
    goto no_save;
  }

  printf("Captured %d segments.\n", buf->count);
  printf("Saving sound data to %s.\n", save_name);

  strcpy(info.name, save_name);
  info.numchan = SCAP_MONO;
  info.freq = SOUND_FREQ;
  info.numbits = SCAP_WORD;
  info.frames = FRAME_COUNT * SAMPLE_COUNT;

  if (! wav_save((void *) buf->data, &info))
    printf("Error: %s.\n", scap_errmsg(scap_last_error()));
  else
    printf("OK\n");

no_save:

  rbuf_destroy(buf);
  scap_exit();
  pause();
  return 0;  
}
