// 
// FILE:
// serial.h
//
// FUNCTION:
// The linSerial class provides a simple, basic framework
// for managing a serial port.  The design of this class 
// is focused on making serial port easy to use.  The 
// methods are simple and take simple arguments, hiding
// the complexity of the full Linux/Unix termios interface. 
// Only a small portion of the full termios function is 
// exposed; to get at additional function, study the 
// termios(3), stty(1), setserial(8), and statserial(1)
// manual pages.
//
// One convenience offered by this class is fairly verbose
// error message reporting.  Although most methods return
// a non-zero value to indicate that an error has occured,
// one can instead have error, warning and informational
// messages printed to stdout.  See the ErrLog class for 
// details on enabling, disabling, and changing the amount
// of error reporting.
//
// METHODS:
// The OpenDevice() method accepts a file path name and opens
//    it for reading and writing.  The path name can denote 
//    an ordinary file, or a serial port device.  Of course,
//    most of this class cannot be used on ordinary files;
//    ordinary files are supported only for convenience.
//
//    If the pathname points to a serial device, that device
//    is initialized as described in the SetupDefault() method,
//    below.  After the device has been opened, the standard
//    C Language routines fread(), fprintf(), etc. and/or the
//    standard Unix routines read(), write(), select(), etc.
//    can be used.  Use the GetFileDescriptor() and/or the
//    GetFileHandle() methods to get the appropriate handles.
//
//    This method retrun zero if successful, else a non-zero
//    errno value.
//
// The GetFileDescriptor() method returns a file handle which
//    can be used with the Unix read(), write() etc. routines.
//
// The GetFileHandle() method returns a file handle which can 
//    be used with the C Language fread(), fprintf() routines.
//
// The Close() method will close the device.  If the port
//    is attached to a modem, the modem will be hung up when 
//    it is closed.
//
// The SetBits() method can be used to set the number of bits,
//    the parity and the number of stop bits for the port.
//    Bits can be set to 5,6,7 or 8; parity can be set to E,O 
//    or N for Even, Odd, or None, and the stop bits can be set
//    one or two.  Common configurations are 8N1 and 7E2.
//
// The Set8N1() and Set7E2() methods set common bit and parity 
//    configurations.  See the SetBits method for details.
//
// The SetHardwareFlowControl() sets the port to use hardware 
//    flow control. Most modern modems use the RTS/CTS wires
//    of the serial port to avoid buffer overflows.  The 
//    default setting for this class is to use hardware flow 
//    control.
//
// The SetSoftwareFlowControl() sets the port to use software
//    flow control.  Older modems use this, as well as many 
//    serial priinters and mice.  Software flow control sends
//    the ctrl-S and ctrl-Q to stop and start data flow.
//
// The SetSpeed() method sets the speed of the serial port.
//    It takes as an argument an integer denoting the Baud 
//    rate.  For most modern modems, e.g. 14.4, 28.8, etc.
//    set the speed to 115200; the flow control mechanism 
//    will automatically limit the port speed to what the 
//    modem actually connects at.
//
// The SetDTR() method will raise the DTR line.
// The ClearDTR() method will drop the DTR line.
//
// The SetupDefault() method will restore the port setting 
//    to what it was when the OpenDevice() method was called.
//    This defualt setting is as follows:
//    HUPCL  -- the port/modem will hang up when the Close() 
//              method is called.  
//    CTSRTS -- hardware flow control will be used.
//    8N1    -- 8 bits, no parity, one stop bit.
//    IGNBRK -- breaks will be ignored.
//    CBAUD  -- port speed set to 115200
//
// The SetupRaw() method will put the port setting into a 
//    fairly "raw" mode: it will disable special control 
//    characters and canonical control-character processing.
//    That is, there will be no special handling for the 
//    erase, suspend, tab, end-of-line, form-feed, etc. 
//    characters.  
//
//    This method will NOT change the port speed, the 
//    parity and number of stop bits, etc.
//    
// The SetupTerminal() method will put the port setting into a 
//    terminal-like mode: it will enable the canonical
//    processing of the common control characters:
//    erase, suspend, tab, end-of-line, form-feed, etc. 
//
//    This method will NOT change the port speed, the 
//    parity and number of stop bits, etc.
//
// HISTORY:
// Copyright (c) 1995, 1997 Linas Vepstas linas@linas.org
// Released under the  conditions  of  the  GNU General 
// Public License.

#ifndef __LIN_SERIAL_H__
#define __LIN_SERIAL_H__

#include <stdio.h>

#include "errlog.h"

class linSerial :
   protected linErrLog
{
   public:
      linSerial (void);

      int OpenDevice  (const char *);
      int GetFileDescriptor (void) { return fd; }
      FILE * GetFileHandle (void) { return serialport; }
      void Close (void);

      int SetupDefault (void);
      int SetupRaw (void);
      int SetupTerminal (void);

      int SetSpeed (int);

      int SetHardwareFlowControl (void) { return SetFlowControl (1); }
      int SetSoftwareFlowControl (void) { return SetFlowControl (0); }

      int SetBits (int, char, int);
      int Set8N1 (void) { return SetBits (8, 'N', 1); }
      int Set7E2 (void) { return SetBits (7, 'E', 2); }

      void SetDTR (void);
      void ClearDTR (void);

   protected:
      int SetFlowControl (int);

      int fd;                   // file descriptor
      FILE * serialport;
      char * filename;

   private:
      int speed;
      short istty;
};

#endif /* __LIN_SERIAL_H__ */
// ====================== END OF FILE ============================
