//
// FILE: errlog.h
//
// FUNCTION:
// The linErrLog class provides some simple error logging
// facilities.  It should be realtively easily extensible 
// to provide more extensive services.
//
// USAGE:
// Classes wishing to use this error reporting & debugging
// framework should inherit from this class.  Six levels
// of logging are provided: Fatal, Severe, Warning, 
// Informational, Debug and Trace.  The logging level 
// can be changed dynamically at runtime. Logging can be
// enabled and disabled at runtime.  If desired, logging 
// can be compiled out of the application.
//
// METHODS:
// The SetLogLevel() method sets the error logging severity 
//    level
//
// The GetLogLevel() method returns the currnt error 
//    logging severity level.
//
// The Enable() method enables logging.
// The Disable() method disables logging.
//
// The SetScope() method should be used to declare the 
//    current scope.  Typically, it should be called 
//    at the begining of a soubroutine, method or large
//    block and should name that block.  The scope is 
//    printed with the error message.
//
// The PrtErr() method prints a message to the log file.
//    The message is prefixed with a printed severity
//    level, and the current scope.  The first argument
//    to this method indicates the severity level of the
//    message; it is used to avoid the printing of 
//    low-severity messages.  The second and further 
//    arguments are identical to the arguments of the 
//    printf() function.
//
//    This method should not be used directly; instead,
//    the LOGERR macro should be used instead.
//
// The PrtMore() method prints a message to the log file.
//    The first argument to this method indicates the 
//    severity level of the message; it is used to avoid 
//    the printing of low-severity messages.  The second 
//    and further arguments are identical to the arguments 
//    of the printf() function.
//
// SUGGESTED ENHANCEMENTS:
// -- Provide Internationalization framework.
// -- Allow a logfile or pipe to be specified.
// -- Allow printing to syslogd.
// -- Provide locking mechanism to enable the use
//    in a multi-porcess environment.
//
// HISTORY:
// Copyright (c) 1995, 1997 Linas Vepstas linas@linas.org
// Released under the  conditions  of  the  GNU General 
// Public License.

#ifndef __LIN_ERR_LOG_H__
#define __LIN_ERR_LOG_H__

#include <stdio.h>

class linErrLog {
   protected:
      enum severity {
         FATAL,
         SEVERE,
         WARN,
         INFO,
         TRACE,
         DEBUG
      };
   public:
      linErrLog (void);
      ~linErrLog ();

      void Enable (void) { enabled = 1; };
      void Disable (void) {enabled = 0; };
      void SetLogLevel (severity);
      severity GetLogLevel (void);

      void SetScope (const char * scp) { scope = scp; }

      void PrtErr (severity, char * fmt, ...); 
      void PrtMore (severity, char * fmt, ...); 

   protected:
      short enabled;        // error printeing enabled?
      severity loglevel;    // logging level
      const char * scope;   // name of scope
      FILE * errf;          // output file

};

#define LIN_COMPILE_IN_DIAGNOSTICS
#ifdef LIN_COMPILE_IN_DIAGNOSTICS
#define LOGERR(x) PrtErr x
#define LOGMORE(x) PrtMore x
#define SETSCOPE(x) SetScope(x)
#else
#define LOGERR(x) 
#define LOGMORE(x) 
#define SETSCOPE(x) 
#endif /* LIN_COMPILE_IN_DIAGNOSTICS */

#endif /* __LIN_ERR_LOG_H__ */

// ================== END OF FILE =============================
