/* tests/t-message.c -- test libretto's message functions
 *
 * Aaron Crane <aaronc@pobox.com>
 * 28 August 1997
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libretto/libretto.h>
#include <assert.h>

enum format_type
{ TYPE_VOID, TYPE_INT, TYPE_CHAR_P };

union object
{
    int v_int;
    const char *v_char_p;
};

struct testinfo
{
    const char *format;
    enum format_type type;
    union object object;
    const char *expected;
};

/* This hardly gives the printf code a real workout, but I'm fairly
 * confident my printf stuff isn't going to blow up.  (I've been through it
 * by hand and with gdb a number of times, and it *does* work on these
 * simple cases.)  However, it might be a nice project for someone to write
 * some *real* test code. */
static const struct testinfo tests[] =
{
    { "%V", TYPE_VOID, { 0 }, "t-message" },
    { "%12V", TYPE_VOID, { 0 }, "   t-message" },
    { "%d", TYPE_INT, { 12 }, "12" },
    { "%4d", TYPE_INT, { 12 }, "  12" },
    { "%04d", TYPE_INT, { 12 }, "0012" },
    { "%-4d", TYPE_INT, { 12 }, "12  " },
#ifdef __GNUC__
    { "%s", TYPE_CHAR_P, { v_char_p: "abcdef" }, "abcdef" },
    { "%8s", TYPE_CHAR_P, { v_char_p: "  abcdef" }, "  abcdef" },
    { "%-8s", TYPE_CHAR_P, { v_char_p: "abcdef" }, "abcdef  " },
#endif
    { 0, TYPE_VOID, { 0 }, 0 },
};

int
main (int argc, char **argv)
{
    const struct testinfo *test;
    char *buf = 0;
    size_t n = 0;

    (void) argc;
    msg_set_invocation_name (argv[0]);

#ifndef __GNUC__
    msg_write ("warning: message functions aren't fully tested unless you "
	       "use the GNU C compiler\n");
#endif

    for (test = tests;  test->format;  test++)
	switch (test->type)
	{
	case TYPE_VOID:
	    asn_printf (&buf, &n, test->format);
	    assert (strcmp (buf, test->expected) == 0);
	    break;
	case TYPE_INT:
	    asn_printf (&buf, &n, test->format, test->object.v_int);
	    assert (strcmp (buf, test->expected) == 0);
	    break;
	case TYPE_CHAR_P:
	    asn_printf (&buf, &n, test->format, test->object.v_char_p);
	    assert (strcmp (buf, test->expected) == 0);
	    break;
	default:
	    msg_fatal ("BUG: unknown type value\n");
	}

    return 0;
}
