/* tests/t-file.c -- test libretto's file reading functions
 *
 * Aaron Crane <aaronc@pobox.com>
 * 1 January 1998
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libretto/libretto.h>

#include <config.h>

#include <string.h>
#include <assert.h>

#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif

/* A simple function which merely waits for a single process PID to finish */
static void wait_for_process (pid_t pid);

typedef struct expected
{
    char *out;
    ssize_t out_len;
    char *in;
} expected_t;

static int do_test (const expected_t *exp);

static const expected_t test_tab[] =
{
    { "foo bar\n", 8, "foo bar" },
    { "foo\0 bar\n", 9, "foo bar" },
    { "0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef"
      "0123456789abcdef\n", 81,
      "0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef"
      "0123456789abcdef" },
    { "0123456789abcdef\0" "0123456789abcdef\0" "0123456789abcdef\0" "0123456789abcdef\0"
      "0123456789abcdef\0\n", 86,
      "0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef"
      "0123456789abcdef" },
    { "foo bar", 7, "foo bar" },
    { 0, -1, 0 }
};

int
main (int argc, char **argv)
{
    const expected_t *t;

    (void) argc;
    msg_set_invocation_name (argv[0]);

    for (t = test_tab;  t->out;  t++)
	assert (do_test (t) == 0);

    return 0;
}

static int
do_test (const expected_t *exp)
{
    int fd[2];
    pid_t pid;

    if (pipe (fd) == -1)
	msg_fatal ("couldn't create pipe: %m\n");

    pid = fork ();
    if (pid == -1)		/* failure */
	msg_fatal ("couldn't fork: %m\n");
    else if (pid == 0)		/* child process */
    {
	close (fd[0]);
	write (fd[1], exp->out, exp->out_len);
	close (fd[1]);
	exit (0);
    }
    else			/* parent */
    {
	FILE *f;
	size_t buflen = 0;
	ssize_t len, desired_len;
	char *buf = 0;

	f = fdopen (fd[0], "r");
	close (fd[1]);
	len = file_getline (f, &buf, &buflen);
	desired_len = strlen (exp->in);
	assert (len == desired_len);
	assert (strcmp (buf, exp->in) == 0);
	len = file_getline (f, &buf, &buflen);
	assert (len == EOF);
	wait_for_process (pid);
    }

    return 0;
}

#ifdef HAVE_WAITPID
static void
wait_for_process (pid_t pid)
{
    /* There may be no prototype in scope. */
    waitpid (pid, (int *) 0, 0);
}
#elif defined HAVE_WAIT
static void
wait_for_process (pid_t pid)
{
    pid_t i;

    while ((i = wait ((int *) 0)) != pid)
	if (i == -1)
	    break;
}
#else
#error Sorry, I need either wait() or waitpid() to test file functions.
#endif
