/* tests/t-darray.c -- test libretto's darray functions
 *
 * Aaron Crane <aaronc@pobox.com>
 * 6 November 1997
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libretto/libretto.h>
#include <libretto/darray.h>
#include <assert.h>

typedef int (*testable_f) (void);

static int t_create (void);
static int t_append (void);
static int t_set (void);
static int t_last (void);
static int t_exchange (void);
static int t_resize (void);
static int t_insert (void);
static int t_delete (void);
static int t_concat (void);
static int t_insert_array (void);
static int t_prune (void);
static int t_find (void);
static int t_rfind (void);
static int t_bsearch (void);
static int t_sort (void);
static int t_push (void);
static int t_pop (void);
static int t_drop (void);
static int t_top (void);
static int t_swap (void);
static int t_dup (void);

static const testable_f test_tab[] =
{
    &t_create,
    &t_append,
    &t_set,
    &t_last,
    &t_exchange,
    &t_resize,
    &t_insert,
    &t_delete,
    &t_concat,
    &t_insert_array,
    &t_prune,
    &t_find,
    &t_rfind,
    &t_bsearch,
    &t_sort,
    &t_push,
    &t_pop,
    &t_drop,
    &t_top,
    &t_swap,
    &t_dup
};

static int char_compare (const void *left, const void *right, void *cmp_args);
static int char_prune (void *obj, void *prune_args);

int
main (int argc, char **argv)
{
    size_t i;

    (void) argc;
    msg_set_invocation_name (argv[0]);

    for (i = 0;  i < sizeof (test_tab) / sizeof (test_tab[0]);  i++)
	assert (test_tab[i] () == 0);

    return 0;
}


/*
 * Functions to handle darrays of characters
 */

static int
char_compare (const void *left, const void *right, void *cmp_args)
{
    const char *a = left, *b = right;

    (void) cmp_args;

    return *a - *b;
}

static int
char_prune (void *obj, void *prune_args)
{
    char *p = obj;

    (void) prune_args;

    return *p > 'c';
}


static int
t_create (void)
{
    Darray *da;

    da = da_create (1);
    assert (da_valid_p (da));
    assert (da_length (da) == 0);
    da_destroy (da);
    return 0;
}

static int
t_append (void)
{
    char c = 'a';
    char *p;
    Darray *da = da_create (1);

    da_append (da, &c);
    assert (da_valid_p (da));
    assert (da_length (da) == 1);
    p = da_ref (da, 0);
    assert (p != NULL);
    assert (*p == 'a');
    da_destroy (da);

    return 0;
}

static int
t_set (void)
{
    char a = 'a', b = 'b';
    Darray *da = da_create (1);
    char *p;

    da_append (da, &a);
    da_set (da, 0, &b);
    assert (da_valid_p (da));
    assert (da_length (da) == 1);
    p = da_ref (da, 0);
    assert (*p == 'b');
    da_destroy (da);

    return 0;
}

static int
t_last (void)
{
    char a = 'a', b = 'b';
    Darray *da = da_create (1);
    char *p;

    da_append (da, &a);
    da_append (da, &b);
    p = da_last (da);
    assert (*p == 'b');
    da_destroy (da);

    return 0;
}

static int
t_exchange (void)
{
    char a = 'a', b = 'b';
    Darray *da = da_create (1);
    char *pa, *pb;

    da_append (da, &a);
    da_append (da, &b);
    da_exchange (da, 0, 1);
    assert (da_valid_p (da));
    assert (da_length (da) == 2);
    pb = da_ref (da, 0);
    pa = da_ref (da, 1);
    assert (*pa == 'a');
    assert (*pb == 'b');
    da_destroy (da);

    return 0;
}

static int
t_resize (void)
{
    Darray *da = da_create (1);

    da_resize (da, 10);
    assert (da_valid_p (da));
    assert (da_length (da) == 10);

    da_resize (da, 10);
    assert (da_valid_p (da));
    assert (da_length (da) == 10);

    da_resize (da, 5);
    assert (da_valid_p (da));
    assert (da_length (da) == 5);

    da_destroy (da);
    return 0;
}

static int
t_insert (void)
{
    Darray *da = da_create (1);
    char a = 'a', b = 'b', c = 'c';
    char *p;

    da_insert (da, 0, &a);
    assert (da_valid_p (da));
    assert (da_length (da) == 1);
    p = da_ref (da, 0);
    assert (*p == 'a');

    da_insert (da, 1, &b);
    assert (da_valid_p (da));
    assert (da_length (da) == 2);
    p = da_ref (da, 0);
    assert (*p == 'a');
    p = da_ref (da, 1);
    assert (*p == 'b');

    da_insert (da, 1, &c);
    assert (da_valid_p (da));
    assert (da_length (da) == 3);
    p = da_ref (da, 0);
    assert (*p == 'a');
    p = da_ref (da, 1);
    assert (*p == 'c');
    p = da_ref (da, 2);
    assert (*p == 'b');

    da_destroy (da);
    return 0;
}

static int
t_delete (void)
{
    Darray *da = da_create (1);
    char a = 'a', b = 'b', c = 'c';
    char *p;

    da_append (da, &a);
    da_append (da, &b);
    da_append (da, &c);

    da_delete (da, 1);
    assert (da_valid_p (da));
    assert (da_length (da) == 2);
    p = da_ref (da, 0);
    assert (*p == 'a');
    p = da_ref (da, 1);
    assert (*p == 'c');
    da_delete (da, 1);
    assert (da_valid_p (da));
    assert (da_length (da) == 1);
    p = da_ref (da, 0);
    assert (*p == 'a');

    da_destroy (da);
    return 0;
}

static int
t_concat (void)
{
    Darray *da[2];
    char a = 'a', b = 'b', c = 'c', d = 'd';
    char *p;

    da[0] = da_create (1);
    da[1] = da_create (1);

    da_append (da[0], &a);
    da_append (da[0], &b);
    da_append (da[1], &c);
    da_append (da[1], &d);

    da_concat_da (da[0], da[1]);
    assert (da_valid_p (da[0]));
    assert (da_length (da[0]) == 4);
    p = da_ref (da[0], 0);
    assert (*p == 'a');
    p = da_ref (da[0], 1);
    assert (*p == 'b');
    p = da_ref (da[0], 2);
    assert (*p == 'c');
    p = da_ref (da[0], 3);
    assert (*p == 'd');

    da_destroy (da[0]);
    da_destroy (da[1]);
    return 0;
}

static int
t_insert_array (void)
{
    Darray *da[2];
    char a = 'a', b = 'b', c = 'c', d = 'd';
    char *p;

    da[0] = da_create (1);
    da[1] = da_create (1);

    da_append (da[0], &a);
    da_append (da[0], &b);
    da_append (da[1], &c);
    da_append (da[1], &d);

    da_insert_da (da[0], da[1], 2);
    assert (da_valid_p (da[0]));
    assert (da_length (da[0]) == 4);
    p = da_ref (da[0], 0);
    assert (*p == 'a');
    p = da_ref (da[0], 1);
    assert (*p == 'b');
    p = da_ref (da[0], 2);
    assert (*p == 'c');
    p = da_ref (da[0], 3);
    assert (*p == 'd');

    da_destroy (da[0]);
    da_destroy (da[1]);

    da[0] = da_create (1);
    da[1] = da_create (1);

    da_append (da[0], &a);
    da_append (da[0], &b);
    da_append (da[1], &c);
    da_append (da[1], &d);

    da_insert_da (da[0], da[1], 1);
    assert (da_valid_p (da[0]));
    assert (da_length (da[0]) == 4);
    p = da_ref (da[0], 0);
    assert (*p == 'a');
    p = da_ref (da[0], 1);
    assert (*p == 'c');
    p = da_ref (da[0], 2);
    assert (*p == 'd');
    p = da_ref (da[0], 3);
    assert (*p == 'b');

    da_destroy (da[0]);
    da_destroy (da[1]);

    da[0] = da_create (1);
    da[1] = da_create (1);

    da_append (da[0], &a);
    da_append (da[0], &b);
    da_append (da[1], &c);
    da_append (da[1], &d);

    da_insert_da (da[0], da[1], 0);
    assert (da_valid_p (da[0]));
    assert (da_length (da[0]) == 4);
    p = da_ref (da[0], 0);
    assert (*p == 'c');
    p = da_ref (da[0], 1);
    assert (*p == 'd');
    p = da_ref (da[0], 2);
    assert (*p == 'a');
    p = da_ref (da[0], 3);
    assert (*p == 'b');

    da_destroy (da[0]);
    da_destroy (da[1]);

    return 0;
}

static int
t_prune (void)
{
    Darray *da = da_create (1);
    char a = 'a', b = 'b', c = 'c', d = 'd', e = 'e';
    char *p;

    da_append (da, &a);
    da_append (da, &c);
    da_append (da, &d);
    da_append (da, &b);
    da_append (da, &e);

    da_prune (da, char_prune, 0);
    assert (da_valid_p (da));
    assert (da_length (da) == 3);
    p = da_ref (da, 0);
    assert (*p == 'a');
    p = da_ref (da, 1);
    assert (*p == 'c');
    p = da_ref (da, 2);
    assert (*p == 'b');

    da_destroy (da);

    return 0;
}

static int
t_find (void)
{
    Darray *da = da_create (1);
    char a = 'a', b = 'b';
    ssize_t position[] = { 1, 2, 4, -1 };
    ssize_t i, pos;
    int j = 0;

    da_append (da, &a);
    da_append (da, &b);
    da_append (da, &b);
    da_append (da, &a);
    da_append (da, &b);
    da_append (da, &a);

    i = -1;
    do
    {
	pos = da_find (da, i, &b, char_compare, 0);
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    da_destroy (da);

    return 0;
}

static int
t_rfind (void)
{
    Darray *da = da_create (1);
    char a = 'a', b = 'b';
    ssize_t position[] = { 4, 2, 1, -1 };
    ssize_t i, pos;
    int j = 0;

    da_append (da, &a);
    da_append (da, &b);
    da_append (da, &b);
    da_append (da, &a);
    da_append (da, &b);
    da_append (da, &a);

    i = -1;
    do
    {
	pos = da_rfind (da, i, &b, char_compare, 0);
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    da_destroy (da);

    return 0;
}

static int
t_bsearch (void)
{
    Darray *da = da_create (1);
    char *s = "abcdefghi", *p, k = 'k';
    ssize_t i, pos;

    for (p = s;  *p;  p++)
	da_append (da, p);

    for (p = s;  *p;  p++)
    {
	i = p - s;
	pos = da_bsearch (da, p, char_compare, 0);
	assert (pos == i);
    }
    pos = da_bsearch (da, &k, char_compare, 0);
    assert (pos == -1);

    da_destroy (da);

    return 0;
}

static int
t_sort (void)
{
    Darray *sorted = da_create (1);
    char *sorted_s = "abcd";
    char *t[] = { "abcd", "dcba", "bdac", 0 };
    char **s, *p;

    for (p = sorted_s;  *p;  p++)
	da_append (sorted, p);

    for (s = t;  *s;  s++)
    {
	Darray *da = da_create (1);

	for (p = *s;  *p;  p++)
	    da_append (da, p);
	da_sort (da, char_compare, 0);
	assert (da_equal (sorted, da, char_compare, 0));

	da_destroy (da);
    }

    da_destroy (sorted);

    return 0;
}

static int
t_push (void)
{
    char c = 'a';
    char *p;
    Darray *da = da_create (1);

    da_push (da, &c);
    assert (da_valid_p (da));
    assert (da_length (da) == 1);
    p = da_ref (da, 0);
    assert (p != NULL);
    assert (*p == 'a');
    da_destroy (da);

    return 0;
}

static int
t_pop (void)
{
    char c = 'a', buf;
    Darray *da = da_create (1);

    da_push (da, &c);
    da_pop (da, &buf);
    assert (da_valid_p (da));
    assert (da_length (da) == 0);
    assert (c == buf);
    da_destroy (da);

    return 0;
}

static int
t_drop (void)
{
    char c = 'a';
    Darray *da = da_create (1);

    da_push (da, &c);
    da_drop (da);
    assert (da_valid_p (da));
    assert (da_length (da) == 0);
    da_destroy (da);

    return 0;
}

static int
t_top (void)
{
    char a = 'a', b = 'b';
    Darray *da = da_create (1);
    char *p;

    da_push (da, &a);
    da_push (da, &b);
    p = da_top (da);
    assert (*p == 'b');
    da_destroy (da);

    return 0;
}

static int
t_swap (void)
{
    char a = 'a', b = 'b';
    Darray *da = da_create (1);
    char *p;

    da_push (da, &a);
    da_push (da, &b);
    da_swap (da);
    assert (da_valid_p (da));
    assert (da_length (da) == 2);
    p = da_top (da);
    assert (*p == 'a');
    da_drop (da);
    p = da_top (da);
    assert (*p == 'b');

    da_destroy (da);

    return 0;
}

static int
t_dup (void)
{
    char a = 'a';
    Darray *da = da_create (1);
    char *p;

    da_push (da, &a);
    da_dup (da);
    assert (da_valid_p (da));
    assert (da_length (da) == 2);
    p = da_top (da);
    assert (*p == 'a');
    da_drop (da);
    p = da_top (da);
    assert (*p == 'a');

    da_destroy (da);

    return 0;
}
