/* tests/chain.c -- test libretto's chain functions
 *
 * Aaron Crane <aaronc@pobox.com>
 * 30 December 1997
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libretto/libretto.h>
#include <libretto/chain.h>
#include <libretto/autostr.h>
#include <string.h>
#include <assert.h>

typedef int (*testable_f) (void);

static int t_create (void);
static int t_nth (void);
static int t_walk (void);
static int t_insert (void);
static int t_push (void);
static int t_drop (void);
static int t_pop (void);
static int t_enqueue (void);
static int t_dequeue (void);
static int t_append (void);
static int t_front_insert (void);
static int t_back_insert (void);
static int t_front_remove (void);
static int t_back_remove (void);
static int t_find (void);
static int t_member (void);
static int t_prune (void);
static int t_concat (void);
static int t_reverse (void);

static const testable_f test_tab[] =
{
    &t_create,
    &t_push,
    &t_drop,
    &t_pop,
    &t_enqueue,
    &t_dequeue,
    &t_append,
    &t_front_insert,
    &t_back_insert,
    &t_front_remove,
    &t_back_remove,
    &t_walk,
    &t_insert,
    &t_nth,
    &t_find,
    &t_member,
    &t_prune,
    &t_concat,
    &t_reverse
};

int
main (int argc, char **argv)
{
    size_t i;

    (void) argc;
    msg_set_invocation_name (argv[0]);

    for (i = 0;  i < sizeof (test_tab) / sizeof (test_tab[0]);  i++)
	assert (test_tab[i] () == 0);

    return 0;
}

static int autostr_compare (const void *left, const void *right, void *cmp_args);
static int autostr_walk (void *data, void *walk_args);
static int autostr_walk_destroy (void *data, void *walk_args);
static int autostr_prune (void *obj, void *prune_args);

static int
autostr_compare (const void *left, const void *right, void *cmp_args)
{
    (void) cmp_args;
    return astr_cmp (left, right);
}

static int
autostr_walk (void *data, void *walk_args)
{
    (void) walk_args;
    astr_capitalise (data);
    return 0;
}

static int
autostr_walk_destroy (void *data, void *walk_args)
{
    (void) walk_args;
    astr_destroy (data);
    return 0;
}

static int
autostr_prune (void *obj, void *prune_args)
{
    if (astr_equal (obj, prune_args))
    {
	astr_destroy (obj);
	return 1;
    }
    return 0;
}


static int
t_create (void)
{
    Chain *chain;

    chain = chain_create ();
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 0);
    chain_destroy (chain);
    return 0;
}

static int
t_push (void)
{
    Chain *chain = chain_create ();

    chain_push (chain, "abc");
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 1);
    assert (strcmp (chainlink_data (chain_top (chain)), "abc") == 0);
    chain_destroy (chain);
    return 0;
}

static int
t_drop (void)
{
    Chain *chain = chain_create ();

    chain_push (chain, "abc");
    chain_drop (chain);
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 0);

    return 0;
}

static int
t_pop (void)
{
    Chain *chain = chain_create ();
    void *p;

    chain_push (chain, "abc");
    p = chain_pop (chain);
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 0);
    assert (strcmp (p, "abc") == 0);

    return 0;
}

static int
t_walk (void)
{
    Chain *chain = chain_create ();
    Autostr *str;

    str = astr_create_s ("abc");
    chain_push (chain, str);
    str = astr_create_s ("def");
    chain_push (chain, str);
    str = astr_create_s ("ghi");
    chain_push (chain, str);

    chain_walk (chain, autostr_walk, 0);

    str = chain_pop (chain);
    assert (strcmp (astr_chars (str), "GHI"));
    astr_destroy (str);

    str = chain_pop (chain);
    assert (strcmp (astr_chars (str), "DEF"));
    astr_destroy (str);

    str = chain_pop (chain);
    assert (strcmp (astr_chars (str), "ABC"));
    astr_destroy (str);

    chain_destroy (chain);

    return 0;
}

static int
t_insert (void)
{
    Chain *chain = chain_create ();
    Chainlink *node;
    Autostr *str;

    str = astr_create_s ("def");
    chain_insert (chain, str, autostr_compare, 0);
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 1);

    str = astr_create_s ("ghi");
    chain_insert (chain, str, autostr_compare, 0);
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 2);

    str = astr_create_s ("abc");
    chain_insert (chain, str, autostr_compare, 0);
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 3);

    node = chain_first (chain);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "abc") == 0);

    node = chain_succ (chain, node);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "def") == 0);

    node = chain_succ (chain, node);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "ghi") == 0);

    node = chain_succ (chain, node);
    assert (node == 0);

    chain_walk (chain, autostr_walk_destroy, 0);

    chain_destroy (chain);

    return 0;
}

static int
t_enqueue (void)
{
    Chain *chain = chain_create ();

    chain_enqueue (chain, "abc");
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 1);
    assert (strcmp (chainlink_data (chain_first (chain)), "abc") == 0);
    chain_destroy (chain);
    return 0;
}

static int
t_dequeue (void)
{
    Chain *chain = chain_create ();
    void *p;

    chain_enqueue (chain, "abc");
    p = chain_dequeue (chain);
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 0);
    assert (strcmp (p, "abc") == 0);

    return 0;
}

static int
t_append (void)
{
    Chain *chain = chain_create ();
    Chainlink *node;

    chain_append (chain, "abc");
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 1);
    node = chain_first (chain);
    assert (strcmp (chainlink_data (node), "abc") == 0);
    chain_destroy (chain);
    return 0;
}

static int
t_front_insert (void)
{
    Chain *chain = chain_create ();
    Chainlink *node;

    chain_front_insert (chain, "abc");
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 1);
    node = chain_first (chain);
    assert (strcmp (chainlink_data (node), "abc") == 0);
    chain_destroy (chain);
    return 0;
}

static int
t_back_insert (void)
{
    Chain *chain = chain_create ();
    Chainlink *node;

    chain_back_insert (chain, "abc");
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 1);
    node = chain_first (chain);
    assert (strcmp (chainlink_data (node), "abc") == 0);
    chain_destroy (chain);
    return 0;
}

static int
t_front_remove (void)
{
    Chain *chain = chain_create ();
    void *p;

    chain_front_insert (chain, "abc");
    p = chain_front_remove (chain);
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 0);
    assert (strcmp (p, "abc") == 0);

    return 0;
}

static int
t_back_remove (void)
{
    Chain *chain = chain_create ();
    void *p;

    chain_back_insert (chain, "abc");
    p = chain_back_remove (chain);
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 0);
    assert (strcmp (p, "abc") == 0);

    return 0;
}

static int
t_nth (void)
{
    Chain *chain = chain_create ();
    Chainlink *node;
    Autostr *str;

    str = astr_create_s ("abc");
    chain_append (chain, str);
    str = astr_create_s ("def");
    chain_append (chain, str);
    str = astr_create_s ("ghi");
    chain_append (chain, str);

    node = chain_nth (chain, 0);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "abc") == 0);

    node = chain_nth (chain, 1);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "def") == 0);

    node = chain_nth (chain, 2);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "ghi") == 0);

    chain_walk (chain, autostr_walk_destroy, 0);
    chain_destroy (chain);

    return 0;
}

static int
t_find (void)
{
    Chain *chain = chain_create ();
    Chainlink *node;
    Autostr *str, *ns;

    str = astr_create_s ("abc");
    chain_append (chain, str);
    str = astr_create_s ("def");
    chain_append (chain, str);
    str = astr_create_s ("ghi");
    chain_append (chain, str);

    ns = astr_create_s ("abc");
    node = chain_find (chain, 0, ns, autostr_compare, 0);
    assert (node);
    str = chainlink_data (node);
    assert (astr_equal (str, ns));
    astr_destroy (ns);

    ns = astr_create_s ("def");
    node = chain_find (chain, 0, ns, autostr_compare, 0);
    assert (node);
    str = chainlink_data (node);
    assert (astr_equal (str, ns));
    astr_destroy (ns);

    ns = astr_create_s ("ghi");
    node = chain_find (chain, 0, ns, autostr_compare, 0);
    assert (node);
    str = chainlink_data (node);
    assert (astr_equal (str, ns));
    astr_destroy (ns);

    ns = astr_create_s ("jkl");
    node = chain_find (chain, 0, ns, autostr_compare, 0);
    assert (!node);
    astr_destroy (ns);

    chain_walk (chain, autostr_walk_destroy, 0);
    chain_destroy (chain);

    return 0;
}

static int
t_member (void)
{
    Chain *chain = chain_create ();
    Chainlink *node;
    Autostr *str = astr_create_s ("abc");

    chain_append (chain, str);

    node = chain_front (chain);
    assert (chain_member (chain, node));
    node = (Chainlink *) str;
    assert (!chain_member (chain, node));

    chain_walk (chain, autostr_walk_destroy, 0);
    chain_destroy (chain);

    return 0;
}

static int
t_prune (void)
{
    Chain *chain = chain_create ();
    Chainlink *node;
    Autostr *str;

    chain_append (chain, astr_create_s ("abc"));
    chain_append (chain, astr_create_s ("def"));
    chain_append (chain, astr_create_s ("abc"));
    chain_append (chain, astr_create_s ("ghi"));
    chain_append (chain, astr_create_s ("abc"));

    str = astr_create_s ("abc");
    chain_prune (chain, autostr_prune, str);
    astr_destroy (str);
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 2);
    node = chain_first (chain);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "def") == 0);
    node = chain_succ (chain, node);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "ghi") == 0);

    chain_walk (chain, autostr_walk_destroy, 0);
    chain_destroy (chain);

    return 0;
}

static int
t_concat (void)
{
    Chain *chain[2];
    Chainlink *node;
    Autostr *str;

    chain[0] = chain_create ();
    chain[1] = chain_create ();
    chain_append (chain[0], astr_create_s ("abc"));
    chain_append (chain[0], astr_create_s ("def"));
    chain_append (chain[1], astr_create_s ("ghi"));
    chain_append (chain[1], astr_create_s ("jkl"));

    chain_concat_chain (chain[0], chain[1]);

    /* Must destroy chain[1] -- but not its contents, as the pointers were
     * copied into chain[0] */
    chain_destroy (chain[1]);

    assert (chain_valid_p (chain[0]));
    assert (chain_length (chain[0]) == 4);

    node = chain_first (chain[0]);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "abc") == 0);

    node = chain_succ (chain[0], node);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "def") == 0);

    node = chain_succ (chain[0], node);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "ghi") == 0);

    node = chain_succ (chain[0], node);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "jkl") == 0);

    chain_walk (chain[0], autostr_walk_destroy, 0);
    chain_destroy (chain[0]);

    return 0;
}

static int
t_reverse (void)
{
    Chain *chain = chain_create ();
    Chainlink *node;
    Autostr *str;

    chain_append (chain, astr_create_s ("ghi"));
    chain_append (chain, astr_create_s ("def"));
    chain_append (chain, astr_create_s ("abc"));

    chain_reverse (chain);
    assert (chain_valid_p (chain));
    assert (chain_length (chain) == 3);

    node = chain_first (chain);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "abc") == 0);

    node = chain_succ (chain, node);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "def") == 0);

    node = chain_succ (chain, node);
    str = chainlink_data (node);
    assert (strcmp (astr_chars (str), "ghi") == 0);

    chain_walk (chain, autostr_walk_destroy, 0);
    chain_destroy (chain);

    return 0;
}
