/* tests/bstree.c -- test libretto's bstree functions
 *
 * Aaron Crane <aaronc@pobox.com>
 * 1 January 1998
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libretto/libretto.h>
#include <libretto/bstree.h>
#include <libretto/autostr.h>
#include <assert.h>

typedef int (*testable_f) (void);

static int t_create (void);
static int t_simple (void);
static int t_move_forward (void);
static int t_move_backward (void);
static int t_nodes (void);
static int t_depth (void);
static int t_lookup (void);
static int t_delete (void);
static int t_wipe (void);
static int t_prune (void);

static const testable_f test_tab[] =
{
    &t_create,
    &t_simple,
    &t_move_forward,
    &t_move_backward,
    &t_nodes,
    &t_depth,
    &t_lookup,
    &t_delete,
    &t_wipe,
    &t_prune
};

int
main (int argc, char **argv)
{
    size_t i;

    (void) argc;
    msg_set_invocation_name (argv[0]);

    for (i = 0;  i < sizeof (test_tab) / sizeof (test_tab[0]);  i++)
	assert (test_tab[i] () == 0);

    return 0;
}

static int
autostr_compare (void *left, void *right, void *cmp_args)
{
    (void) cmp_args;
    return astr_cmp (left, right);
}

static int
autostr_walk_destroy (void *data, void *walk_args)
{
    (void) walk_args;
    astr_destroy (data);
    return 0;
}

static int
autostr_walk_verify (void *data, void *walk_args)
{
    static Autostr *saved = 0;

    (void) walk_args;
    if (saved && data)
	assert (astr_cmp (saved, data) <= 0);
    saved = data;
    return 0;
}

static int
autostr_prune (void *data, void *prune_args)
{
    if (astr_equal (data, prune_args))
    {
	astr_destroy (data);
	return 1;
    }
    return 0;
}

static int
t_create (void)
{
    Bstree *bst;

    bst = bst_create (autostr_compare);
    bst_destroy (bst);
    return 0;
}

static int
t_simple (void)
{
    Bstree *bst = bst_create (autostr_compare);

    bst_insert (bst, astr_create_s ("abc"), 0);
    bst_insert (bst, astr_create_s ("def"), 0);
    bst_insert (bst, astr_create_s ("ghi"), 0);

    bst_walk (bst, autostr_walk_verify, 0);
    autostr_walk_verify (0, 0);
    bst_walk (bst, autostr_walk_destroy, 0);

    bst_destroy (bst);
    return 0;
}

static int
t_move_forward (void)
{
    Bstree *bst = bst_create (autostr_compare);
    Bstnode *node;
    Autostr *str = 0;

    bst_insert (bst, astr_create_s ("abc"), 0);
    bst_insert (bst, astr_create_s ("def"), 0);
    bst_insert (bst, astr_create_s ("ghi"), 0);

    for (node = bst_smallest (bst);  node;  node = bst_succ (bst, node))
    {
	if (str)
	    assert (astr_cmp (str, bstnode_data (node)) <= 0);
	str = bstnode_data (node);
    }

    bst_walk (bst, autostr_walk_destroy, 0);
    bst_destroy (bst);

    return 0;
}

static int
t_move_backward (void)
{
    Bstree *bst = bst_create (autostr_compare);
    Bstnode *node;
    Autostr *str = 0;

    bst_insert (bst, astr_create_s ("abc"), 0);
    bst_insert (bst, astr_create_s ("def"), 0);
    bst_insert (bst, astr_create_s ("ghi"), 0);

    for (node = bst_largest (bst);  node;  node = bst_pred (bst, node))
    {
	if (str)
	    assert (astr_cmp (str, bstnode_data (node)) >= 0);
	str = bstnode_data (node);
    }

    bst_walk (bst, autostr_walk_destroy, 0);
    bst_destroy (bst);

    return 0;
}

static int
t_nodes (void)
{
    Bstree *bst = bst_create (autostr_compare);

    bst_insert (bst, astr_create_s ("abc"), 0);
    bst_insert (bst, astr_create_s ("def"), 0);
    bst_insert (bst, astr_create_s ("ghi"), 0);

    assert (bst_nodes (bst) == 3);

    bst_walk (bst, autostr_walk_destroy, 0);
    bst_destroy (bst);

    return 0;
}

static int
t_depth (void)
{
    Bstree *bst = bst_create (autostr_compare);
    ssize_t depth;

    bst_insert (bst, astr_create_s ("abc"), 0);
    bst_insert (bst, astr_create_s ("def"), 0);
    bst_insert (bst, astr_create_s ("ghi"), 0);

    depth = bst_depth (bst);
    assert (depth > 0);
    assert (depth <= 3);

    bst_walk (bst, autostr_walk_destroy, 0);
    bst_destroy (bst);

    return 0;
}

static int
t_lookup (void)
{
    Bstree *bst = bst_create (autostr_compare);
    Bstnode *node;
    Autostr *str;

    bst_insert (bst, astr_create_s ("abc"), 0);
    bst_insert (bst, astr_create_s ("def"), 0);
    bst_insert (bst, astr_create_s ("ghi"), 0);

    str = astr_create_s ("abc");
    node = bst_lookup (bst, str, 0);
    assert (astr_equal (str, bstnode_data (node)));
    astr_destroy (str);

    str = astr_create_s ("def");
    node = bst_lookup (bst, str, 0);
    assert (astr_equal (str, bstnode_data (node)));
    astr_destroy (str);

    str = astr_create_s ("ghi");
    node = bst_lookup (bst, str, 0);
    assert (astr_equal (str, bstnode_data (node)));
    astr_destroy (str);

    str = astr_create_s ("jkl");
    node = bst_lookup (bst, str, 0);
    assert (node == 0);
    astr_destroy (str);

    bst_walk (bst, autostr_walk_destroy, 0);
    bst_destroy (bst);

    return 0;
}

static int
t_delete (void)
{
    Bstree *bst = bst_create (autostr_compare);
    Bstnode *node;
    Autostr *str;

    bst_insert (bst, astr_create_s ("abc"), 0);
    bst_insert (bst, astr_create_s ("def"), 0);
    bst_insert (bst, astr_create_s ("ghi"), 0);

    node = bst_smallest (bst);
    str = bstnode_data (node);
    assert (strcmp (astr_chars (str), "abc") == 0);
    astr_destroy (str);
    bst_delete (bst, node, 0);
    assert (bst_nodes (bst) == 2);
    bst_walk (bst, autostr_walk_verify, 0);
    autostr_walk_verify (0, 0);

    node = bst_smallest (bst);
    str = bstnode_data (node);
    assert (strcmp (astr_chars (str), "def") == 0);
    astr_destroy (str);
    bst_delete (bst, node, 0);
    assert (bst_nodes (bst) == 1);

    node = bst_smallest (bst);
    str = bstnode_data (node);
    assert (strcmp (astr_chars (str), "ghi") == 0);
    astr_destroy (str);
    bst_delete (bst, node, 0);
    assert (bst_nodes (bst) == 0);

    bst_destroy (bst);

    return 0;
}

static int
t_wipe (void)
{
    Bstree *bst = bst_create (autostr_compare);

    bst_insert (bst, astr_create_s ("abc"), 0);
    bst_insert (bst, astr_create_s ("def"), 0);
    bst_insert (bst, astr_create_s ("ghi"), 0);

    bst_walk (bst, autostr_walk_destroy, 0);
    bst_wipe (bst);

    assert (bst_nodes (bst) == 0);

    bst_destroy (bst);

    return 0;
}

static int
t_prune (void)
{
    Bstree *bst = bst_create (autostr_compare);
    Autostr *str = astr_create_s ("abc");

    bst_insert (bst, astr_create_s ("abc"), 0);
    bst_insert (bst, astr_create_s ("def"), 0);
    bst_insert (bst, astr_create_s ("ghi"), 0);

    bst_prune (bst, autostr_prune, str, 0);

    assert (bst_nodes (bst) == 2);

    bst_destroy (bst);

    return 0;
}
