/* tests/t-autostr.c -- test libretto's Autostr functions
 *
 * Aaron Crane <aaronc@pobox.com>
 * 27 December 1997
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * Most Autostr functions are given a thorough workout by this program.
 * But not all.  Specifically, the following functions are not tested here
 * (though I promise that they have been tested in some way):
 *
 * astr_create_size, astr_initialise_size, astr_getline, astr_getdelim,
 * astr_scanf, astr_vscanf, astr_printf, astr_vprintf, astr_printa, astr_vprinta,
 * astr_esc_char_c.
 */

#include <libretto/libretto.h>
#include <libretto/autostr.h>
#include <assert.h>

typedef int (*testable_f) (void);

static int t_create (void);
static int t_create_s (void);
static int t_copy (void);
static int t_copy_s (void);
static int t_copy_c (void);
static int t_cat (void);
static int t_cat_s (void);
static int t_cat_c (void);
static int t_insert (void);
static int t_insert_s (void);
static int t_insert_c (void);
static int t_zero (void);
static int t_delete (void);
static int t_find (void);
static int t_find_s (void);
static int t_find_c (void);
static int t_rfind_c (void);
static int t_equal (void);
static int t_iequal (void);
static int t_cmp (void);
static int t_icmp (void);
static int t_slice (void);
static int t_upcase (void);
static int t_downcase (void);
static int t_flipcase (void);
static int t_capitalise (void);
static int t_upcase_initials (void);
static int t_esc_char (void);

static const testable_f test_tab[] =
{
    &t_create,
    &t_create_s,
    &t_copy_c,
    &t_copy_s,
    &t_copy,
    &t_cat_c,
    &t_cat_s,
    &t_cat,
    &t_insert_c,
    &t_insert_s,
    &t_insert,
    &t_zero,
    &t_delete,
    &t_find_c,
    &t_find_s,
    &t_find,
    &t_rfind_c,
    &t_equal,
    &t_iequal,
    &t_cmp,
    &t_icmp,
    &t_slice,
    &t_upcase,
    &t_downcase,
    &t_flipcase,
    &t_capitalise,
    &t_upcase_initials,
    &t_esc_char
};

int
main (int argc, char **argv)
{
    size_t i;

    (void) argc;
    msg_set_invocation_name (argv[0]);

    for (i = 0;  i < sizeof (test_tab) / sizeof (test_tab[0]);  i++)
	assert (test_tab[i] () == 0);

    return 0;
}

/* I don't like macros in general, but in this situation, I want to make
 * sure that any `failed assertion' messages appear to come from the right
 * function and the right line within that function.
 */

#define check(STR, CONTENTS) assert (strcmp (astr_chars (STR), CONTENTS) == 0)


static int
t_create (void)
{
    Autostr *str;

    str = astr_create ();
    assert (astr_valid_p (str));
    check (str, "");
    astr_destroy (str);
    return 0;
}

static int
t_create_s (void)
{
    Autostr *str;

    str = astr_create_s ("abcdef");
    assert (astr_valid_p (str));
    check (str, "abcdef");
    astr_destroy (str);
    return 0;
}

static int
t_copy_c (void)
{
    Autostr *str = astr_create ();

    astr_copy_c (str, 'a');
    assert (astr_valid_p (str));
    check (str, "a");
    astr_destroy (str);
    return 0;
}

static int
t_copy_s (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "abc");
    assert (astr_valid_p (str));
    check (str, "abc");
    astr_destroy (str);
    return 0;
}

static int
t_copy (void)
{
    Autostr *str = astr_create (), *s = astr_create ();

    astr_copy_s (s, "abc");
    astr_copy (str, s);
    assert (astr_valid_p (str));
    check (str, "abc");
    astr_destroy (str);
    astr_destroy (s);
    return 0;
}

static int
t_cat_c (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "abc");
    astr_cat_c (str, 'd');
    assert (astr_valid_p (str));
    check (str, "abcd");
    astr_destroy (str);
    return 0;
}

static int
t_cat_s (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "abc");
    astr_cat_s (str, "def");
    assert (astr_valid_p (str));
    check (str, "abcdef");
    astr_destroy (str);
    return 0;
}

static int
t_cat (void)
{
    Autostr *str = astr_create (), *s = astr_create ();

    astr_copy_s (str, "abc");
    astr_copy_s (s, "def");
    astr_cat (str, s);
    assert (astr_valid_p (str));
    check (str, "abcdef");
    astr_destroy (str);
    astr_destroy (s);
    return 0;
}

static int
t_insert_c (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "abcd");
    astr_insert_c (str, 0, 'x');
    assert (astr_valid_p (str));
    check (str, "xabcd");

    astr_copy_s (str, "abcd");
    astr_insert_c (str, 2, 'x');
    assert (astr_valid_p (str));
    check (str, "abxcd");

    astr_copy_s (str, "abcd");
    astr_insert_c (str, 4, 'x');
    assert (astr_valid_p (str));
    check (str, "abcdx");

    astr_destroy (str);

    return 0;
}

static int
t_insert_s (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "abcd");
    astr_insert_s (str, 0, "xy");
    assert (astr_valid_p (str));
    check (str, "xyabcd");

    astr_copy_s (str, "abcd");
    astr_insert_s (str, 2, "xy");
    assert (astr_valid_p (str));
    check (str, "abxycd");

    astr_copy_s (str, "abcd");
    astr_insert_s (str, 4, "xy");
    assert (astr_valid_p (str));
    check (str, "abcdxy");

    astr_destroy (str);

    return 0;
}

static int
t_insert (void)
{
    Autostr *str = astr_create (), *xy = astr_create ();

    astr_copy_s (xy, "xy");

    astr_copy_s (str, "abcd");
    astr_insert (str, 0, xy);
    assert (astr_valid_p (str));
    check (str, "xyabcd");

    astr_copy_s (str, "abcd");
    astr_insert (str, 2, xy);
    assert (astr_valid_p (str));
    check (str, "abxycd");

    astr_copy_s (str, "abcd");
    astr_insert (str, 4, xy);
    assert (astr_valid_p (str));
    check (str, "abcdxy");

    astr_destroy (str);
    astr_destroy (xy);

    return 0;
}

static int
t_zero (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "abc");
    astr_zero (str);
    assert (astr_valid_p (str));
    check (str, "");
    astr_destroy (str);
    return 0;
}

static int
t_delete (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "abcdef");
    astr_delete (str, 0, 2);
    assert (astr_valid_p (str));
    check (str, "cdef");

    astr_copy_s (str, "abcdef");
    astr_delete (str, 2, 2);
    assert (astr_valid_p (str));
    check (str, "abef");

    astr_copy_s (str, "abcdef");
    astr_delete (str, 4, 2);
    assert (astr_valid_p (str));
    check (str, "abcd");

    astr_copy_s (str, "abcdef");
    astr_delete (str, 3, -1);
    assert (astr_valid_p (str));
    check (str, "abc");

    astr_destroy (str);
    return 0;
}

static int
t_find_c (void)
{
    Autostr *str = astr_create ();
    ssize_t position[] = { 11, 18, 21, 34, 41, 43, -1 };
    ssize_t i, pos;
    int j = 0;

    astr_copy_s (str, "just some random garbage with repeated characters");
    /*                 0123456789012345678901234567890123456789012345678 */
    /*                 0         1         2         3         4         */

    i = -1;
    do
    {
	pos = astr_find_c (str, i, 'a');
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    astr_destroy (str);

    return 0;
}

static int
t_find_s (void)
{
    Autostr *str = astr_create ();
    ssize_t position[] = { 10, 18, 40, -1 };
    ssize_t i, pos;
    int j = 0;

    astr_copy_s (str, "just some random trash with repeated characters");
    /*                 01234567890123456789012345678901234567890123456 */
    /*                 0         1         2         3         4       */

    i = -1;
    do
    {
	pos = astr_find_s (str, i, "ra");
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    astr_destroy (str);

    return 0;
}

static int
t_find (void)
{
    Autostr *str = astr_create (), *ra = astr_create ();
    ssize_t position[] = { 10, 18, 40, -1 };
    ssize_t i, pos;
    int j = 0;

    astr_copy_s (str, "just some random trash with repeated characters");
    /*                 01234567890123456789012345678901234567890123456 */
    /*                 0         1         2         3         4       */
    astr_copy_s (ra, "ra");

    i = -1;
    do
    {
	pos = astr_find (str, i, ra);
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    astr_destroy (str);
    astr_destroy (ra);

    return 0;
}

static int
t_rfind_c (void)
{
    Autostr *str = astr_create_s ("abcdefb");
    ssize_t pos;

    pos = astr_rfind_c (str, -1, 'f');
    assert (pos == 5);
    pos = astr_rfind_c (str, -1, 'c');
    assert (pos == 2);
    pos = astr_rfind_c (str, -1, 'a');
    assert (pos == 0);
    pos = astr_rfind_c (str, -1, 'b');
    assert (pos == 6);
    pos = astr_rfind_c (str, -1, 'x');
    assert (pos == -1);

    pos = -1;
    pos = astr_rfind_c (str, pos, 'c');
    assert (pos == 2);
    pos = astr_rfind_c (str, pos, 'c');
    assert (pos == -1);

    pos = -1;
    pos = astr_rfind_c (str, pos, 'x');
    assert (pos == -1);
    pos = astr_rfind_c (str, pos, 'x');
    assert (pos == -1);

    pos = -1;
    pos = astr_rfind_c (str, pos, 'b');
    assert (pos == 6);
    pos = astr_rfind_c (str, pos, 'b');
    assert (pos == 1);
    pos = astr_rfind_c (str, pos, 'b');
    assert (pos == -1);

    astr_destroy (str);
    return 0;
}

static int
t_equal (void)
{
    Autostr *a = astr_create (), *b = astr_create ();

    astr_copy_s (a, "hello");

    astr_copy_s (b, "world");
    assert (!astr_equal (a, b));

    astr_copy_s (b, "HeLlO");
    assert (!astr_equal (a, b));

    astr_copy_s (b, "hello");
    assert (astr_equal (a, b));

    astr_destroy (a);
    astr_destroy (b);
    return 0;
}

static int
t_iequal (void)
{
    Autostr *a = astr_create (), *b = astr_create ();

    astr_copy_s (a, "hello");

    astr_copy_s (b, "world");
    assert (!astr_iequal (a, b));

    astr_copy_s (b, "HeLlO");
    assert (astr_iequal (a, b));

    astr_copy_s (b, "hello");
    assert (astr_iequal (a, b));

    astr_destroy (a);
    astr_destroy (b);
    return 0;
}

static int
t_cmp (void)
{
    Autostr *a = astr_create (), *b = astr_create ();

    astr_copy_s (a, "hello");

    astr_copy_s (b, "earth");
    assert (astr_cmp (a, b) > 0);
    assert (astr_cmp (b, a) < 0);

    astr_copy_s (b, "world");
    assert (astr_cmp (a, b) < 0);
    assert (astr_cmp (b, a) > 0);

    astr_copy_s (b, "hello");
    assert (astr_cmp (a, b) == 0);
    assert (astr_cmp (b, a) == 0);

    astr_destroy (a);
    astr_destroy (b);

    return 0;
}

static int
t_icmp (void)
{
    Autostr *a = astr_create (), *b = astr_create ();

    astr_copy_s (a, "hello");

    astr_copy_s (b, "earth");
    assert (astr_icmp (a, b) > 0);
    assert (astr_icmp (b, a) < 0);

    astr_copy_s (b, "world");
    assert (astr_icmp (a, b) < 0);
    assert (astr_icmp (b, a) > 0);

    astr_copy_s (b, "hello");
    assert (astr_icmp (a, b) == 0);
    assert (astr_icmp (b, a) == 0);

    astr_copy_s (b, "HeLlO");
    assert (astr_icmp (a, b) == 0);
    assert (astr_icmp (b, a) == 0);

    astr_destroy (a);
    astr_destroy (b);

    return 0;
}

static int
t_slice (void)
{
    Autostr *a = astr_create (), *b = astr_create ();

    astr_copy_s (a, "abcdefghi");

    astr_slice (b, a, 0, 0);
    assert (astr_valid_p (b));
    check (b, "");

    astr_slice (b, a, 0, 3);
    assert (astr_valid_p (b));
    check (b, "abc");

    astr_slice (b, a, 3, 3);
    assert (astr_valid_p (b));
    check (b, "def");

    astr_slice (b, a, 3, -1);
    assert (astr_valid_p (b));
    check (b, "defghi");

    astr_copy (b, a);
    astr_slice (b, a, 0, 0);
    assert (astr_valid_p (b));
    check (b, "");

    astr_copy (b, a);
    astr_slice (b, b, 0, 3);
    assert (astr_valid_p (b));
    check (b, "abc");

    astr_copy (b, a);
    astr_slice (b, b, 3, 3);
    assert (astr_valid_p (b));
    check (b, "def");

    astr_copy (b, a);
    astr_slice (b, b, 3, -1);
    assert (astr_valid_p (b));
    check (b, "defghi");

    astr_destroy (a);
    astr_destroy (b);

    return 0;
}

static int
t_upcase (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "abc");
    astr_upcase (str);
    assert (astr_valid_p (str));
    check (str, "ABC");

    astr_copy_s (str, "ABC");
    astr_upcase (str);
    assert (astr_valid_p (str));
    check (str, "ABC");

    astr_copy_s (str, ".?");
    astr_upcase (str);
    assert (astr_valid_p (str));
    check (str, ".?");

    astr_copy_s (str, ".abc?");
    astr_upcase (str);
    assert (astr_valid_p (str));
    check (str, ".ABC?");

    astr_destroy (str);

    return 0;
}

static int
t_downcase (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "ABC");
    astr_downcase (str);
    assert (astr_valid_p (str));
    check (str, "abc");

    astr_copy_s (str, "abc");
    astr_downcase (str);
    assert (astr_valid_p (str));
    check (str, "abc");

    astr_copy_s (str, ".?");
    astr_downcase (str);
    assert (astr_valid_p (str));
    check (str, ".?");

    astr_copy_s (str, ".ABC?");
    astr_downcase (str);
    assert (astr_valid_p (str));
    check (str, ".abc?");

    astr_destroy (str);

    return 0;
}

static int
t_flipcase (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "ABC");
    astr_flipcase (str);
    assert (astr_valid_p (str));
    check (str, "abc");

    astr_copy_s (str, "abc");
    astr_flipcase (str);
    assert (astr_valid_p (str));
    check (str, "ABC");

    astr_copy_s (str, "aBc");
    astr_flipcase (str);
    assert (astr_valid_p (str));
    check (str, "AbC");

    astr_destroy (str);

    return 0;
}

static int
t_capitalise (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "abc");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, "Abc");

    astr_copy_s (str, "abc def");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, "Abc Def");

    astr_copy_s (str, ".abc");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, ".Abc");

    astr_copy_s (str, "abc.");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, "Abc.");

    astr_copy_s (str, ".abc.");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, ".Abc.");

    astr_copy_s (str, ".abc def.");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, ".Abc Def.");

    astr_copy_s (str, "ABC");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, "Abc");

    astr_copy_s (str, "ABC DEF");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, "Abc Def");

    astr_copy_s (str, ".ABC");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, ".Abc");

    astr_copy_s (str, "ABC.");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, "Abc.");

    astr_copy_s (str, ".ABC.");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, ".Abc.");

    astr_copy_s (str, ".ABC DEF.");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, ".Abc Def.");

    astr_copy_s (str, "aBC");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, "Abc");

    astr_copy_s (str, "aBC dEF");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, "Abc Def");

    astr_copy_s (str, ".aBC");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, ".Abc");

    astr_copy_s (str, "aBC.");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, "Abc.");

    astr_copy_s (str, ".aBC.");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, ".Abc.");

    astr_copy_s (str, ".aBC dEF.");
    astr_capitalise (str);
    assert (astr_valid_p (str));
    check (str, ".Abc Def.");

    astr_destroy (str);

    return 0;
}

static int
t_upcase_initials (void)
{
    Autostr *str = astr_create ();

    astr_copy_s (str, "abc");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, "Abc");

    astr_copy_s (str, "abc def");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, "Abc Def");

    astr_copy_s (str, ".abc");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, ".Abc");

    astr_copy_s (str, "abc.");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, "Abc.");

    astr_copy_s (str, ".abc.");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, ".Abc.");

    astr_copy_s (str, ".abc def.");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, ".Abc Def.");

    astr_copy_s (str, "ABC");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, "ABC");

    astr_copy_s (str, "ABC DEF");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, "ABC DEF");

    astr_copy_s (str, ".ABC");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, ".ABC");

    astr_copy_s (str, "ABC.");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, "ABC.");

    astr_copy_s (str, ".ABC.");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, ".ABC.");

    astr_copy_s (str, ".ABC DEF.");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, ".ABC DEF.");

    astr_copy_s (str, "aBC");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, "ABC");

    astr_copy_s (str, "aBC dEF");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, "ABC DEF");

    astr_copy_s (str, ".aBC");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, ".ABC");

    astr_copy_s (str, "aBC.");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, "ABC.");

    astr_copy_s (str, ".aBC.");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, ".ABC.");

    astr_copy_s (str, ".aBC dEF.");
    astr_upcase_initials (str);
    assert (astr_valid_p (str));
    check (str, ".ABC DEF.");

    astr_destroy (str);

    return 0;
}

static int
t_esc_char (void)
{
    static const int expected[] =
	{
	    'a', '\a', '\b', 0177, 0033, '\f', '\n', '\r', '\t', '\v',
	    0341, 0207, 0277,
	    1, 1, 0177, 0177,
	    0201, 0201, 0201, 0201,
	    0377, 0377, 0237, 0237,
	    0, 0, 0, 1, 011, 0111, 0222, '2',
	    0, 0, 1, 0x1a, 0x1a, 'a',
	    '<'
	};
    int i, j = 0;
    ssize_t offset = 0, len;
    Autostr *str = astr_create ();

    astr_copy_s (str, ("a\\a\\b\\D\\E\\f\\n\\r\\t\\v"
		      "\\M-a\\M-\\a\\M-?"
		      "\\C-a\\^a\\C-?\\^?"
		      "\\M-\\C-a\\C-\\M-a\\M-\\^a\\^\\M-a"
		      "\\M-\\C-?\\M-\\^?\\C-\\M-?\\^\\M-?"
		      "\\0\\00\\000\\1\\11\\111\\2222"
		      "\\x0\\x00\\x1\\x1a\\x1aa"
		      "\\<"));
    len = astr_length (str);

    while (offset < len)
    {
	i = astr_esc_char (str, &offset);
	assert (i == expected[j]);
	j++;
    }

    return 0;
}
