/* tests/autobuf.c -- test libretto's autobuf functions
 *
 * Aaron Crane <aaronc@pobox.com>
 * 10 January 1998
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <libretto/libretto.h>
#include <libretto/autobuf.h>
#include <libretto/autostr.h>
#include <assert.h>
#include <string.h>

typedef int (*testable_f) (void);

static int t_create (void);
static int t_copy (void);
static int t_copy_buf (void);
static int t_copy_astr (void);
static int t_copy_s (void);
static int t_copy_c (void);
static int t_cat (void);
static int t_cat_buf (void);
static int t_cat_astr (void);
static int t_cat_s (void);
static int t_cat_c (void);
static int t_insert (void);
static int t_insert_buf (void);
static int t_insert_astr (void);
static int t_insert_s (void);
static int t_insert_c (void);
static int t_set (void);
static int t_zero (void);
static int t_delete (void);
static int t_find (void);
static int t_find_buf (void);
static int t_find_astr (void);
static int t_find_s (void);
static int t_find_c (void);
static int t_rfind_c (void);
static int t_equal (void);
static int t_cmp (void);
static int t_slice (void);

static const testable_f test_tab[] =
{
    &t_create,
    &t_copy_astr,
    &t_copy_s,
    &t_copy_c,
    &t_copy_buf,
    &t_copy,
    &t_cat_astr,
    &t_cat_s,
    &t_cat_c,
    &t_cat_buf,
    &t_cat,
    &t_insert_astr,
    &t_insert_s,
    &t_insert_c,
    &t_insert_buf,
    &t_insert,
    &t_set,
    &t_zero,
    &t_delete,
    &t_find_astr,
    &t_find_s,
    &t_find_c,
    &t_find_buf,
    &t_find,
    &t_rfind_c,
    &t_equal,
    &t_cmp,
    &t_slice
};

int
main (int argc, char **argv)
{
    size_t i;

    (void) argc;
    msg_set_invocation_name (argv[0]);

    for (i = 0;  i < sizeof (test_tab) / sizeof (test_tab[0]);  i++)
	assert (test_tab[i] () == 0);

    return 0;
}

#define checkm(ABUF, STR, STRLEN) assert (memcmp (abuf_data (ABUF), STR, STRLEN) == 0)
#define check(ABUF, STR) checkm (ABUF, STR, strlen (STR))

static int
t_create (void)
{
    Autobuf *abuf;

    abuf = abuf_create ();
    assert (abuf_valid_p (abuf));
    assert (abuf_length (abuf) == 0);
    abuf_destroy (abuf);
    return 0;
}

static int
t_copy_astr (void)
{
    Autostr *str = astr_create_s ("abc");
    Autobuf *abuf = abuf_create ();

    abuf_copy_astr (abuf, str);
    assert (abuf_valid_p (abuf));
    check (abuf, "abc");
    astr_destroy (str);
    abuf_destroy (abuf);
    return 0;
}

static int
t_copy_s (void)
{
    Autobuf *abuf = abuf_create ();

    abuf_copy_s (abuf, "abc");
    assert (abuf_valid_p (abuf));
    check (abuf, "abc");
    abuf_destroy (abuf);
    return 0;
}

static int
t_copy_c (void)
{
    Autobuf *abuf = abuf_create ();

    abuf_copy_c (abuf, 'x');
    assert (abuf_valid_p (abuf));
    check (abuf, "x");
    abuf_destroy (abuf);
    return 0;
}

static int
t_copy_buf (void)
{
    Autobuf *abuf = abuf_create ();
    const unsigned char buf[3] = { 'a', 'b', 'c' };

    abuf_copy_buf (abuf, buf, 3);
    assert (abuf_valid_p (abuf));
    checkm (abuf, "abc", 3);
    abuf_destroy (abuf);
    return 0;
}

static int
t_copy (void)
{
    Autobuf *abuf = abuf_create (), *b = abuf_create ();

    abuf_copy_s (b, "abc");
    abuf_copy (abuf, b);
    assert (abuf_valid_p (abuf));
    check (abuf, "abc");
    abuf_destroy (b);
    abuf_destroy (abuf);
    return 0;
}

static int
t_cat_astr (void)
{
    Autobuf *abuf = abuf_create ();
    Autostr *str = astr_create_s ("xyz");

    abuf_copy_s (abuf, "abc");
    abuf_cat_astr (abuf, str);
    assert (abuf_valid_p (abuf));
    check (abuf, "abcxyz");
    astr_destroy (str);
    abuf_destroy (abuf);
    return 0;
}

static int
t_cat_s (void)
{
    Autobuf *abuf = abuf_create ();

    abuf_copy_s (abuf, "abc");
    abuf_cat_s (abuf, "xyz");
    assert (abuf_valid_p (abuf));
    check (abuf, "abcxyz");
    abuf_destroy (abuf);
    return 0;
}

static int
t_cat_c (void)
{
    Autobuf *abuf = abuf_create ();

    abuf_copy_s (abuf, "abc");
    abuf_cat_c (abuf, 'x');
    assert (abuf_valid_p (abuf));
    check (abuf, "abcx");
    abuf_destroy (abuf);
    return 0;
}

static int
t_cat_buf (void)
{
    Autobuf *abuf = abuf_create ();
    const unsigned char buf[3] = { 'x', 'y', 'z' };

    abuf_copy_s (abuf, "abc");
    abuf_cat_buf (abuf, buf, 3);
    assert (abuf_valid_p (abuf));
    check (abuf, "abcxyz");
    abuf_destroy (abuf);
    return 0;
}

static int
t_cat (void)
{
    Autobuf *abuf = abuf_create (), *buf = abuf_create ();

    abuf_copy_s (abuf, "abc");
    abuf_copy_s (buf, "xyz");
    abuf_cat (abuf, buf);
    assert (abuf_valid_p (abuf));
    check (abuf, "abcxyz");
    abuf_destroy (abuf);
    abuf_destroy (buf);
    return 0;
}

static int
t_insert_astr (void)
{
    Autobuf *abuf = abuf_create ();
    Autostr *str = astr_create_s ("xyz");

    abuf_copy_s (abuf, "abcd");
    abuf_insert_astr (abuf, 0, str);
    assert (abuf_valid_p (abuf));
    check (abuf, "xyzabcd");

    abuf_copy_s (abuf, "abcd");
    abuf_insert_astr (abuf, 2, str);
    assert (abuf_valid_p (abuf));
    check (abuf, "abxyzcd");

    abuf_copy_s (abuf, "abcd");
    abuf_insert_astr (abuf, 4, str);
    assert (abuf_valid_p (abuf));
    check (abuf, "abcdxyz");

    astr_destroy (str);
    abuf_destroy (abuf);

    return 0;
}

static int
t_insert_s (void)
{
    Autobuf *abuf = abuf_create ();

    abuf_copy_s (abuf, "abcd");
    abuf_insert_s (abuf, 0, "xyz");
    assert (abuf_valid_p (abuf));
    check (abuf, "xyzabcd");

    abuf_copy_s (abuf, "abcd");
    abuf_insert_s (abuf, 2, "xyz");
    assert (abuf_valid_p (abuf));
    check (abuf, "abxyzcd");

    abuf_copy_s (abuf, "abcd");
    abuf_insert_s (abuf, 4, "xyz");
    assert (abuf_valid_p (abuf));
    check (abuf, "abcdxyz");

    abuf_destroy (abuf);

    return 0;
}

static int
t_insert_c (void)
{
    Autobuf *abuf = abuf_create ();

    abuf_copy_s (abuf, "abcd");
    abuf_insert_c (abuf, 0, 'x');
    assert (abuf_valid_p (abuf));
    check (abuf, "xabcd");

    abuf_copy_s (abuf, "abcd");
    abuf_insert_c (abuf, 2, 'x');
    assert (abuf_valid_p (abuf));
    check (abuf, "abxcd");

    abuf_copy_s (abuf, "abcd");
    abuf_insert_c (abuf, 4, 'x');
    assert (abuf_valid_p (abuf));
    check (abuf, "abcdx");

    abuf_destroy (abuf);

    return 0;
}

static int
t_insert_buf (void)
{
    Autobuf *abuf = abuf_create ();
    const unsigned char buf[3] = {'x', 'y', 'z'};

    abuf_copy_s (abuf, "abcd");
    abuf_insert_buf (abuf, 0, buf, sizeof (buf));
    assert (abuf_valid_p (abuf));
    check (abuf, "xyzabcd");

    abuf_copy_s (abuf, "abcd");
    abuf_insert_buf (abuf, 2, buf, sizeof (buf));
    assert (abuf_valid_p (abuf));
    check (abuf, "abxyzcd");

    abuf_copy_s (abuf, "abcd");
    abuf_insert_buf (abuf, 4, buf, sizeof (buf));
    assert (abuf_valid_p (abuf));
    check (abuf, "abcdxyz");

    abuf_destroy (abuf);

    return 0;
}

static int
t_insert (void)
{
    Autobuf *abuf = abuf_create (), *str = abuf_create ();

    abuf_copy_s (str, "xyz");

    abuf_copy_s (abuf, "abcd");
    abuf_insert (abuf, 0, str);
    assert (abuf_valid_p (abuf));
    check (abuf, "xyzabcd");

    abuf_copy_s (abuf, "abcd");
    abuf_insert (abuf, 2, str);
    assert (abuf_valid_p (abuf));
    check (abuf, "abxyzcd");

    abuf_copy_s (abuf, "abcd");
    abuf_insert (abuf, 4, str);
    assert (abuf_valid_p (abuf));
    check (abuf, "abcdxyz");

    abuf_destroy (str);
    abuf_destroy (abuf);

    return 0;
}

static int
t_set (void)
{
    Autobuf *abuf = abuf_create ();

    abuf_set (abuf, 4, 'x');
    assert (abuf_valid_p (abuf));
    check (abuf, "xxxx");

    abuf_destroy (abuf);

    return 0;
}

static int
t_zero (void)
{
    Autobuf *abuf = abuf_create ();

    abuf_copy_s (abuf, "abcd");
    abuf_zero (abuf);
    assert (abuf_valid_p (abuf));
    check (abuf, "");

    abuf_destroy (abuf);

    return 0;
}

static int
t_delete (void)
{
    Autobuf *abuf = abuf_create ();

    abuf_copy_s (abuf, "abcdef");
    abuf_delete (abuf, 0, 2);
    assert (abuf_valid_p (abuf));
    check (abuf, "cdef");

    abuf_copy_s (abuf, "abcdef");
    abuf_delete (abuf, 2, 2);
    assert (abuf_valid_p (abuf));
    check (abuf, "abef");

    abuf_copy_s (abuf, "abcdef");
    abuf_delete (abuf, 4, 2);
    assert (abuf_valid_p (abuf));
    check (abuf, "abcd");

    abuf_copy_s (abuf, "abcdef");
    abuf_delete (abuf, 3, -1);
    assert (abuf_valid_p (abuf));
    check (abuf, "abc");

    abuf_destroy (abuf);

    return 0;
}

static int
t_find (void)
{
    Autobuf *abuf = abuf_create (), *ra = abuf_create ();
    ssize_t position[] = { 10, 18, 40, -1 };
    ssize_t i, pos;
    int j = 0;

    abuf_copy_s (abuf, "just some random trash with repeated characters");
    /*                  01234567890123456789012345678901234567890123456 */
    /*                  0         1         2         3         4       */
    abuf_copy_s (ra, "ra");

    i = -1;
    do
    {
	pos = abuf_find (abuf, i, ra);
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    abuf_destroy (abuf);
    abuf_destroy (ra);

    return 0;
}

static int
t_find_buf (void)
{
    Autobuf *abuf = abuf_create ();
    const unsigned char buf[2] = { 'r', 'a' };
    ssize_t position[] = { 10, 18, 40, -1 };
    ssize_t i, pos;
    int j = 0;

    abuf_copy_s (abuf, "just some random trash with repeated characters");
    /*                  01234567890123456789012345678901234567890123456 */
    /*                  0         1         2         3         4       */

    i = -1;
    do
    {
	pos = abuf_find_buf (abuf, i, buf, sizeof (buf));
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    abuf_destroy (abuf);

    return 0;
}

static int
t_find_astr (void)
{
    Autobuf *abuf = abuf_create ();
    Autostr *str = astr_create_s ("ra");
    ssize_t position[] = { 10, 18, 40, -1 };
    ssize_t i, pos;
    int j = 0;

    abuf_copy_s (abuf, "just some random trash with repeated characters");
    /*                  01234567890123456789012345678901234567890123456 */
    /*                  0         1         2         3         4       */

    i = -1;
    do
    {
	pos = abuf_find_astr (abuf, i, str);
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    abuf_destroy (abuf);
    astr_destroy (str);

    return 0;
}

static int
t_find_s (void)
{
    Autobuf *abuf = abuf_create ();
    ssize_t position[] = { 10, 18, 40, -1 };
    ssize_t i, pos;
    int j = 0;

    abuf_copy_s (abuf, "just some random trash with repeated characters");
    /*                  01234567890123456789012345678901234567890123456 */
    /*                  0         1         2         3         4       */

    i = -1;
    do
    {
	pos = abuf_find_s (abuf, i, "ra");
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    abuf_destroy (abuf);

    return 0;
}

static int
t_find_c (void)
{
    Autobuf *abuf = abuf_create ();
    ssize_t position[] = { 11, 18, 21, 34, 41, 43, -1 };
    ssize_t i, pos;
    int j = 0;

    abuf_copy_s (abuf, "just some random garbage with repeated characters");
    /*                  0123456789012345678901234567890123456789012345678 */
    /*                  0         1         2         3         4         */

    i = -1;
    do
    {
	pos = abuf_find_c (abuf, i, 'a');
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    abuf_destroy (abuf);

    return 0;
}

static int
t_rfind_c (void)
{
    Autobuf *abuf = abuf_create ();
    ssize_t position[] = { 4, 2, 1, -1 };
    ssize_t i, pos;
    int j = 0;

    abuf_copy_s (abuf, "abbaba");

    i = -1;
    do
    {
	pos = abuf_rfind_c (abuf, i, 'b');
	assert (pos == position[j]);
	j++;
	i = pos;
    } while (i >= 0);

    abuf_destroy (abuf);

    return 0;
}

static int
t_equal (void)
{
    Autobuf *a = abuf_create (), *b = abuf_create ();

    abuf_copy_s (a, "hello");

    abuf_copy_s (b, "world");
    assert (!abuf_equal (a, b));

    abuf_copy_s (b, "HeLlO");
    assert (!abuf_equal (a, b));

    abuf_copy_s (b, "hello");
    assert (abuf_equal (a, b));

    abuf_destroy (a);
    abuf_destroy (b);

    return 0;
}

static int
t_cmp (void)
{
    Autobuf *a = abuf_create (), *b = abuf_create ();

    abuf_copy_s (a, "hello");

    abuf_copy_s (b, "earth");
    assert (abuf_cmp (a, b) > 0);
    assert (abuf_cmp (b, a) < 0);

    abuf_copy_s (b, "world");
    assert (abuf_cmp (a, b) < 0);
    assert (abuf_cmp (b, a) > 0);

    abuf_copy_s (b, "hello");
    assert (abuf_cmp (a, b) == 0);
    assert (abuf_cmp (b, a) == 0);

    abuf_destroy (a);
    abuf_destroy (b);

    return 0;
}

static int
t_slice (void)
{
    Autobuf *a = abuf_create (), *b = abuf_create ();

    abuf_copy_s (a, "abcdefghi");

    abuf_slice (b, a, 0, 0);
    assert (abuf_valid_p (b));
    check (b, "");

    abuf_slice (b, a, 0, 3);
    assert (abuf_valid_p (b));
    check (b, "abc");

    abuf_slice (b, a, 3, 3);
    assert (abuf_valid_p (b));
    check (b, "def");

    abuf_slice (b, a, 3, -1);
    assert (abuf_valid_p (b));
    check (b, "defghi");

    abuf_copy (b, a);
    abuf_slice (b, a, 0, 0);
    assert (abuf_valid_p (b));
    check (b, "");

    abuf_copy (b, a);
    abuf_slice (b, b, 0, 3);
    assert (abuf_valid_p (b));
    check (b, "abc");

    abuf_copy (b, a);
    abuf_slice (b, b, 3, 3);
    assert (abuf_valid_p (b));
    check (b, "def");

    abuf_copy (b, a);
    abuf_slice (b, b, 3, -1);
    assert (abuf_valid_p (b));
    check (b, "defghi");

    abuf_destroy (a);
    abuf_destroy (b);

    return 0;
}
