/* vasnprintf -- main libretto printf routine
 *
 * Aaron Crane <aaronc@pobox.com>
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <libretto/libretto.h>
#include <fmt-scan-priv.h>
#include <libretto/autostr.h>
#include <structs.h>

#include <ctype.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>
#include <errno.h>

typedef union
{
    char v_char;
    short v_short;
    int v_int;
    long v_long;
    long long v_llong;
    unsigned short v_ushort;
    unsigned v_uint;
    unsigned long v_ulong;
    unsigned long long v_ullong;
    double v_double;
    long double v_ldouble;
    const char *v_pchar;
    const void *v_pvoid;
    /* The rest aren't used by real_printf */
    short *v_pshort;		/* for "%hn" */
    int *v_pint;		/* for "%n" */
    long *v_plong;		/* for "%ln" */
    long long *v_pllong;	/* for "%qn" */
    const Autostr *v_string;	/* for "%ts" */
} object_t;

#define SZ_int 0		/* must keep int as zero */
#define SZ_short 1
#define SZ_long 2
#define SZ_llong 3
#define SZ_ldouble SZ_llong	/* just a synonym */

typedef struct
{
    int width;
    int precision;
    unsigned use_width : 1;
    unsigned use_precision : 1;
    unsigned flag_ljustify : 1;
    unsigned flag_printplus : 1;
    unsigned flag_printblank : 1;
    unsigned flag_altformat : 1;
    unsigned flag_zeropad : 1;
    unsigned pr_ucase : 1;
    /* Encode type information. */
    unsigned is_character : 1;
    unsigned is_string : 1;
    unsigned is_pointer : 1;
    unsigned ir_size : 2;	/* used by aAbBdeEfFgGiouxX and V conversions */
    unsigned is_rational : 1;
	unsigned pr_exp : 1;
	unsigned pr_fix : 1;
    unsigned is_integral : 1;
	unsigned is_cardinal : 1;
	unsigned pr_oct;
	unsigned pr_hex;
} format_t;

inline static ssize_t
c_chunkof (ssize_t len, ssize_t chunk_size)
{
    len += chunk_size - 1;
    len &= ~ (chunk_size - 1);
    return len;
}

/* `Grow the reallocing buf so at least N chars can be added' */
inline static int
c_grow (char **as, ssize_t *asz, ssize_t offset, ssize_t n)
{
    if (offset + n >= *asz)
    {
	ssize_t sz = c_chunkof (offset + n, 256);
	char *p = mem_realloc (*as, sz);
	if (!p)
	{
	    /* Last-ditch effort */
	    sz = offset + n + 1; /* 1 for null char */
	    p = mem_realloc (*as, sz);
	    if (!p)
		return -1;
	}
	*asz = sz;
	*as = p;
    }
    return 0;
}

inline static ssize_t
c_putc (char **as, ssize_t *asz, ssize_t offset, int c)
{
    if (c == 0)
	return 0;

    if (c_grow (as, asz, offset, 1) == -1)
	return -1;

    as[0][offset] = c;
    return 1;
}

inline static ssize_t
c_putn (char **as, ssize_t *asz, ssize_t offset, const char *s, ssize_t n)
{
    if (c_grow (as, asz, offset, n) == -1)
	return -1;

    strcpy (*as + offset, s);
    return n;
}

inline static ssize_t
ssmax (ssize_t x, ssize_t y)
{
    return x >= y ? x : y;
}

inline static int
real_printf (char **as, ssize_t *asz, ssize_t offset,
	     const format_t *fmt, const object_t *obj)
{
    char sfmt[64];		/* won't actually use more than about 40 chars */
    char *p = sfmt;
    ssize_t len = 0;
    ssize_t chars_done = 0;
    char *obuf = 0;
#ifndef HAVE_ASPRINTF
    /* "The caller [of sprintf] must make sure that the buffer is big
     * enough; this is not always possible in practice."  Too right, mate.
     * And I do *not* want Libretto programs to be able to smash the stack
     * (or the heap, for that matter), so I've got to use some very large
     * limits to ensure that a buffer I allocate is always big enough.  It
     * pans out as follows: integral types (including long long) will max
     * out at 22 digits for ("%qo", ULONG_LONG_MAX).  Normal floating-point
     * stuff will do <400 digits.  Long double could go up to <5000 digits.
     * Then I have width and precision to take into account.
     *
     * So the way I do it is to assume 1k to begin with.  This is increased
     * if we have a long string to print (but at least then we know exactly
     * how many characters will be printed) or if we're doing a long double
     * (when to be safe we'll have to go to 6k).  We should also add a
     * precision, if we're using one, to the width we've calculated.
     *
     * The only other possibility is to pipe(), fdopen() each end, printf()
     * into the write end, and file_getdelim(-1) from the read end.  That's
     * probably too slow (it requires *at least* three syscalls), but it
     * could be worth bearing in mind.
     */
    ssize_t max_width = 1024;
#endif

    *p++ = '%';

    if (fmt->flag_ljustify)
	*p++ = '-';
    if (fmt->flag_printplus)
	*p++ = '+';
    if (fmt->flag_printblank)
	*p++ = ' ';
    if (fmt->flag_altformat)
	*p++ = '#';
    if (fmt->flag_zeropad)
	*p++ = '0';

    if (fmt->use_width)
    {
	char buf[64];		/* guaranteed enough for a decimal int */
	sprintf (buf, "%d", fmt->width);
	strcpy (p, buf);
	p += strlen (buf);
    }

    if (fmt->use_precision)
    {
	char buf[64];		/* guaranteed enough for a decimal int */
	sprintf (buf, ".%d", fmt->width);
	strcpy (p, buf);
	p += strlen (buf);
    }

    if (fmt->is_character)
	*p++ = 'c';
    else if (fmt->is_pointer)
	*p++ = 'p';
    else if (fmt->is_string)
    {
#ifndef HAVE_ASPRINTF
	if (fmt->use_precision)
	    max_width = fmt->precision;
	else
	    max_width = ssmax (max_width, strlen (obj->v_pchar));
#endif
	*p++ = 's';
    }
    else if (fmt->is_rational)
    {
	if (fmt->ir_size == SZ_ldouble)
	{
	    *p++ = 'L';
#ifndef HAVE_ASPRINTF
	    max_width = 6 * 1024;
#endif
	}

	if (fmt->pr_fix)
	    *p++ = fmt->pr_ucase ? 'F' : 'f';
	else if (fmt->pr_exp)
	    *p++ = fmt->pr_ucase ? 'E' : 'e';
	else
	    *p++ = fmt->pr_ucase ? 'G' : 'g';

#ifndef HAVE_ASPRINTF
	if (fmt->use_precision)
	    max_width += fmt->precision;
#endif
    }
    else if (fmt->is_integral)
    {
	if (fmt->ir_size == SZ_short)
	    *p++ = 'h';
	else if (fmt->ir_size == SZ_long)
	    *p++ = 'l';
	else if (fmt->ir_size == SZ_llong)
	{
	    strcpy (p, LONGLONG_PRINTF_MODIFIER);
	    p += strlen (LONGLONG_PRINTF_MODIFIER);
	}

	if (fmt->pr_oct)
	    *p++ = 'o';
	else if (fmt->pr_hex)
	    *p++ = fmt->pr_ucase ? 'X' : 'x';
	else
	    *p++ = fmt->is_cardinal ? 'u' : 'd';

#ifndef HAVE_ASPRINTF
	if (fmt->use_precision)
	    max_width += fmt->precision;
#endif
    }
    else
    {
	msg_write ("fatal error: bug in Libretto's vasnprintf.c\n");
	abort ();
    }

    *p = 0;

#ifdef HAVE_ASPRINTF
#   define obuf_printf(x) asprintf(&obuf, sfmt, x)
#   define obuf_free(buf) free(obuf)
#else
#   define obuf_printf(x) sprintf(obuf, sfmt, x)
#   define obuf_free(buf) mem_free(buf)
    if (fmt->use_width && fmt->width > max_width)
	max_width = fmt->width;
    obuf = mem_alloc (max_width + 1);
    if (!obuf)			/* out of memory */
	return -1;
#endif

    if (fmt->is_character)
	len = obuf_printf (obj->v_char);
    else if (fmt->is_pointer)
	len = obuf_printf (obj->v_pvoid);
    else if (fmt->is_string)
	len = obuf_printf (obj->v_pchar);
    else if (fmt->is_rational)
    {
	if (fmt->ir_size == SZ_ldouble)
	    len = obuf_printf (obj->v_ldouble);
	else
	    len = obuf_printf (obj->v_double);
    }
    else if (fmt->is_integral)
    {
	if (fmt->ir_size == SZ_short)
	    len = obuf_printf (fmt->is_cardinal ? obj->v_ushort : obj->v_short);
	else if (fmt->ir_size == SZ_long)
	    len = obuf_printf (fmt->is_cardinal ? obj->v_ulong : obj->v_long);
	else if (fmt->ir_size == SZ_llong)
	    len = obuf_printf (fmt->is_cardinal ? obj->v_ullong : obj->v_llong);
	else
	    len = obuf_printf (fmt->is_cardinal ? obj->v_uint : obj->v_int);
    }

    if (!obuf)			/* ran out of memory */
	chars_done = -1;
    else
    {
	chars_done = c_putn (as, asz, offset, obuf, len);
	obuf_free (obuf);
    }

    return chars_done;
}

#define oom_error()							\
	do								\
	{								\
	    as[0][chars_done] = 0;					\
	    errno = ENOMEM;						\
	    return EOF;							\
	} while (0)

/* This is a libretto-internal function, so we can make assumptions about
 * its arguments.  Firstly, all args are non-NULL.  Secondly, either *AS is
 * NULL and *ASZ is zero, or *AS was allocated by mem_alloc to have size
 * *ASZ. */
ssize_t
libretto__vasnprintf (char **as, ssize_t *asz, const char *format_str, va_list va)
{
    const char *p;		/* for stepping through FORMAT_STR */
    ssize_t chars_done = 0;

    for (p = format_str;  *p;  p++)
	if (*p != '%')
	{
	    ssize_t i = c_putc (as, asz, chars_done, *p);
	    if (i == -1)
		oom_error ();
	    chars_done += i;
	}
	else
	{
	    p++;
	    if (*p == '%')
	    {
		ssize_t i = c_putc (as, asz, chars_done, '%');
		if (i == -1)
		    oom_error ();
		chars_done += i;
	    }
	    else if (*p == 0)
	    {
		ssize_t i = c_putc (as, asz, chars_done, '%');
		if (i == -1)
		    oom_error ();
		chars_done += i;
		break;
	    }
	    else		/* a genuine format to parse */
	    {
		format_t fmt;
		object_t obj = { 0 };
		int flag_aat = 0;
		int conv_m = 0, conv_M = 0, conv_n = 0, conv_v = 0;
		char conv_none = 0;

		memset (&fmt, 0, sizeof (format_t));

		/* From here on, if we get an unfinished format, we just
		 * bail and print nothing for that last format.  It would be
		 * possible to do something else; for example, we could here
		 * save the value of P, and if we get to a null char without
		 * finding a conversion specifier, we then print a '%' and
		 * the string from P onwards.  But why bother, when it's the
		 * caller's fault?  It's simplest just to ignore their
		 * error.
		 *
		 * Note also that anything specified in the format which is
		 * redundant, or which makes no sense for the conversion
		 * used, is handled while parsing as if it is useful, but
		 * will be ignored when actually printing.
		 */

		/* find zero or more flags */
		for ( ; *p && strchr ("-+ #0", *p);  p++)
		{
		    if (*p == '-')
			fmt.flag_ljustify = 1;
		    else if (*p == '+')
			fmt.flag_printplus = 1;
		    else if (*p == ' ')
			fmt.flag_printblank = 1;
		    else if (*p == '#')
			fmt.flag_altformat = 1;
		    else	/* it must be '0' */
			fmt.flag_zeropad = 1;
		}

		if (*p == 0)
		    break;

		/* find optional width */
		if (*p == '*' || isdigit (*p))
		{
		    fmt.use_width = 1;
		    if (*p == '*')
		    {
			fmt.width = va_arg (va, int);
			p++;
		    }
		    else
			for ( ; isdigit (*p);  p++)
			{
			    fmt.width *= 10;
			    fmt.width += *p - '0'; /* legal for decimal digits */
			}
		}

		if (*p == 0)
		    break;

		/* find optional dot plus optional precision */
		if (*p == '.')
		{
		    p++;
		    fmt.use_precision = 1;
		    if (*p != '*' && !isdigit (*p))
			fmt.precision = 0;
		    else if (*p == '*')
		    {
			fmt.precision = va_arg (va, int);
			p++;
		    }
		    else	/* must be isdigit (*p) */
			for ( ; isdigit (*p);  p++)
			{
			    fmt.precision *= 10;
			    fmt.precision += *p - '0';	/* legal for decimal digits */
			}
		}

		/* find optional modifiers */
		while (*p && strchr ("hlLqatZ", *p))
		{
		    switch (*p)
		    {
		    case 'l':
			if (fmt.ir_size == SZ_long)
			    fmt.ir_size = SZ_llong;
			else
			    fmt.ir_size = SZ_long;
			break;
		    case 'h':
			if (fmt.ir_size == SZ_short)
			    fmt.is_character = 1;
			else
			    fmt.ir_size = SZ_short;
			break;
		    case 'L': case 'q':
			fmt.ir_size = SZ_llong; break;	/* works for long double */
		    case 'a': case 't':
			flag_aat = *p; break;
		    case 'Z':
			/* I can guarantee to get this accurate with
			 * ssize_t.  If it turns out to be an unsigned
			 * value, it might not be quite right.  I have to
			 * assume that sizeof(size_t) == sizeof(ssize_t). */
#			if SSIZE_MAX == INT_MAX
			    fmt.ir_size = SZ_int;
#			elif SSIZE_MAX == LONG_MAX
			    fmt.ir_size = SZ_long;
#			elif SSIZE_MAX == LONG_LONG_MAX
			    fmt.ir_size = SZ_llong;
#			else
#			    error Your system has a bizarre size for size_t and ssize_t
#			endif
			break;
		    /* no default 'cos I checked *p before switching */
		    }
		}

		/* find conversion */
		if (*p)		/* one of: cdeEfFgGimMnopsuVxX */
		{
		    switch (*p)
		    {
		    case 'c':
			fmt.is_character = 1;
			break;
		    case 'd':
			fmt.is_integral = 1;
			break;
		    case 'e':
			fmt.is_rational = 1;
			fmt.pr_exp = 1;
			break;
		    case 'E':
			fmt.is_rational = 1;
			fmt.pr_exp = 1;
			fmt.pr_ucase = 1;
			break;
		    case 'f':
			fmt.is_rational = 1;
			fmt.pr_fix = 1;
			break;
		    case 'F':
			fmt.is_rational = 1;
			fmt.pr_fix = 1;
			fmt.pr_ucase = 1;
			break;
		    case 'g':
			fmt.is_rational = 1;
			break;
		    case 'G':
			fmt.is_rational = 1;
			fmt.pr_ucase = 1;
			break;
		    case 'i':
			fmt.is_integral = 1;
			break;
		    case 'm':
			fmt.is_string = 1;
			conv_m = 1;
			break;
		    case 'M':
			fmt.is_string = 1;
			conv_M = 1;
			break;
		    case 'n':
			conv_n = 1;
			break;
		    case 'o':
			fmt.is_integral = 1;
			fmt.pr_oct = 1;
			fmt.is_cardinal = 1;
			break;
		    case 'p':
			fmt.is_pointer = 1;
			break;
		    case 's':
			fmt.is_string = 1;
			break;
		    case 'u':
			fmt.is_integral = 1;
			fmt.is_cardinal = 1;
			break;
		    case 'V':
			fmt.is_string = 1;
			conv_v = 1;
			break;
		    case 'x':
			fmt.is_integral = 1;
			fmt.pr_hex = 1;
			fmt.is_cardinal = 1;
			break;
		    case 'X':
			fmt.is_integral = 1;
			fmt.pr_hex = 1;
			fmt.is_cardinal = 1;
			fmt.pr_ucase = 1;
			break;
		    default:
			conv_none = *p;
			break;
		    }
		}

		/* We've now built the parsed format representation.
		 *
		 * The remaining tasks are to set a member of OBJ (and it
		 * must be the member that FMT indicates should be used) and
		 * then call real_printf() to do the work for us.
		 *
		 * But first of all we must handle the conversions that
		 * real_printf doesn't know about; that's just the %n group
		 * and any invalid conversion.
		 */

		if (conv_none != 0)
		{
		    ssize_t i = c_putc (as, asz, chars_done, conv_none);
		    if (i == -1)
			oom_error ();
		    chars_done += i;
		}
		else if (conv_n)
		{
		    if (fmt.ir_size == SZ_short)
		    {
			obj.v_pshort = va_arg (va, short *);
			*obj.v_pshort = chars_done;
		    }
		    else if (fmt.ir_size == SZ_long)
		    {
			obj.v_plong = va_arg (va, long *);
			*obj.v_plong = chars_done;
		    }
		    else if (fmt.ir_size == SZ_llong)
		    {
			obj.v_pllong = va_arg (va, long long *);
			*obj.v_pllong = chars_done;
		    }
		    else	/* must be SZ_int */
		    {
			obj.v_pint = va_arg (va, int *);
			*obj.v_pint = chars_done;
		    }
		}
		else		/* the rest all go to real_printf in some form */
		{
		    void *to_free = 0, *to_mem_free = 0;
		    Autostr *to_str_destroy = 0;
		    ssize_t written;

		    if (conv_v)
			obj.v_pchar = (fmt.ir_size == SZ_long
				       ? libretto_program_name
				       : libretto_program_short_name);
		    else if (conv_m)
			obj.v_pchar = strerror (errno);
		    else if (conv_M)
		    {
			if (flag_aat == 't')
			{
			    to_str_destroy = va_arg (va, Autostr *);
			    obj.v_pchar = to_str_destroy->s;
			}
			else if (flag_aat == 'a')
			{
			    to_mem_free = va_arg (va, char *); /* should be const */
			    obj.v_pchar = to_mem_free;
			}
			else	/* no flag that's meaningful here */
			{
			    to_free = va_arg (va, char *); /* should be const */
			    obj.v_pchar = to_free;
			}
		    }
		    else if (fmt.is_string)
		    {
			if (flag_aat == 't')
			{
			    obj.v_string = va_arg (va, const Autostr *);
			    obj.v_pchar = obj.v_string->s;
			}
			else	/* no flag that's meaningful here */
			    obj.v_pchar = va_arg (va, const char *);
		    }
		    else if (fmt.is_character)
			obj.v_char = va_arg (va, int);
		    else if (fmt.is_pointer)
			obj.v_pvoid = va_arg (va, const void *);
		    else if (fmt.is_rational)
		    {
			if (fmt.ir_size == SZ_ldouble)
			    obj.v_ldouble = va_arg (va, long double);
			else
			    obj.v_double = va_arg (va, double);
		    }
		    else if (fmt.is_integral)
		    {
			if (fmt.is_cardinal)
			{
			    if (fmt.ir_size == SZ_short)
				obj.v_ushort = va_arg (va, unsigned int);
			    else if (fmt.ir_size == SZ_long)
				obj.v_ulong = va_arg (va, unsigned long);
			    else if (fmt.ir_size == SZ_llong)
				obj.v_ullong = va_arg (va, unsigned long long);
			    else /* ir_size == SZ_int */
				obj.v_uint = va_arg (va, unsigned int);
			}
			else	/* signed values */
			{
			    if (fmt.ir_size == SZ_short)
				obj.v_short = va_arg (va, int);
			    else if (fmt.ir_size == SZ_long)
				obj.v_long = va_arg (va, long);
			    else if (fmt.ir_size == SZ_llong)
				obj.v_llong = va_arg (va, long long);
			    else /* ir_size == SZ_int */
				obj.v_int = va_arg (va, int);
			}
		    }

		    /* and finally, print the stuff */
		    written = real_printf (as, asz, chars_done, &fmt, &obj);
		    if (written == -1)
			oom_error ();
		    chars_done += written;

		    if (to_free)
			free (to_free);
		    else if (to_mem_free)
			mem_free (to_mem_free);
		    else if (to_str_destroy)
			astr_destroy (to_str_destroy);
		}

	    }
	}

    as[0][chars_done] = 0;

    return chars_done;
}
