#line 2 "file.iif"		/* -*- C -*- */

/* file.iif -- interface to scanf for FILE *
 *
 * Aaron Crane <aaronc@pobox.com>
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <libretto/libretto.h>
#include <libretto/autostr.h>
#include <libretto/autobuf.h>
#include <structs.h>

#include <ctype.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>

#define scan_func libretto__vfscanf
#define scan_source_type FILE *
#define extra_source_type int	/* actually, it's unused */

inline static int
get_c (scan_source_type stream, extra_source_type *p)
{
    (void) p;
    return getc (stream);
}

inline static void
init_extra (scan_source_type stream, extra_source_type *p)
{
    (void) stream;
    (void) p;
}

inline static void
finish_extra (scan_source_type stream, extra_source_type *p)
{
    (void) stream;
    (void) p;
}
#line 2 "scan.c.in"		/* -*- C -*- */

/* scan.c.in -- file to build scanf functions from
 *
 * Aaron Crane <aaronc@pobox.com>
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 * Much of the code in this file was taken from the GNU C Library, Copyright
 * (C) 1991, 92, 93, 94, 95, 96, 97 Free Software Foundation, Inc.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Interface */

/* You are responsible for my include files.  I need <stdlib.h>, <assert.h>,
 * <stdarg.h>, <sys/types.h> (or more specifically, size_t and ssize_t),
 * <stdio.h> (or more specifically, EOF), <string.h>, <ctype.h>, and most of
 * all, <libretto-priv.h> (which itself must include <config.h> and
 * <libretto.h>).  This is in addition to anything you may need for your
 * functions -- for example, you'll need <stdio.h> for sure if you're doing
 * `FILE *' input sources.
 */

/* Define `scan_func' to be the name of the function you want.
 *
 * Define `scan_source_type' to be the type of the argument which will be
 * the source of the scanning.
 *
 * Define `extra_source_type' to be the type of an extra argument to a get
 * function.
 */

/* Also define this function; it must behave in the obvious way. */
inline static int get_c (scan_source_type s, extra_source_type *e);

/* init_extra() is called before getting any characters from SOURCE.
 * finish_extra() is called after all characters have been got from
 * SOURCE. */
inline static void init_extra (scan_source_type s, extra_source_type *e);
inline static void finish_extra (scan_source_type s, extra_source_type *e);


/* Implementation */


/* And finally, I define this function.  This is the one that the script
 * building a C file from this actually wants. */

int scan_func (scan_source_type source, const char *fmt, va_list va);

#define inchar()	((c = get_c (source, &extra)) == EOF		\
			 ? ((c = EOF), EOF)				\
			 : ++read_in, c)
#define conv_error()	do { finish_extra (source, &extra); return done; } while (0)
#define input_error()	do { finish_extra (source, &extra);		\
			     return (done == 0) ? EOF : done; } while (0)

int
scan_func (scan_source_type source, const char *format, va_list va)
{
    extra_source_type extra;

    const char *f = format;
    char fc;			/* Current character of the format.  */
    size_t done = 0;		/* Assignments done.  */
    size_t read_in = 0;		/* Chars read in.  */
    int c;			/* Last char read.  */
    int do_assign;		/* Whether to do an assignment.  */
    int width;			/* Maximum field width.  */
    
#define	is_longlong is_long_double
				/* use the `L' modifier for `long long' */
    char is_short, is_long, is_long_double; /* Type modifiers */
    int use_malloc_string;		/* Args are char ** to be filled in */
    int use_auto_string = 0;	/* args are Autostr * to be filled in */
    int use_ab_string = 0;	/* args are Autobuf * to be filled in */
    char got_dot, got_e;	/* Status for reading F-P nums */
    char not_in;		/* If a [...] is a [^...] */
    int base;			/* Base for integral numbers */
    int number_signed = 0;	/* Signedness for integral numbers */
    long int num = 0;		/* Integral holding variables */
    unsigned long int unum = 0;	/* (ditto) */
    long double fp_num;		/* Floating-point holding variable */
    register char *str = 0, **strptr = 0; /* Character-buffer pointer */
    Autostr *autostr = 0;	/* Autostr pointer to read into */
    Autobuf *ab = 0;		/* Autobuf pointer to read into */
    size_t strsize = 0;
    char work[200];		/* Workspace */
    char *w;			/* Pointer into WORK */
    char decimal = '.';		/* Decimal point character */

    init_extra (source, &extra);

    c = inchar();

    /* Run through the format string.  */
    while (*f != '\0')
    {
	fc = *f++;
	if (fc != '%')
	{
	    /* Characters other than format specs must just match.  */
	    if (c == EOF)
		input_error();
	    if (isspace(fc))
	    {
		/* Whitespace characters match any amount of whitespace.  */
		while (isspace (c))
		    inchar ();
		continue;
	    }
	    else if (c == fc)
		(void) inchar();
	    else
		conv_error();
	    continue;
	}

	/* Check for the assignment-suppressant.  */
	if (*f != '*')
	    do_assign = 1;
	else
	{
	    do_assign = 0;
	    ++f;
	}
		
	/* Find the maximum field width.  */
	width = 0;
	while (isdigit(*f))
	{
	    width *= 10;
	    width += *f++ - '0';
	}
	if (width == 0)
	    width = -1;

	/* Check for type modifiers.  */

	is_short = is_long = is_long_double = use_malloc_string = use_auto_string
	    = use_ab_string = 0;

	while (strchr ("hlLqatA", *f))
	    switch (*f++)
	    {
	    case 'h':
		/* int's are short int's.  */
		is_short = 1;
		break;
	    case 'l':
		if (is_long)
		    /* A double `l' is equivalent to an `L'.  */
		    is_longlong = 1;
		else
		    /* int's are long int's.  */
		    is_long = 1;
		break;
	    case 'q':
	    case 'L':
		/* double's are long double's, and int's are long long int's.  */
		is_long_double = 1;
		break;
	    case 'a':
		/* String conversions (%s, %[) take a `char **' arg and fill it
		 * in with a malloc'd pointer. */
		use_malloc_string = 1;
		break;
	    case 't':
		/* String conversions (%s, %[) take an `Autostr *' arg and
		 * fill it in using autostr functions. */
		use_auto_string = 1;
		break;
	    case 'A':
		/* String conversions (%s, %[) take an `Autobuf *' arg and
		 * fill it in using autobuf functions. */
		use_ab_string = 1;
	    }

	/* End of the format string?  */
	if (*f == '\0')
	    conv_error();

	/* Find the conversion specifier.  */
	w = work;
	fc = *f++;
	if (fc != '[' && fc != 'c' && fc != 'n')
	    /* Eat whitespace.  */
	    while (isspace(c))
		(void) inchar();
	switch (fc)
	{
	case '%':		/* Must match a literal '%'.  */
	    if (c != fc)
		conv_error();
	    break;

	case 'n':		/* Answer number of assignments done.  */
	    if (do_assign)
		*va_arg(va, int *) = read_in;
	    break;

	case 'c':		/* Match characters.  */
	    if (do_assign)
	    {
		str = va_arg (va, char *);
		if (str == NULL)
		    conv_error ();
	    }

	    if (c == EOF)
		input_error();

	    if (width == -1)
		width = 1;

	    if (do_assign)
	    {
		do
		    *str++ = c;
		while (inchar() != EOF && --width > 0);
	    }
	    else
		while (inchar() != EOF && width > 0)
		    --width;

	    if (do_assign)
		++done;

	    break;

	case 's':		/* Read a string.  */
#define STRING_ARG								\
	    if (do_assign)							\
	    {									\
		if (use_malloc_string)						\
		{								\
		    /* The string is to be stored in a malloc'd buffer. */	\
		    strptr = va_arg (va, char **);				\
		    if (strptr == NULL)						\
			conv_error ();						\
		    /* Allocate an initial buffer. */				\
		    strsize = 100;						\
		    *strptr = str = malloc (strsize);				\
		}								\
		else if (use_auto_string)					\
		{								\
		    /* The string is stored in the autostr. */			\
		    autostr = va_arg (va, Autostr *);				\
		    assert (autostr);						\
		    astr_zero (autostr);						\
		    /* This next line is needed only to keep `if (str ==	\
		     * NULL) conv_error();' below happy -- we rely on		\
		     * autostr code to do the right thing. */			\
		    str = autostr->s;						\
		}								\
		else if (use_ab_string)						\
		{								\
		    /* The string is stored in the autobuf. */			\
		    ab = va_arg (va, Autobuf *);				\
		    assert (ab);						\
		    abuf_zero (ab);						\
		    /* This next line is needed only to keep `if (str ==	\
		     * NULL) conv_error();' below happy -- we rely on		\
		     * autobuf code to do the right thing. */			\
		    str = (char *) ab;						\
		}								\
		else								\
		    str = va_arg (va, char *);					\
		if (str == NULL)						\
		    conv_error ();						\
	    }

	    STRING_ARG;
	    
	    if (c == EOF)
		input_error ();

	    do
	    {
		if (isspace (c))
		    break;
#define	STRING_ADD_CHAR(c)								\
		if (do_assign)								\
		{									\
		    if (use_auto_string)						\
		    {									\
			if (astr_cat_c (autostr, c) == -1)				\
			{								\
			    /* We lose.  Oh well.  Stop converting, so at		\
			     * least we don't swallow any input. */			\
			    ++done;							\
			    conv_error ();						\
			}								\
		    }									\
		    else if (use_ab_string)						\
		    {									\
			if (abuf_cat_c (ab, c) == -1)					\
			{								\
			    /* We lose.  Oh well.  Stop converting, so at		\
			     * least we don't swallow any input. */			\
			    ++done;							\
			    conv_error ();						\
			}								\
		    }									\
		    else								\
		    {									\
			*str++ = c;							\
			if (use_malloc_string && str == *strptr + strsize)		\
			{								\
			    /* Enlarge the buffer.  */					\
			    str = realloc (*strptr, strsize * 2);			\
			    if (str == NULL)						\
			    {								\
				/* Can't allocate that much.  Last-ditch		\
                                 * effort. */						\
				str = realloc (*strptr, strsize + 1);			\
				if (str == NULL)					\
				{							\
				    /* We lose.  Oh well. Terminate the string		\
				     * and stop converting, so at least we		\
				     * don't swallow any input. */			\
				    (*strptr)[strsize] = '\0';				\
				    ++done;						\
				    conv_error ();					\
				}							\
				else							\
				{							\
				    *strptr = str;					\
				    str += strsize;					\
				    ++strsize;						\
				}							\
			    }								\
			    else							\
			    {								\
				*strptr = str;						\
				str += strsize;						\
				strsize *= 2;						\
			    }								\
			}								\
		    }									\
		}

		STRING_ADD_CHAR (c);
	    } while (inchar () != EOF && (width <= 0 || --width > 0));

#define STRING_TERMINATE()						\
	    do								\
	    {								\
		if (do_assign)						\
		{							\
		    if (!use_auto_string && !use_ab_string)		\
			*str = '\0';					\
		    ++done;						\
		}							\
	    } while (0)

	    STRING_TERMINATE ();

	    break;

	case 'x':		/* Hexadecimal integer.  */
	case 'X':		/* Ditto.  */ 
	    base = 16;
	    number_signed = 0;
	    goto number;

	case 'o':		/* Octal integer.  */
	    base = 8;
	    number_signed = 0;
	    goto number;

	case 'u':		/* Unsigned decimal integer.  */
	    base = 10;
	    number_signed = 0;
	    goto number;

	case 'd':		/* Signed decimal integer.  */
	    base = 10;
	    number_signed = 1;
	    goto number;

	case 'i':		/* Generic number.  */
	    base = 0;
	    number_signed = 1;

	number:
	    if (c == EOF)
		input_error();

	    /* Check for a sign.  */
	    if (c == '-' || c == '+')
	    {
		*w++ = c;
		if (width > 0)
		    --width;
		(void) inchar();
	    }

	    /* Look for a leading indication of base.  */
	    if (c == '0')
	    {
		if (width > 0)
		    --width;
		*w++ = '0';

		(void) inchar();

		if (tolower(c) == 'x')
		{
		    if (base == 0)
			base = 16;
		    if (base == 16)
		    {
			if (width > 0)
			    --width;
			(void) inchar();
		    }
		}
		else if (base == 0)
		    base = 8;
	    }

	    if (base == 0)
		base = 10;

	    /* Read the number into WORK.  */
	    do
	    {
		if (base == 16 ? !isxdigit(c) :
		    (!isdigit(c) || c - '0' >= base))
		    break;
		*w++ = c;
		if (width > 0)
		    --width;
	    } while (inchar() != EOF && width != 0);

	    if (w == work ||
		(w - work == 1 && (work[0] == '+' || work[0] == '-')))
		/* There was on number.  */
		conv_error();

	    /* Convert the number.  */
	    *w = '\0';
	    if (number_signed)
		num = strtol (work, &w, base);
	    else
		unum = strtoul (work, &w, base);
	    if (w == work)
		conv_error ();

	    if (do_assign)
	    {
		if (! number_signed)
		{
		    if (is_longlong)
			*va_arg (va, unsigned long long *) = unum;
		    else if (is_long)
			*va_arg (va, unsigned long *) = unum;
		    else if (is_short)
			*va_arg (va, unsigned short *) = (unsigned short) unum;
		    else
			*va_arg(va, unsigned int *) = (unsigned int) unum;
		}
		else
		{
		    if (is_longlong)
			*va_arg(va, long long int *) = num;
		    else if (is_long)
			*va_arg(va, long int *) = num;
		    else if (is_short)
			*va_arg(va, short int *) = (short int) num;
		    else
			*va_arg(va, int *) = (int) num;
		}
		++done;
	    }
	    break;

	case 'e':		/* Floating-point numbers.  */
	case 'E':
	case 'f':
	case 'F':
	case 'g':
	case 'G':
	    if (c == EOF)
		input_error();

	    /* Check for a sign.  */
	    if (c == '-' || c == '+')
	    {
		*w++ = c;
		if (inchar() == EOF)
		    /* EOF is only an input error before we read any chars.  */
		    conv_error();
		if (width > 0)
		    --width;
	    }

	    got_dot = got_e = 0;
	    do
	    {
		if (isdigit(c))
		    *w++ = c;
		else if (got_e && w[-1] == 'e' && (c == '-' || c == '+'))
		    *w++ = c;
		else if (!got_e && tolower(c) == 'e')
		{
		    *w++ = 'e';
		    got_e = got_dot = 1;
		}
		else if (c == decimal && !got_dot)
		{
		    *w++ = c;
		    got_dot = 1;
		}
		else
		    break;
		if (width > 0)
		    --width;
	    } while (inchar() != EOF && width != 0);

	    if (w == work)
		conv_error();
	    if (w[-1] == '-' || w[-1] == '+' || w[-1] == 'e')
		conv_error();

#ifndef MIB_HACKS
	    /* Convert the number.  */
	    *w = '\0';
	    fp_num = strtod(work, &w);
	    if (w == work)
		conv_error();

	    if (do_assign)
	    {
		if (is_long_double)
		    *va_arg(va, long double *) = fp_num;
		else if (is_long)
		    *va_arg(va, double *) = (double) fp_num;
		else
		    *va_arg(va, float *) = (float) fp_num;
		++done;
	    }
	    break;
#endif /* MIB_HACKS */

	case '[':	/* Character class.  */
	    STRING_ARG;

	    if (c == EOF)
		input_error();

	    if (*f == '^')
	    {
		++f;
		not_in = 1;
	    }
	    else
		not_in = 0;

	    while ((fc = *f++) != '\0' && fc != ']')
	    {
		if (fc == '-' && *f != '\0' && *f != ']' &&
		    w > work && w[-1] <= *f)
		    /* Add all characters from the one before the '-' up to
		     * (but not including) the next format char.  */
		    for (fc = w[-1] + 1; fc < *f; ++fc)
			*w++ = fc;
		else
		    /* Add the character to the list.  */
		    *w++ = fc;
	    }
	    if (fc == '\0')
		conv_error();

	    *w = '\0';
	    unum = read_in;
	    do
	    {
		if ((strchr (work, c) == NULL) != not_in)
		    break;
		STRING_ADD_CHAR (c);
		if (width > 0)
		    --width;
	    } while (inchar () != EOF && width != 0);
	    if (read_in == unum)
		conv_error ();

	    STRING_TERMINATE ();

	    break;

	case 'p':		/* Generic pointer.  */
	    base = 16;
	    /* A PTR must be the same size as a `long int'.  */
	    is_long = 1;
	    goto number;
	}
    }

    finish_extra (source, &extra);
    return done;
}
