/* mustm.h -- implementation for malloc-or-die functions in libibretto
 *
 * Aaron Crane <aaronc@pobox.com>
 * 10 July 1996
 * 21 September 1997:
 *	the Great Renaming (mustmalloc -> mem_alloc)
 *	Removed the (useless) muststrdup
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <libretto/libretto.h>

#include <assert.h>
#include <stdlib.h>

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif
#ifndef SIGSEGV
/* This looks evil, but it's defined as 11 all the way back to Seventh
 * Edition manpages. */
#define SIGSEGV (11)
#endif

static enum mem_oom_mode failure_mode = LIBRETTO_OOM_QUIT;

static void force_segfault (void) __attribute__ ((__noreturn__));

inline static void
check (void *p)
{
    if (!p)
    {
	if (failure_mode == LIBRETTO_OOM_QUIT)
	{
	    /* We don't want to attempt memory allocation here, so we can't
	     * use my printf stuff.  In fact, we don't want to use *any*
	     * printf stuff, on the off chance that there isn't enough
	     * memory. */
	    fputs (libretto_program_short_name, stderr);
	    fputs (": out of memory -- quitting\n", stderr);
	    exit (1);
	}
	else if (failure_mode == LIBRETTO_OOM_SEGV)
	{
	    fputs (libretto_program_short_name, stderr);
	    fputs (": out of memory -- trying to segfault\n", stderr);
	    force_segfault ();
	}
    }
}

enum mem_oom_mode
mem_failure_mode (void)
{
    return failure_mode;
}

int
mem_set_failure_mode (enum mem_oom_mode mode)
{
    if (mode <= 0 || mode >= LIBRETTO_OOM_MAX_MODE)
	return -1;

    failure_mode = mode;
    return 0;
}

void
mem_free (void *p)
{
#ifdef FREE_IS_BROKEN
    if (p)
#endif
	free (p);
}

void *
mem_alloc (size_t n)
{
    void *p;

    p = malloc (n);
    check (p);
    return p;
}

void *
mem_realloc (void *p, size_t n)
{
    p = realloc (p, n);
    check (p);
    return p;
}

int
mem_try_realloc (void **pp, size_t n)
{
    void *p;

    assert (pp);

    if (n == 0)
    {
	free (*pp);
	*pp = 0;
    }
    else
    {
	p = realloc (*pp, n);
	if (!p)
	    return -1;
	*pp = p;
    }

    return 0;
}

static void
force_segfault (void)
{
#ifdef HAVE_RAISE
    raise (SIGSEGV);
#elif defined HAVE_GSIGNAL
    gsignal (SIGSEGV);
#else
    kill (getpid (), SIGSEGV);
#endif

    /* Perhaps that didn't work for some reason.  Let's try another
     * method, just in case. */
    *((volatile char *) 0) = 1;

    /* And on the off chance that we're still going, let's just abort. */
    abort ();
}
