/* message.c -- implementation for message functions in libretto
 *
 * Aaron Crane <aaronc@pobox.com>
 * 7 July 1996
 * 30 April 1997
 * 5 February 1998 -- preparing for platform independence
 * 21 February 1998 -- now works (I hope) on many platforms
 * 27 February 1998 -- renamed everything for release 2
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <libretto/libretto.h>
#include <fmt-scan-priv.h>

#include <stdlib.h>
#include <assert.h>
#include <string.h>

const char *libretto_program_name = "";
const char *libretto_program_short_name = "";

int
msg_set_invocation_name (const char *argv0)
{
    char *p;
    static char *invocation_name = 0;

    assert (argv0);

    p = mem_realloc (invocation_name, strlen (argv0) + 1);
    if (!p)
	return -1;

    invocation_name = p;
    strcpy (invocation_name, argv0);
    libretto_program_name = invocation_name;
    p = strrchr (libretto_program_name, '/');
    libretto_program_short_name = p ? p + 1 : libretto_program_name;

    return 0;
}

inline static void
vmessage (const char *fmt, va_list va)
{
    fprintf (stderr, "%s: ", libretto_program_short_name); /* faster than libretto__vfp */
    /* No meaningful return on out-of-memory */
    libretto__vfprintf (stderr, fmt, va);
}

inline static void
vmessage_cc (const char *file, ssize_t line, const char *fmt, va_list va)
{
    fprintf (stderr, "%s:%ld: ", file, (long) line);
    /* No meaningful return on out-of-memory */
    libretto__vfprintf (stderr, fmt, va);
}

inline static void
vmessage_l (const char *file, ssize_t line, const char *fmt, va_list va)
{
    fprintf (stderr, "%s:%s:%ld: ", libretto_program_short_name, file, (long) line);
    /* No meaningful return on out-of-memory */
    libretto__vfprintf (stderr, fmt, va);
}

inline static void
vmessage_pid (const char *fmt, va_list va)
{
    fprintf (stderr, "%s[%ld]: ", libretto_program_short_name, (long) getpid ());
    /* No meaningful return on out-of-memory */
    libretto__vfprintf (stderr, fmt, va);
}

void
msg_printf (const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    /* No meaningful return on out-of-memory */
    libretto__vfprintf (stderr, fmt, va);
    va_end (va);
}

void
msg_write (const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage (fmt, va);
    va_end (va);
}

void
msg_fatal (const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage (fmt, va);
    va_end (va);
    exit (1);
}

void
msg_fataln (int status, const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage (fmt, va);
    va_end (va);
    if (status >= 0)
	exit (status);
    abort ();
}

void
msg_cc_write (const char *file, ssize_t line, const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage_cc (file, line, fmt, va);
    va_end (va);
}

void
msg_cc_fatal (const char *file, ssize_t line, const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage_cc (file, line, fmt, va);
    va_end (va);
    exit (1);
}

void
msg_cc_fataln (int status, const char *file, ssize_t line, const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage_cc (file, line, fmt, va);
    va_end (va);
    if (status >= 0)
	exit (status);
    abort ();
}

void
msg_l_write (const char *file, ssize_t line, const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage_l (file, line, fmt, va);
    va_end (va);
}

void
msg_l_fatal (const char *file, ssize_t line, const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage_l (file, line, fmt, va);
    va_end (va);
    exit (1);
}

void
msg_l_fataln (int status, const char *file, ssize_t line, const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage_l (file, line, fmt, va);
    va_end (va);
    if (status >= 0)
	exit (status);
    abort ();
}

void
msg_pid_write (const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage_pid (fmt, va);
    va_end (va);
}

void
msg_pid_fatal (const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage_pid (fmt, va);
    va_end (va);
    exit (1);
}

void
msg_pid_fataln (int status, const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    vmessage_pid (fmt, va);
    va_end (va);
    if (status >= 0)
	exit (status);
    abort ();
}

void
msg_usage (int status, const char *fmt, ...)
{
    va_list va;

    assert (fmt);

    va_start (va, fmt);
    /* No meaningful return on out-of-memory */
    libretto__vfprintf (stderr, fmt, va);
    va_end (va);

    if (status >= 0)
	exit (status);
}
