/* fmt-scan.c -- implementation for formatting and scanning funcs in Libretto
 *
 * Aaron Crane <aaronc@pobox.com>
 * 26 February 1998
 *
 * This file is part of Libretto, a library of useful functions.
 *
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <libretto/libretto.h>
#include <libretto/autostr.h>
#include <structs.h>
#include <fmt-scan-priv.h>
#include <autostr-priv.h>

#include <assert.h>
#include <string.h>

ssize_t
libretto__vfprintf (FILE *stream, const char *fmt, va_list va)
{
    char *as = 0;
    ssize_t asz = 0, i;

    i = libretto__vasnprintf (&as, &asz, fmt, va);
    if (as)
    {
	fputs (as, stream);
	mem_free (as);
    }
    return i;
}

ssize_t
file_printf (FILE *stream, const char *fmt, ...)
{
    ssize_t i;

    va_list (va);

    va_start (va, fmt);
    i = libretto__vfprintf (stream, fmt, va);
    va_end (va);
    return i;
}

ssize_t
file_vprintf (FILE *stream, const char *fmt, va_list va)
{
    return libretto__vfprintf (stream, fmt, va);
}

/* We play a sneaky but valid trick in i_str_vprintf: instead of copying
 * *again* once we've done the vasnprintf, we just set the fields of the
 * Autostr appropriately (remembering to free the existing contents
 * beforehand). */

inline static ssize_t
i_str_vprintf (Autostr *str, const char *format, va_list va)
{
    ssize_t i;
    ssize_t asz = 0;
    char *as = 0;

    i = libretto__vasnprintf (&as, &asz, format, va);

    /* Possible values of I and AS:
     *
     * 1. I is positive.  AS is non-null.
     *       Everything worked, all memory was allocated.
     * 2. I is negative.  AS is null.
     *       Nothing worked.  No buffer was allocated at all.  errno should
     *       be ENOMEM.
     * 3. AS is non-null.  I is negative.
     *       Some things worked.  We managed to write *some* stuff into the
     *       buffer, but had to stop part way through.  errno should be
     *       ENOMEM.  If we want to know how many chars were written, we
     *       have to use strlen.
     * 4. I is positive.  AS is null.
     *       Bug.  Can't happen.
     */

    if (!as)
    {
	assert (i == EOF);
	s_zero (str);
    }
    else
    {
	/* We wrote some stuff at least.  Put it into STR. */
	mem_free (str->s);
	str->s = as;
	str->alloced = asz;
	str->length = (i != EOF) ? i : strlen (str->s);
    }

    return i;
}

ssize_t
astr_printf (Autostr *str, const char *format, ...)
{
    ssize_t i;
    va_list va;

    assert (str);
    assert (format);

    va_start (va, format);
    i = i_str_vprintf (str, format, va);
    va_end (va);

    return i;
}

ssize_t
astr_vprintf (Autostr *str, const char *format, va_list va)
{
    assert (str);
    assert (format);

    return i_str_vprintf (str, format, va);
}

/* To do the str_printa() stuff, I just vasnprintf() and then copy from that
 * string.  I could possibly speed things up by changing the call convention
 * for libretto__vasnprintf such that it takes not only a `char **' and an
 * `ssize_t *', but also a `ssize_t' which is the offset at which to start
 * writing.  That should bolt on fairly neatly.  For now, though, I'll just
 * stick to the simple, slow version.
 *
 * Another reason for making that change is that it reduces the chance of
 * memory allocation failure. */

inline static ssize_t
i_str_vprinta (Autostr *str, const char *format, va_list va)
{
    char *as = 0;
    ssize_t asz = 0;
    ssize_t i;

    i = libretto__vasnprintf (&as, &asz, format, va);
    if (as)
    {
	if (astr_cat_s (str, as) == -1)
	    i = -1;
	mem_free (as);
    }

    return i;
}

ssize_t
astr_printa (Autostr *str, const char *format, ...)
{
    ssize_t i;
    va_list va;

    assert (str);
    assert (format);

    va_start (va, format);
    i = i_str_vprinta (str, format, va);
    va_end (va);

    return i;
}

ssize_t
astr_vprinta (Autostr *str, const char *format, va_list va)
{
    assert (str);
    assert (format);

    return i_str_vprinta (str, format, va);
}

inline static int
i_as_vprintf (char **as, const char *fmt, va_list va)
{
    ssize_t asz = 0;

    if (*as)
    {
	mem_free (*as);
	*as = 0;
    }

    return libretto__vasnprintf (as, &asz, fmt, va);
}

ssize_t
as_printf (char **as, const char *fmt, ...)
{
    va_list va;
    ssize_t i;

    assert (as);
    assert (fmt);

    va_start (va, fmt);
    i = i_as_vprintf (as, fmt, va);
    va_end (va);

    return i;
}

ssize_t
as_vprintf (char **as, const char *fmt, va_list va)
{
    assert (as);
    assert (fmt);

    return i_as_vprintf (as, fmt, va);
}

ssize_t
asn_printf (char **as, ssize_t *asz, const char *fmt, ...)
{
    va_list va;
    ssize_t i;

    assert (as);
    assert (asz);
    assert (fmt);
    assert ((*as == NULL) == (*asz == 0));

    va_start (va, fmt);
    i = libretto__vasnprintf (as, asz, fmt, va);
    va_end (va);
    return i;
}

ssize_t
asn_vprintf (char **as, ssize_t *asz, const char *fmt, va_list va)
{
    assert (as);
    assert (asz);
    assert (fmt);
    assert ((*as == NULL) == (*asz == 0));

    return libretto__vasnprintf (as, asz, fmt, va);
}

inline static ssize_t
i_asn_vprinta (char **as, ssize_t *asz, const char *fmt, va_list va)
{
    ssize_t i;

    if (!*as)
	i = libretto__vasnprintf (as, asz, fmt, va);
    else
    {
	char *ias = 0;
	ssize_t iasz = 0;

	i = libretto__vasnprintf (&ias, &iasz, fmt, va);
	if (ias)		/* did the format yield any results? */
	{
	    char *p;
	    ssize_t newlen = strlen (*as) + strlen (ias) + 1;

	    p = mem_realloc (*as, newlen);
	    if (!p)
		i = EOF;
	    else
	    {
		*as = p;
		*asz = newlen;
		strcat (*as, ias);
	    }

	    mem_free (ias);
	}
    }

    return i;
}

ssize_t
asn_printa (char **as, ssize_t *asz, const char *fmt, ...)
{
    va_list va;
    ssize_t i;

    assert (as);
    assert (asz);
    assert (fmt);
    assert ((*as == NULL) == (*asz == 0));

    va_start (va, fmt);
    i = i_asn_vprinta (as, asz, fmt, va);
    va_end (va);

    return i;
}

ssize_t
asn_vprinta (char **as, ssize_t *asz, const char *fmt, va_list va)
{
    assert (as);
    assert (asz);
    assert (fmt);
    assert ((*as == NULL) == (*asz == 0));

    return i_asn_vprinta (as, asz, fmt, va);
}


int
astr_scanf (const Autostr *str, const char *format, ...)
{
    va_list va;
    int i;

    assert (str);
    assert (format);

    va_start (va, format);
    i = libretto__vstrscanf (str, format, va);
    va_end (va);
    return i;
}

int
astr_vscanf (const Autostr *str, const char *format, va_list va)
{
    assert (str);
    assert (format);

    return libretto__vstrscanf (str, format, va);
}

int
file_scanf (FILE *stream, const char *fmt, ...)
{
    va_list va;
    ssize_t i;

    assert (stream);
    assert (fmt);

    va_start (va, fmt);
    i = libretto__vfscanf (stream, fmt, va);
    va_end (va);
    return i;
}

int
file_vscanf (FILE *stream, const char *fmt, va_list va)
{
    assert (stream);
    assert (fmt);

    return libretto__vfscanf (stream, fmt, va);
}

int
s_scanf (const char *s, const char *fmt, ...)
{
    va_list (va);
    int i;

    assert (s);
    assert (fmt);

    va_start (va, fmt);
    i = libretto__vsscanf (s, fmt, va);
    va_end (va);
    return i;
}

int
s_vscanf (const char *s, const char *fmt, va_list va)
{
    assert (s);
    assert (fmt);

    return libretto__vsscanf (s, fmt, va);
}
