/* file.c -- implementation for file funcs in libibretto
 *
 * Aaron Crane <aaronc@pobox.com>
 * 8 July 1996
 * 30 April 1997
 * 21 February 1998 -- Great Renaming.
 *
 * This file is part of Libretto, a library of useful functions.
 *
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>.
 * Portions of this file are Copyright  1991, 1992 Free Software
 * Foundation, Inc.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <libretto/libretto.h>

#include <stdlib.h>
#include <errno.h>
#include <assert.h>

/* add_chunkily -- add char C to the reallocing buffer in *P (size *N) at OFFSET
 *
 * OFFSET should be the number of chars already written into *P.
 */
#define CHUNK 256
inline static int
add_chunkily (char **buf, ssize_t *n, ssize_t offset, int c)
{
    char *p;

    while (offset >= *n - 1)	/* extra 1 to always keep room for null */
    {
	p = mem_realloc (*buf, *n + CHUNK);
	if (!p)
	    return -1;
	*n += CHUNK;
	*buf = p;
    }
    buf[0][offset] = (char) c;
    return 0;
}

inline static ssize_t
init_buffer (char **buf, ssize_t *n)
{
    ssize_t size;
    char *p;

    if (!*buf)
    {
	size = CHUNK;
	p = mem_alloc (size);
	if (!p)
	{
	    size = 8;
	    p = mem_alloc (size); /* `minimum' allocatable block */
	    if (!p)
	    {
		errno = ENOMEM;
		return EOF;
	    }
	}
	*buf = p;
	*n = size;
    }

    return 0;
}

inline static ssize_t
terminate (char **buf, ssize_t *n, ssize_t offset)
{
    assert (offset < *n - 1);
    buf[0][offset] = 0;
    return offset;
}

/* f_getdelim -- read into *BUF (size *N) until DELIM found
 *
 * Read up to but not including DELIM from STREAM into *BUF (and null-terminate
 * it).  *BUF is a pointer returned from malloc() (or NULL), pointing to *N
 * chars of space; it will be re-alloced as necessary.  Null characters in
 * STREAM are silently dropped.  The terminating DELIM is *not* included in the
 * buffer.  Returns the number of characters read (not including the '\0'
 * terminator), or a negative number on error.  Does not return if a realloc()
 * fails; instead it uses mem_realloc() and the behaviour of that function.
 *
 * If the file does not contain a full record (that is, if EOF or error is
 * encountered immediately after a char other than DELIM), then the net effect
 * is to silently `add' an extra DELIM to the stream.  The EOF will then be
 * encountered and handled on the user's next call of the function.
 */
inline static ssize_t
f_getdelim (FILE *stream, char **buf, ssize_t *n, int idelim)
{
    ssize_t offset = 0;
    int c;
    char delim = (char) idelim;

    /* initialise buffer */
    if (init_buffer (buf, n) == EOF)
	return EOF;

    for (;;)
    {
	c = getc (stream);

	if (c == delim)
	    /* end of record: null-terminate buffer, and return length of buffer */
	    return terminate (buf, n, offset);

	if (c == 0)		/* silently drop NUL chars */
	    continue;

	if (c == EOF)
	{
	    if (offset == 0)
	    {
		/* If this is the first char read on this call, then return
		 * EOF, and we haven't changed *BUF if it was non-null. */
		return EOF;
	    }
	    /* `Unexpected' EOF: null-terminate buffer, and return length */
	    return terminate (buf, n, offset);
	}
	
	/* A normal character: add it to BUF. */
	add_chunkily (buf, n, offset, c);
	offset++;
    }
}

ssize_t
file_getdelim (FILE *stream, char **buf, ssize_t *n, int delim)
{
    assert (stream);
    assert (buf);
    assert (n);
    assert (*n >= 0);
    assert ((*buf == NULL) == (*n == 0));

    return f_getdelim (stream, buf, n, delim);
}

ssize_t
file_getline (FILE *stream, char **buf, ssize_t *n)
{
    assert (stream);
    assert (buf);
    assert (n);
    assert (*n >= 0);
    assert ((*buf == NULL) == (*n == 0));

    return f_getdelim (stream, buf, n, '\n');
}
