/* darray.c -- implementation for dynamic array functions in libretto
 *
 * Aaron Crane <aaronc@pobox.com>
 * 26 May 1997
 * 27 July 1997
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Changes:
 *
 * 27 Jul 1997 <aaronc@pobox.com>
 *   Major rewrite: this time it works and is easier to use.
 *   Also put the dstack stuff here; the implementation is similar enough to
 *    reuse it.
 *
 * 12 Aug 1997 <aaronc@pobox.com>
 *   Removed the dstack type; now the functions merely work on darrays.
 */

#include <config.h>
#include <libretto/libretto.h>
#include <libretto/darray.h>
#include <structs.h>

#include <assert.h>
#include <stdlib.h>
#include <string.h>


/* must be a power of two; see smallest_chunk() */
#define DA_CHUNK	(32u)

/*
 * Some inlined helper functions.
 *
 * All of them assume that the array is non-NULL and valid.  In general, they
 * tend not to perform sanity checks on what they do, so be careful.
 */

inline static void *
aref (const Darray *da, size_t index)
{
    return da->data + (index * da->obj_size);
}

inline static void *
atop (const Darray *da)
{
    return aref (da, da->length - 1);
}

inline static void
aset (Darray *da, size_t index, const void *node)
{
    memcpy (aref (da, index), node, da->obj_size);
}

inline static void
aswap (Darray *da, size_t x, size_t y)
{
    if (x != y)			/* cheap performance hack */
    {
	if (da->length < da->alloced) /* a spare node; use one as a buffer */
	{
	    void *p = aref (da, x);
	    void *q = aref (da, y);
	    void *t = aref (da, da->length);

	    aset (da, da->length, p);
	    aset (da, x, q);
	    aset (da, y, t);
	}
	else			/* no spare nodes; do it slowly */
	{
	    unsigned char t, *p, *q, *endp;

	    p = aref (da, x);
	    q = aref (da, y);
	    endp = p + da->obj_size;
	    while (p < endp)
	    {
		t = *p;
		*p++ = *q;
		*q++ = t;
	    }
	}
    }
}

/* WARNING: this function does not expect to be passed a valid darray.  Use
 * it only if you know exactly what you are doing.  Note that arealloc()
 * will only reallocate DA's data if the reallocation succeeds; otherwise,
 * it leaves it alone. */
inline static int
arealloc (Darray *da, ssize_t new_alloced)
{
    if (new_alloced == 0)
    {
	da->alloced = 0;
	mem_free (da->data);
	da->data = 0;
	return 0;
    }
	
    if (da->alloced != new_alloced)
    {
	void *p;

	p = mem_realloc (da->data, new_alloced * da->obj_size);
	if (!p)
	    return -1;

	da->alloced = new_alloced;
	da->data = p;
    }

    return 0;
}

inline static ssize_t
smallest_chunk (ssize_t len)
{
    len += DA_CHUNK - 1;
    len &= ~ (DA_CHUNK - 1);
    return len;
}

inline static void
try_shrink (Darray *da)
{
    if (da->alloced - da->length >= (ssize_t) DA_CHUNK)
    {
	ssize_t new_alloced = smallest_chunk (da->length) * da->obj_size;
	void *buf = da->data;

	/* This is an attempted shrinkage, so it doesn't matter if the
	 * reallocation fails. */
	if (mem_try_realloc (&buf, new_alloced) != -1)
	{
	    da->data = buf;
	    da->alloced = new_alloced;
	}
    }
}

inline static void
adelete (Darray *da, ssize_t index)
{
    da->length--;
    if (da->length != index)
	memmove (aref (da, index), aref (da, index + 1),
		 (da->length - index) * da->obj_size);
}

/* Leaves it consistent on reallocation (and ensures that DA is unchanged if
 * allocation failed */
inline static int
aresize (Darray *da, ssize_t newlen)
{
    ssize_t oldlen;

    oldlen = da->length;
    da->length = newlen;

    if (newlen > oldlen)	/* did it grow? */
    {
	if (newlen >= da->alloced)
	{
	    int i = arealloc (da, smallest_chunk (newlen));
	    if (i == -1)
		da->length = oldlen;
	    return i;
	}
    }
    else if (newlen < oldlen)	/* did it shrink? */
	try_shrink (da);
    /* else newlen == oldlen, so no reallocating */

    return 0;
}

inline static ssize_t
apush (Darray *da, const void *node)
{
    ssize_t oldlen;
    int i;

    oldlen = da->length;
    i = aresize (da, oldlen + 1);
    if (i == -1)
	return -1;
    else
    {
	aset (da, oldlen, node);
	return oldlen;
    }
}

inline static int
acmp (Darray *da, ssize_t x, ssize_t y, Da_compare_f cmp, void *cmp_args)
{
    return cmp (aref (da, x), aref (da, y), cmp_args);
}

inline static void
initialise (Darray *da, size_t obj_size)
{
    da->obj_size = obj_size;
    da->length = 0;
    da->data = 0;
    da->alloced = 0;
}

inline static void
finalise (Darray *da)
{
    mem_free (da->data);
}

inline static int
avalidp (const Darray *da)
{
    if (da->length > da->alloced)
	return 0;
    if (da->length > 0 && !da->data)
	return 0;
    /* There is no good way to check that all the memory that DA claims to
     * be allocated is in fact mapped and accessible to the running
     * process. */
    return 1;
}

inline static int
acat_array (Darray *dest, const Darray *src)
{
    ssize_t dest_oldlen;
    int i;

    dest_oldlen = dest->length;
    i = aresize (dest, dest_oldlen + src->length);
    if (i == 0)
	memcpy (aref (dest, dest_oldlen), src->data, src->obj_size * src->length);
    return i;
}

/*
 * Stack functions
 */

int
da_push (Darray *da, const void *elt)
{
    assert (da);
    assert (elt);

    return (apush (da, elt) == -1) ? -1 : 0;
}

void
da_drop (Darray *da)
{
    assert (da);
    assert (da->length >= 1);

    aresize (da, da->length - 1); /* no failure on shrink */
}

void
da_pop (Darray *da, void *elt)
{
    assert (da);
    assert (da->length >= 1);
    assert (elt);

    memcpy (elt, aref (da, da->length - 1), da->obj_size);
    aresize (da, da->length - 1); /* no failure on shrink */
}

void *
da_top (const Darray *da)
{
    assert (da);
    assert (da->length >= 1);

    return atop (da);
}

void
da_swap (Darray *da)
{
    assert (da);
    assert (da->length >= 2);

    aswap (da, da->length - 1, da->length - 2);
}

int
da_dup (Darray *da)
{
    assert (da);
    assert (da->length >= 1);

    return (apush (da, atop (da)) == -1) ? -1 : 0;
}

int
da_empty (const Darray *da)
{
    assert (da);

    return da->length < 1;
}

/*
 * Darray implementation
 */

int da_valid_p (const Darray *da)
{
    assert (da);

    return avalidp (da);
}

void
da_destroy (Darray *da)
{
    assert (da);

    finalise (da);
    mem_free (da);
}

Darray *
da_create (size_t obj_size)
{
    Darray *da;

    da = mem_alloc (sizeof (*da));
    if (da)
	initialise (da, obj_size);
    return da;
}

Darray *
da_create_len (size_t obj_size, ssize_t len)
{
    Darray *da;

    assert (len >= 0);

    da = mem_alloc (sizeof (*da));
    if (da)
    {
	initialise (da, obj_size);
	if (aresize (da, len) == -1)
	{
	    finalise (da);
	    da = 0;
	}
    }
    return da;
}

ssize_t
da_length (const Darray *da)
{
    assert (da);

    return da->length;
}

size_t
da_object_size (const Darray *da)
{
    assert (da);

    return da->obj_size;
}

void *
da_ref (const Darray *da, ssize_t index)
{
    assert (da);
    assert (index >= 0);
    assert (index < da->length);

    return aref (da, index);
}

void
da_set (Darray *da, ssize_t index, const void *new)
{
    assert (da);
    assert (new);
    assert (index >= 0);
    assert (index < da->length);

    aset (da, index, new);
}

void *
da_last (const Darray *da)
{
    assert (da);
    assert (da->length > 0);

    return atop (da);
}

void
da_exchange (Darray *da, ssize_t x, ssize_t y)
{
    assert (da);
    assert (x >= 0);
    assert (x < da->length);
    assert (y >= 0);
    assert (y < da->length);

    aswap (da, x, y);
}

/* Adds a new NODE to end of DA, incrementing DA's length.  Returns index of
 * new node. */
ssize_t
da_append (Darray *da, const void *node)
{
    assert (da);
    assert (node);

    return apush (da, node);
}

int
da_resize (Darray *da, ssize_t newlen)
{
    assert (da);
    assert (newlen >= 0);

    return aresize (da, newlen);
}

int
da_equal (const Darray *left, const Darray *right, Da_compare_f cmp, void *cmp_args)
{
    ssize_t i;

    assert (left);
    assert (right);

    if (left->obj_size != right->obj_size)
	return 0;		/* must have same-sized objects */

    if (left->length != right->length)
	return 0;		/* must have same length */

    if (cmp)
	for (i = 0;  i < left->length;  i++)
	    if (cmp (aref (left, i), aref (right, i), cmp_args) != 0)
		return 0;

    return 1;
}

int
da_insert (Darray *da, ssize_t index, const void *node)
{
    assert (da);
    assert (node);
    assert (index >= 0);
    /* It's permissible to call with INDEX == DA->length.  In this case, the
     * function degenerates into a call to da_append() */
    assert (index <= da->length);

    if (index == da->length)
	return (apush (da, node) == -1) ? -1 : 0;
    else
    {
	ssize_t oldlen;

	oldlen = da->length;
	if (aresize (da, oldlen + 1) == -1)
	    return -1;
	else
	{
	    /* There's now an uninitialised object at aref(da, oldlen) */
	    memmove (aref (da, index + 1), aref (da, index),
		     (oldlen - index) * da->obj_size);
	    aset (da, index, node);
	    return 0;
	}
    }
}

void
da_delete (Darray *da, ssize_t index)
{
    assert (da);
    assert (index >= 0);
    assert (index < da->length);

    adelete (da, index);
    try_shrink (da);
}

int
da_concat_da (Darray *dest, const Darray *src)
{
    assert (dest);
    assert (src);
    assert (dest->obj_size == src->obj_size);

    return acat_array (dest, src);
}

int
da_insert_da (Darray *dest, const Darray *src, ssize_t index)
{
    size_t n;
    int i;

    assert (dest);
    assert (src);
    assert (dest->obj_size == src->obj_size);
    assert (index >= 0);
    assert (index <= dest->length);

    if (index == dest->length)
	return acat_array (dest, src);

    n = dest->length - index;
    i = aresize (dest, dest->length + src->length);
    if (i == 0)
    {
	memmove (aref (dest, dest->length - n), aref (dest, index), n * dest->obj_size);
	memcpy (aref (dest, index), aref (src, 0), src->length * src->obj_size);
    }
    return i;
}

void
da_prune (Darray *da, Da_prune_f prunable, void *args)
{
    ssize_t i;

    assert (da);
    assert (prunable);

    for (i = da->length - 1;  i >= 0;  i--)
	if (prunable (aref (da, i), args))
	    adelete (da, i);
    try_shrink (da);
}

int
da_apply (const Darray *da, Da_apply_f func, void *args)
{
    int ret = 0;
    ssize_t i;

    for (i = 0;  i < da->length;  i++)
    {
	ret = func (aref (da, i), args);
	if (ret == -1)
	    return -1;
    }

    return ret;
}

ssize_t
da_find (const Darray *da, ssize_t offset, const void *target, Da_compare_f cmp,
	 void *cmp_args)
{
    ssize_t i;

    assert (da);
    assert (cmp);
    /* don't assert TARGET, because (*CMP)() might handle that case */

    offset++;			/* make OFFSET the first posn to look at */
    if (offset < 0)
	offset = 0;

    assert (offset <= da->length); /* <= 'cos we've incremented it */

    for (i = offset;  i < da->length;  i++)
	if (cmp (aref (da, i), target, cmp_args) == 0)
	    return i;

    return -1;
}

ssize_t
da_rfind (const Darray *da, ssize_t last, const void *target, Da_compare_f cmp,
	  void *cmp_args)
{
    ssize_t i;

    assert (da);
    assert (cmp);
    /* don't assert TARGET as CMP may handle that situation */
    assert (last < da->length);

    if (last < 0)
	last = da->length;

    for (i = last - 1;  i >= 0;  i--)
    {
	if (cmp (aref (da, i), target, cmp_args) == 0)
	    return i;
    }

    return -1;
}

inline static ssize_t
bin_search (const Darray *da, ssize_t lower, ssize_t upper,
	    const void *target, Da_compare_f cmp, void *cmp_args)
{
    ssize_t mid = 0;
    int comparison = 0;

    while (lower <= upper)
    {
	mid = (lower + upper) / 2;
	comparison = cmp (target, aref (da, mid), cmp_args);
	if (comparison == 0)
	    return mid;
	if (comparison < 0)
	    upper = mid - 1;
	else			/* if (comparison > 0) */
	    lower = mid + 1;
    }

    /* This looks bizarre, but I promise it works. */
    return -(mid + 1 + (comparison > 0));
}

ssize_t
da_bsearch (const Darray *da, const void *target, Da_compare_f cmp, void *cmp_args)
{
    ssize_t i;

    assert (da);
    assert (cmp);
    /* don't assert that TARGET is non-NULL, because (*CMP)() might handle that
     * case */

    i = bin_search (da, 0, da->length - 1, target, cmp, cmp_args);
    return i < 0 ? -1 : i;
}

ssize_t
da_bsearch_lim (const Darray *da, ssize_t lower, ssize_t upper,
		const void *target, Da_compare_f cmp, void *cmp_args)
{
    ssize_t i;

    assert (da);
    assert (cmp);
    assert (lower <= upper);
    assert (lower >= 0);
    assert (upper < da->length);

    i = bin_search (da, lower, upper, target, cmp, cmp_args);
    return i < 0 ? -1 : i;
}

ssize_t
da_bsearch_index (const Darray *da, const void *target, Da_compare_f cmp, void *cmp_args)
{
    assert (da);
    assert (cmp);

    return bin_search (da, 0, da->length - 1, target, cmp, cmp_args);
}

#if 0
inline static ssize_t
find_pivot (void *array[], ssize_t lo, ssize_t hi, Da_compare_f cmp, void *cmp_args)
{
    ssize_t diff = hi - lo, mid = lo + (diff >> 1);

    if (diff < 6)
	return mid;
    else
    {
	/* Find median of three elts.  This is done in three comparisons,
	 * the theoretical minimum.  Of course, it's not very readable,
	 * understandable or maintainable, but it *is* efficient. */
	int cmp_lm, cmp_lh, cmp_mh;
	cmp_lm = acmp (array, lo, mid, cmp, cmp_args);
	cmp_lh = acmp (array, lo, hi, cmp, cmp_args);
	cmp_mh = acmp (array, mid, hi, cmp, cmp_args);

	if (cmp_lh < 0)
	{
	    if (cmp_lm > 0)
		return lo;
	    if (cmp_mh > 0)
		return hi;
	    return mid;
	}
	if (cmp_lm < 0)
	    return lo;
	if (cmp_mh < 0)
	    return hi;
	return mid;
    }
}
#endif

inline static void
rquick (Darray *da, ssize_t low, ssize_t high, Da_compare_f cmp, void *cmp_args)
{
    ssize_t i, j, pivot;

    /* gcc can't optimise out the tail-recursion, so I do it by hand. */
    while (low < high)
    {
	i = low;
	j = high;

	pivot = high;		/* arbitrary, non-randomised, non-median */
	do
	{
	    while (i < j && acmp (da, i, pivot, cmp, cmp_args) <= 0)
		i++;
	    while (j > i && acmp (da, j, pivot, cmp, cmp_args) >= 0)
		j--;
	    if (i < j)
		aswap (da, i, j);
	} while (i < j);
	aswap (da, i, high);

	if (i - low < high - i)
	{
	    rquick (da, low, i-1, cmp, cmp_args);
	    low = i+1;		/* rquick(da, i+1, high, cmp, cmp_args) */
	}
	else
	{
	    rquick (da, i+1, high, cmp, cmp_args);
	    high = i-1;		/* rquick(da, low, i-1, cmp, cmp_args) */
	}
    }
}

void
da_sort (Darray *da, Da_compare_f cmp, void *cmp_args)
{
    assert (da);
    assert (cmp);

    if (da->length <= 1)	/* cheap performance hack */
	return;

    rquick (da, 0, ((ssize_t) da->length) - 1, cmp, cmp_args);
}

/* Reheaping
 *
 * I use `heap' in the sense of Kernighan and Plauger: a container with the
 * properties that its smallest entry may be found immediately, and a new
 * element can be put into the proper position in the heap in time O(log N)
 * where N is the number of elements.  A heap is conceptually a maximally
 * binary tree where each node is less than or equal to its children.  I
 * represent a heap as an array of elements in which the children of element K
 * are stored at positions 2K+1 and 2K+2.  The smallest element is therefore
 * the one at position 0 (da_ref (DA, 0)).
 *
 * A typical usage pattern might be as follows:
 *
 * - Fill an array DA with elements
 * - da_sort(DA) (since a sorted array is a heap)
 * - Iterate:
 *   - Use the element at da_ref(DA, 0)
 *   - da_set(DA, 0, new)
 *   - da_reheap(DA)
 *
 * So, da_reheap() assumes that DA is a heap (as defined above) except that
 * da_ref(DA, 0) is not necessarily in the right place.  da_reheap() therefore
 * moves da_ref(DA, 0) into the right place, in time O(log2 DA->length).
 */

void
da_reheap (Darray *da, Da_compare_f cmp, void *cmp_args)
{
    ssize_t i, j, max;

    assert (da);
    assert (cmp);

    /* cheap performance hack */
    if (da->length <= 1)
	return;

    max = da->length - 1;
    for (i = 0, j = 2*i + 1;  j <= max;  i = j,  j = 2*i + 1)
    {
	if (j < max)		/* find smaller child */
	    if (acmp (da, j, j+1, cmp, cmp_args) > 0)
		j++;
	if (acmp (da, i, j, cmp, cmp_args) <= 0)
	    break;		/* proper position found */
	else
	    aswap (da, i, j); /* percolate */
    }
}
