/* chain.c -- implementation for `chain' functions in libretto
 *
 * Aaron Crane <aaronc@pobox.com>
 * 9 August 1997
 * 15 August 1997 (after re-reading the relevant portions of Knuth vol. I)
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <libretto/libretto.h>
#include <libretto/chain.h>
#include <structs.h>

#include <assert.h>
#include <stdlib.h>
#include <string.h>

/*
 * Inlined helpers
 */

/* Unfortunately, this won't necessarily even terminate on certain classes
 * of ill-formed chains -- for example, if the links are circular.  It's
 * still better than nothing though -- I'd rather have a buggy program that
 * loops doing nothing than a buggy program that scribbles on potentially
 * important data. */
inline static int
ch_validp (const Chain *chain)
{
    Chainlink *node;

    if (chain->length < 0)
	return 0;

    if (chain->length == 0)
	return (chain->first == NULL) && (chain->last == NULL);

    /* The chain has nodes, so check the link structure */

    if (!chain->first || !chain->last)
	return 0;

    for (node = chain->first;  node;  node = node->next)
	if (!node->next && node != chain->last)
	    return 0;

    for (node = chain->last;   node;  node = node->prev)
	if (!node->prev && node != chain->first)
	    return 0;

    return 1;
}

inline static void
ch_init (Chain *p)
{
    p->length = 0;
    p->first = 0;
    p->last = 0;
}

inline static Chainlink *
chlink_create (void)
{
    return mem_alloc (sizeof (Chainlink));
}

inline static void
chlink_destroy (Chainlink *link)
{
    mem_free (link);
}

inline static void
ch_final (Chain *p)
{
    Chainlink *curr, *temp;

    for (curr = p->first;  curr;  curr = temp)
    {
	temp = curr->next;
	chlink_destroy (curr);
    }
}

inline static Chainlink *
insert_at_front (Chain *chain, void *data)
{
    Chainlink *link;

    link = chlink_create ();
    if (link)
    {
	link->data = data;
	link->next = chain->first;
	link->prev = 0;
	if (chain->first)	/* any nodes at all so far? */
	    chain->first->prev = link;
	else
	    chain->last = link;
	chain->first = link;
	chain->length++;
    }

    return link;
}

inline static Chainlink *
insert_at_end (Chain *chain, void *data)
{
    Chainlink *link;

    link = chlink_create ();
    if (link)
    {
	link->data = data;
	link->next = 0;
	link->prev = chain->last;
	if (chain->first)	/* any nodes at all so far? */
	    chain->last->next = link;
	else
	    chain->first = link;
	chain->last = link;
	chain->length++;
    }

    return link;
}

inline static void *
remove_at_front (Chain *chain)
{
    Chainlink *link;
    void *data;

    link = chain->first;
    chain->first = link->next;
    if (!chain->first)		/* was LINK the only link? */
	chain->last = 0;
    else
	chain->first->prev = 0;
    chain->length--;
    data = link->data;
    chlink_destroy (link);
    return data;
}

inline static void *
remove_at_end (Chain *chain)
{
    Chainlink *link;
    void *data;

    link = chain->last;
    chain->last = link->prev;
    if (!chain->last)		/* was LINK the only link? */
	chain->first = 0;
    else
	chain->last->next = 0;
    chain->length--;
    data = link->data;
    chlink_destroy (link);
    return data;
}

inline static Chainlink *
chlink_nth (const Chain *chain, ssize_t n)
{
    Chainlink *link;

    /* decide whether to scan forwards or backwards */
    if (n <= chain->length / 2)
	/* forwards */
	for (link = chain->first;  n > 0;  n--)
	    link = link->next;
    else
    {
	/* backwards */
	n = chain->length - n - 1;
	for (link = chain->last;  n > 0;  n--)
	    link = link->prev;
    }

    return link;
}

inline static Chainlink *
insert_between (Chain *chain, void *data, Chainlink *prev, Chainlink *curr)
{
    Chainlink *new;

    new = chlink_create ();
    if (new)
    {
	new->data = data;
	new->next = curr;
	new->prev = prev;

	if (!curr)		/* inserted at end? */
	    chain->last = new;
	else			/* inserted at front or in middle */
	    curr->prev = new;

	if (!prev)		/* inserted at front? */
	    chain->first = new;
	else			/* inserted at end or in middle */
	    prev->next = new;

	chain->length++;
    }

    return new;
}

inline static void
remove_link (Chain *chain, Chainlink *link)
{
    if (link->prev)
	link->prev->next = link->next;
    else
	chain->first = link->next;

    if (link->next)
	link->next->prev = link->prev;
    else
	chain->last = link->prev;

    chain->length--;
    chlink_destroy (link);
}

/*
 * Public functions
 */

int
chain_valid_p (const Chain *chain)
{
    assert (chain);

    return ch_validp (chain);
}

Chain *
chain_create (void)
{
    Chain *p;

    p = mem_alloc (sizeof (*p));
    if (p)
	ch_init (p);
    return p;
}

void
chain_destroy (Chain *chain)
{
    assert (chain);

    ch_final (chain);
    mem_free (chain);
}

ssize_t
chain_length (const Chain *chain)
{
    assert (chain);

    return chain->length;
}

void *chainlink_data (const Chainlink *node)
{
    assert (node);

    return node->data;
}

void
chain_walk (const Chain *chain, Chain_walk_f walker, void *params)
{
    Chainlink *curr;

    assert (chain);
    assert (walker);

    for (curr = chain->first;  curr;  curr = curr->next)
	if (walker (curr->data, params) < 0)
	    return;
}

void
chain_rwalk (const Chain *chain, Chain_walk_f walker, void *params)
{
    Chainlink *curr;

    assert (chain);
    assert (walker);

    for (curr = chain->last;  curr;  curr = curr->prev)
	if (walker (curr->data, params) < 0)
	    return;
}

Chainlink *
chain_front_insert (Chain *chain, void *data)
{
    assert (chain);

    return insert_at_front (chain, data);
}

Chainlink *
chain_back_insert (Chain *chain, void *data)
{
    assert (chain);

    return insert_at_end (chain, data);
}

void *
chain_front_remove (Chain *chain)
{
    assert (chain);
    assert (chain->first);	/* something to remove? */

    return remove_at_front (chain);
}

void *
chain_back_remove (Chain *chain)
{
    assert (chain);
    assert (chain->last);	/* something to remove? */

    return remove_at_end (chain);
}

Chainlink *
chain_enqueue (Chain *chain, void *data)
{
    assert (chain);

    return insert_at_end (chain, data);
}

Chainlink *
chain_append (Chain *chain, void *data)
{
    assert (chain);

    return insert_at_end (chain, data);
}

Chainlink *
chain_push (Chain *chain, void *data)
{
    assert (chain);

    return insert_at_front (chain, data);
}

void *
chain_dequeue (Chain *chain)
{
    assert (chain);
    assert (chain->last);	/* something to remove? */

    return remove_at_front (chain);
}

void *
chain_pop (Chain *chain)
{
    assert (chain);
    assert (chain->first);	/* something to remove? */

    return remove_at_front (chain);
}

void
chain_drop (Chain *chain)
{
    assert (chain);
    assert (chain->first);	/* something to remove? */

    remove_at_front (chain);
}

Chainlink *
chain_first (const Chain *chain)
{
    assert (chain);

    return chain->first;
}

Chainlink *
chain_front (const Chain *chain)
{
    assert (chain);

    return chain->first;
}

Chainlink *
chain_top (const Chain *chain)
{
    assert (chain);

    return chain->first;
}

Chainlink *
chain_last (const Chain *chain)
{
    assert (chain);

    return chain->last;
}

Chainlink *
chain_rear (const Chain *chain)
{
    assert (chain);

    return chain->last;
}

Chainlink *
chain_nth (const Chain *chain, ssize_t n)
{
    Chainlink *link;

    assert (chain);
    assert (n >= 0);
    assert (n < chain->length);

    link = chlink_nth (chain, n);
    return link;
}

Chainlink *
chain_pred (const Chain *chain, const Chainlink *node)
{
    assert (chain);
    assert (node);
    /* would take too long to verify that NODE is in CHAIN */

    return node->prev;
}

Chainlink *
chain_succ (const Chain *chain, const Chainlink *node)
{
    assert (chain);
    assert (node);
    /* would take too long to verify that NODE is in CHAIN */

    return node->next;
}

int
chain_member (const Chain *chain, const Chainlink *node)
{
    Chainlink *p;

    assert (chain);
    assert (node);

    for (p = chain->first;  p;  p = p->next)
	if (p == node)
	    return 1;

    return 0;
}

Chainlink *
chain_cpred (const Chain *chain, const Chainlink *node)
{
    assert (chain);
    assert (node);
    /* would take too long to verify that NODE is in CHAIN */

    return node->prev ? node->prev : chain->last;
}

Chainlink *
chain_csucc (const Chain *chain, const Chainlink *node)
{
    assert (chain);
    assert (node);
    /* would take too long to verify that NODE is in CHAIN */

    return node->next ? node->next : chain->first;
}

Chainlink *
chain_insert (Chain *chain, void *data, Chain_compare_f compare, void *params)
{
    Chainlink *curr;
    Chainlink *prev;

    assert (chain);
    assert (compare);

    for (prev = 0, curr = chain->first;  curr;  prev = curr, curr = curr->next)
	if (compare (curr->data, data, params) > 0)
	    break;

    return insert_between (chain, data, prev, curr);
}

void
chain_move_front (Chain *chain, Chainlink *node)
{
    assert (chain);
    assert (node);
    /* would take too long to verify that NODE is in CHAIN */

    if (node->prev)		/* not already at front */
    {
	node->prev->next = node->next;
	if (node->next)
	    node->next->prev = node->prev;
	else
	    chain->last = node->prev;
	node->prev = 0;
	node->next = chain->first;
	chain->first = node;
    }
}

void
chain_move_back (Chain *chain, Chainlink *node)
{
    assert (chain);
    assert (node);
    /* would take too long to verify that NODE is in CHAIN */

    if (node->next)		/* not already at back */
    {
	node->next->prev = node->prev;
	if (node->prev)
	    node->prev->next = node->next;
	else
	    chain->first = node->next;
	node->next = 0;
	node->prev = chain->last;
	chain->last = node;
    }
}

void
chain_remove (Chain *chain, Chainlink *node)
{
    assert (chain);
    assert (node);
    /* would take too long to verify that NODE is in CHAIN */

    remove_link (chain, node);
}

Chainlink *
chain_find (const Chain *chain, const Chainlink *last_found, const void *target,
	    Chain_compare_f cmp, void *params)
{
    Chainlink *curr, *start, *last;

    assert (chain);
    assert (cmp);
    /* Too slow to check that LAST_FOUND (if non-null) is in CHAIN. */

    last = (Chainlink *) last_found; /* cast away const */
    start = last ? last : chain->first;

    for (curr = start;  curr;  curr = curr->next)
	if (cmp (curr->data, target, params) == 0)
	    break;

    /* CURR is either node with matching data, or 0 */
    return curr;
}

Chainlink *
chain_rfind (const Chain *chain, const Chainlink *last_found, const void *target,
	     Chain_compare_f cmp, void *params)
{
    Chainlink *curr, *start, *last;

    assert (chain);
    assert (cmp);
    /* Too slow to check that LAST_FOUND (if non-null) is in CHAIN. */

    last = (Chainlink *) last_found; /* cast away const */
    start = last ? last : chain->last;

    for (curr = start;  curr;  curr = curr->prev)
	if (cmp (curr->data, target, params) == 0)
	    break;

    /* CURR is either node with matching data, or 0 */
    return curr;
}

void
chain_wipe (Chain *chain)
{
    Chainlink *link, *temp;

    assert (chain);

    for (link = chain->first;  link;  link = temp)
    {
	temp = link->next;
	chlink_destroy (link);
    }

    chain->first = 0;
    chain->last = 0;
    chain->length = 0;
}

void
chain_prune (Chain *chain, Chain_prune_f prunable, void *params)
{
    Chainlink *link, *temp;

    assert (chain);
    assert (prunable);

    for (link = chain->first;  link;  link = temp)
    {
	temp = link->next;
	if (prunable (link->data, params))
	    remove_link (chain, link);
    }
}

int
chain_concat_chain (Chain *dest, const Chain *src)
{
    Chainlink *link;

    assert (dest);
    assert (src);

    for (link = src->first;  link;  link = link->next)
	if (insert_at_end (dest, link->data) == 0)
	    return -1;

    return 0;
}

void
chain_reverse (Chain *chain)
{
    Chainlink *link;
    Chainlink *t;

    assert (chain);

    for (link = chain->first;  link;  link = t)	/* T is pre-reversal copy of LINK->next */
    {
	t = link->next;
	link->next = link->prev;
	link->prev = t;
    }

    t = chain->first;
    chain->first = chain->last;
    chain->last = t;
}
