/* bstree.c -- implementation for binary search tree functions in libretto
 *
 * Aaron Crane <aaronc@pobox.com>
 * 16 August 1997
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <libretto/libretto.h>
#include <libretto/bstree.h>
#include <structs.h>

#include <assert.h>
#include <stdlib.h>
#include <string.h>

/*
 * Inlined helpers
 */

inline static Bstnode *
node_create (void *data)
{
    Bstnode *node;

    node = mem_alloc (sizeof (*node));
    if (node)
    {
	node->data = data;
	node->left = 0;
	node->right = 0;
	/* We don't set NODE->parent here, 'cos we'd have to make it 0, and
	 * the caller is about to set it to the correct value. */
    }
    return node;
}

inline static void
node_destroy (Bstnode *node)
{
    mem_free (node);
}

inline static void
initialise (Bstree *bst, Bst_compare_f cmp)
{
    bst->root = 0;
    bst->cmp = cmp;
}

/* WARNING: never call with NODE == NULL */
inline static void
finalise (Bstnode *node)
{
    if (node->left)
	finalise (node->left);
    if (node->right)
	finalise (node->right);
    node_destroy (node);
}

inline static ssize_t
count_nodes (const Bstnode *root)
{
    ssize_t n;

    n = 1;			/* this root node */
    if (root->left)
	n += count_nodes (root->left);
    if (root->right)
	n += count_nodes (root->right);
    return n;
}

inline static ssize_t
ssmax (ssize_t x, ssize_t y)
{
    return (x > y) ? x : y;
}

inline static ssize_t
depth_ping (const Bstnode *root)
{
    ssize_t ldepth, rdepth;

    ldepth = root->left ? depth_ping (root->left) : 0;
    rdepth = root->right ? depth_ping (root->right) : 0;
    return ssmax (ldepth, rdepth) + 1; /* + 1 for this node */
}

inline static Bstnode *
smallest (Bstnode *root)
{
    Bstnode *node;

    node = root;
    if (!node)
	return 0;
    while (node->left)
	node = node->left;
    return node;
}

inline static Bstnode *
largest (Bstnode *root)
{
    Bstnode *node;

    node = root;
    if (!node)
	return 0;
    while (node->right)
	node = node->right;
    return node;
}

inline static Bstnode *
successor (const Bstnode *node)
{
    Bstnode *parent;

    if (node->right)
	return smallest (node->right);

    /* go up until we followed a left link to get here */
    parent = node->parent;
    while (parent && parent->left != node)
    {
	node = parent;
	parent = node->parent;
    }

    return parent;
}

inline static Bstnode *
predecessor (const Bstnode *node)
{
    Bstnode *parent;

    if (node->left)
	return largest (node->left);

    /* go up until we followed a right link to get here */
    parent = node->parent;
    while (parent && parent->right != node)
    {
	node = parent;
	parent = node->parent;
    }

    return parent;
}

inline static void
inorder_walk (const Bstnode *node, Bst_walk_f walker, void *walk_args)
{
    /* Unfortunately, gcc can't eliminate the tail-recursion here, so I have
     * to do it by hand. */
    while (node)
    {
	inorder_walk (node->left, walker, walk_args);
	walker (node->data, walk_args);
	node = node->right;
    }
}

inline static void
preorder_walk (const Bstnode *node, Bst_walk_f walker, void *walk_args)
{
    /* Unfortunately, gcc can't eliminate the tail-recursion here, so I have
     * to do it by hand. */
    while (node)
    {
	walker (node->data, walk_args);
	preorder_walk (node->left, walker, walk_args);
	node = node->right;
    }
}

inline static void
postorder_walk (const Bstnode *node, Bst_walk_f walker, void *walk_args)
{
    if (node)
    {
	postorder_walk (node->left, walker, walk_args);
	postorder_walk (node->right, walker, walk_args);
	walker (node->data, walk_args);
    }
}

inline static void
insert_node (Bstree *bst, Bstnode *node, void *cmp_args)
{
    Bstnode *curr;
    Bstnode *prev = 0;
    int cmp = 0;

    for (curr = bst->root;  curr;  curr = (cmp <= 0) ? curr->left : curr->right)
    {
	cmp = bst->cmp (node->data, curr->data, cmp_args);
	prev = curr;
    }

    node->parent = prev;	/* either parent, or NULL for root of tree */

    if (!prev)
	bst->root = node;
    else if (cmp < 0)
	prev->left = node;
    else
	prev->right = node;
}

inline static void
delete_node (Bstree *bst, Bstnode *node, void *cmp_args)
{
    /* start by unlinking the node from its parent */
    if (!node->parent)
	bst->root = 0;
    else
    {
	if (node->parent->left == node)
	    node->parent->left = 0;
	else
	{
	    assert (node->parent->right == node);
	    node->parent->right = 0;
	}
    }

    /* and now link the node's children into the tree */
    if (node->left)
	insert_node (bst, node->left, cmp_args);
    if (node->right)
	insert_node (bst, node->right, cmp_args);

    /* finally, free the node itself */
    node_destroy (node);
}

inline static void
prune (Bstree *bst, Bstnode *root, Bst_prune_f prunable,
       void *prune_args, void *cmp_args)
{
    /* use postorder, since it's easier to delete at the bottom than the top */
    if (root->left)
	prune (bst, root->left, prunable, prune_args, cmp_args);
    if (root->right)
	prune (bst, root->right, prunable, prune_args, cmp_args);
    if (prunable (root->data, prune_args))
	delete_node (bst, root, cmp_args);
}

inline static int
in_tree (const Bstree *bst, const Bstnode *node)
{
    for ( ; node;  node = node->parent)
	if (node == bst->root)
	    return 1;

    return 0;
}

/*
 * Public functions
 */

int
bst_valid_p (const Bstree *bst)
{
    (void) bst;
    return 1;			/* no algorithm for this at the moment */
}

Bstree *
bst_create (Bst_compare_f compare)
{
    Bstree *bst;

    assert (compare);

    bst = mem_alloc (sizeof (*bst));
    if (bst)
	initialise (bst, compare);
    return bst;
}

void
bst_destroy (Bstree *bst)
{
    assert (bst);

    if (bst->root)
	finalise (bst->root);
    mem_free (bst);
}

ssize_t
bst_nodes (const Bstree *bst)
{
    assert (bst);

    return bst->root ? count_nodes (bst->root) : 0;
}

ssize_t
bst_depth (const Bstree *bst)
{
    assert (bst);

    return bst->root ? depth_ping (bst->root) : 0;
}

void *
bstnode_data (const Bstnode *node)
{
    assert (node);

    return node->data;
}

void
bst_walk (const Bstree *bst, Bst_walk_f walker, void *walk_args)
{
    assert (bst);
    assert (walker);

    inorder_walk (bst->root, walker, walk_args);
}

void
bst_walk_pre (const Bstree *bst, Bst_walk_f walker, void *walk_args)
{
    assert (bst);
    assert (walker);

    preorder_walk (bst->root, walker, walk_args);
}

void
bst_walk_post (const Bstree *bst, Bst_walk_f walker, void *walk_args)
{
    assert (bst);
    assert (walker);

    postorder_walk (bst->root, walker, walk_args);
}

Bstnode *
bst_insert (Bstree *bst, void *data, void *cmp_args)
{
    Bstnode *new;

    assert (bst);

    new = node_create (data);
    if (new)
	insert_node (bst, new, cmp_args);
    return new;
}

Bstnode *
bst_lookup (const Bstree *bst, void *target, void *cmp_args)
{
    Bstnode *node;
    int cmp;

    assert (bst);

    node = bst->root;
    while (node)
    {
	cmp = bst->cmp (target, node->data, cmp_args);
	if (cmp < 0)
	    node = node->left;
	else if (cmp > 0)
	    node = node->right;
	else			/* cmp == 0, so found it */
	    break;
    }

    return node;
}

Bstnode *
bst_smallest (const Bstree *bst)
{
    assert (bst);

    return smallest (bst->root);
}

Bstnode *
bst_largest (const Bstree *bst)
{
    assert (bst);

    return largest (bst->root);
}

void
bst_delete (Bstree *bst, Bstnode *node, void *cmp_args)
{
    assert (bst);
    assert (node);
    assert (bst->root);
    assert (in_tree (bst, node));

    delete_node (bst, node, cmp_args);
}

void
bst_wipe (Bstree *bst)
{
    assert (bst);

    if (bst->root)
    {
	finalise (bst->root);
	bst->root = 0;		/* a valid, empty tree */
    }
}

void
bst_prune (Bstree *bst, Bst_prune_f pruner, void *prune_args, void *cmp_args)
{
    assert (bst);
    assert (pruner);

    if (bst->root)
	prune (bst, bst->root, pruner, prune_args, cmp_args);
}

Bstnode *
bst_succ (const Bstree *bst, const Bstnode *node)
{
    assert (bst);
    assert (node);
    assert (in_tree (bst, node));

    return successor (node);
}

Bstnode *
bst_pred (const Bstree *bst, const Bstnode *node)
{
    assert (bst);
    assert (node);
    assert (in_tree (bst, node));

    return predecessor (node);
}
