/* autostr.c -- implementation for dynamic string functions in libretto
 *
 * Aaron Crane <aaronc@pobox.com>
 * 10 August 1997
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 * Portions of this file are Copyright  1991, 1992 Free Software Foundation, Inc.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <libretto/libretto.h>
#include <libretto/autostr.h>
#include <structs.h>
#include <autostr-priv.h>

#include <assert.h>
#include <stdlib.h>

int
astr_valid_p (const Autostr *str)
{
    assert (str);

    return s_validp (str);
}

Autostr *
astr_create (void)
{
    Autostr *str;

    str = mem_alloc (sizeof (*str));
    if (str)
	if  (s_init (str, s_chunkof (1)) == -1
	     && s_init (str, 1) == -1)
	{
	    mem_free (str);
	    str = 0;
	}
    return str;
}

Autostr *
astr_create_s (const char *s)
{
    Autostr *str;
    size_t len;

    assert (s);

    str = mem_alloc (sizeof (*str));
    if (str)
    {
	len = strlen (s);
	if  (s_init (str, s_chunkof (len + 1)) == -1
	     && s_init (str, len + 1) == -1)
	{
	    mem_free (str);
	    str = 0;
	}
	else
	{
	    strcpy (str->s, s);
	    str->length = len;
	}
    }
    
    return str;
}

void
astr_destroy (Autostr *str)
{
    assert (str);

    s_final (str);
    mem_free (str);
}

ssize_t
astr_length (const Autostr *str)
{
    assert (str);

    return str->length;
}

const char *
astr_chars (const Autostr *str)
{
    assert (str);

    return str->s;
}

void
astr_zero (Autostr *str)
{
    assert (str);

    s_zero (str);
}

int
astr_copy (Autostr *dest, const Autostr *src)
{
    assert (dest);
    assert (src);

    return s_copy_s (dest, src->s, src->length);
}

int
astr_copy_s (Autostr *str, const char *s)
{
    assert (str);
    assert (s);

    return s_copy_s (str, s, strlen (s));
}

int
astr_copy_c (Autostr *str, int ch)
{
    char c[2];

    assert (str);
    assert (ch != 0);

    c[0] = ch;
    c[1] = 0;

    return s_copy_s (str, c, 1);
}

int
astr_cat (Autostr *dest, const Autostr *src)
{
    assert (dest);
    assert (src);

    return s_cat_s (dest, src->s, src->length);
}

int
astr_cat_s (Autostr *str, const char *s)
{
    assert (str);
    assert (s);

    return s_cat_s (str, s, strlen (s));
}

int
astr_cat_c (Autostr *str, int ch)
{
    char c[2];

    assert (str);
    assert (ch != 0);

    c[0] = ch;
    c[1] = 0;

    return s_cat_s (str, c, 1);
}

int
astr_insert (Autostr *dest, ssize_t index, const Autostr *src)
{
    assert (dest);
    assert (src);
    assert (index >= 0);
    assert (index <= dest->length);

    return s_insert_s (dest, index, src->s, src->length);
}

int
astr_insert_s (Autostr *str, ssize_t index, const char *s)
{
    assert (str);
    assert (s);
    assert (index >= 0);
    assert (index <= str->length);

    return s_insert_s (str, index, s, strlen (s));
}

int
astr_insert_c (Autostr *str, ssize_t index, int ch)
{
    char c[2];

    assert (str);
    assert (ch != 0);
    assert (index >= 0);
    assert (index <= str->length);

    c[0] = ch;
    c[1] = 0;

    return s_insert_s (str, index, c, 1);
}

void
astr_delete (Autostr *str, ssize_t index, ssize_t n)
{
    char *src, *dest;

    assert (str);
    assert (index < str->length);
    assert (index >= 0);

    if (n == 0)
	return;
    if (n < 0)
	n = str->length - index;

    assert (index + n <= str->length);

    /* I think that this time it actually works...  RHA: You were right, a
     * loop is clearer than a condition and memmove/strncpy.  Especially
     * once you realise that it's not actually necessary to count the
     * characters that you're moving. :)
     */
    dest = str->s + index;
    src = dest + n;
    while ((*dest++ = *src++) != 0)
	;

    str->length -= n;
    s_shrink (str);
}

int
astr_null (const Autostr *astr)
{
    assert (astr);

    return astr->length == 0;
}

int
astr_equal (const Autostr *a, const Autostr *b)
{
    assert (a);
    assert (b);

    return a->length != b->length ? 0 : strcmp (a->s, b->s) == 0;
}

int
astr_iequal (const Autostr *a, const Autostr *b)
{
    assert (a);
    assert (b);

    return a->length != b->length ? 0 : strcasecmp (a->s, b->s) == 0;
}

int
astr_cmp (const Autostr *a, const Autostr *b)
{
    assert (a);
    assert (b);

    return strcmp (a->s, b->s);
}

int
astr_icmp (const Autostr *a, const Autostr *b)
{
    assert (a);
    assert (b);

    return strcasecmp (a->s, b->s);
}

int
astr_equal_s (const Autostr *astr, const char *s)
{
    assert (astr);
    assert (s);

    return strcmp (astr->s, s) == 0;
}

int
astr_iequal_s (const Autostr *astr, const char *s)
{
    assert (astr);
    assert (s);

    return strcasecmp (astr->s, s) == 0;
}

int
astr_cmp_s (const Autostr *astr, const char *s)
{
    assert (astr);
    assert (s);

    return strcmp (astr->s, s);
}

int
astr_icmp_s (const Autostr *astr, const char *s)
{
    assert (astr);
    assert (s);

    return strcasecmp (astr->s, s);
}

ssize_t
astr_find_c (const Autostr *str, ssize_t index, int c)
{
    const char *p;

    assert (str);
    assert (c != 0);

    index++;			/* make index the first posn to look at */
    if (index < 0)
	index = 0;

    assert (index <= str->length); /* <= 'cos we've incremented it */

    p = strchr (str->s + index, c);
    return p ? p - str->s : -1;
}

ssize_t
astr_rfind_c (const Autostr *str, ssize_t last, int c)
{
    const char *p;

    assert (str);
    assert (last < str->length);
    assert (c != 0);

    if (last < 0)
	last = str->length;

    for (p = str->s + last - 1;  p >= str->s;  p--)
	if (*p == (char) c)
	    return p - str->s;

    return -1;
}

ssize_t
astr_find_s (const Autostr *str, ssize_t index, const char *s)
{
    const char *p;

    assert (str);
    assert (s);

    index++;			/* make index the first posn to look at */
    if (index < 0)
	index = 0;

    assert (index < str->length);

    p = strstr (str->s + index, s);
    return p ? p - str->s : -1;
}

ssize_t
astr_find (const Autostr *str, ssize_t index, const Autostr *needle)
{
    const char *p;

    assert (str);
    assert (needle);

    index++;			/* make index the first posn to look at */
    if (index < 0)
	index = 0;

    assert (index < str->length);

    p = strstr (str->s + index, needle->s);
    return p ? p - str->s : -1;
}

int
astr_slice (Autostr *dest, const Autostr *src, ssize_t index, ssize_t n)
{
    assert (dest);
    assert (src);
    assert (index < src->length);
    assert (index >= 0);

    if (n < 0)
	n = src->length - index;

    assert (index + n <= src->length);

    return s_slice (dest, src, index, n);
}

int
astr_slice_i (Autostr *dest, const Autostr *src, ssize_t start, ssize_t end)
{
    ssize_t n;

    assert (dest);
    assert (src);
    assert (start < src->length);
    assert (start >= 0);

    if (end < 0)
	n = src->length - start;
    else
    {
	assert (end <= src->length); /* `<=' because we stop before that index */
	assert (end >= start);
	n = end - start;
    }

    return s_slice (dest, src, start, n);
}
