/* autobuf.c -- implementation for dynamic autobuf functions in libretto
 *
 * Aaron Crane <aaronc@pobox.com>
 * 12 December 1997
 *
 * This file is part of Libretto, a library of useful functions.
 * Libretto is Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <libretto/libretto.h>
#include <libretto/autobuf.h>
#include <structs.h>

#include <assert.h>
#include <string.h>
#include <errno.h>

/* A few implementation notes:
 *
 * An Autobuf with a null .data member is invalid.  On initialise, we
 * allocate at least one chunk (or one byte if the allocation of a chunk
 * failed).  abuf_zero() returns to that state if its realloc succeeds.
 */

/* A short synonym for `unsigned char'.  Give it a silly name so we don't
 * conflict with stupid systems like Solaris. */
typedef unsigned char uchar__t;

/* must be a power of two; see smallest_chunk() */
#define ABUF_CHUNK	(64u)

inline static int
abvalidp (const Autobuf *abuf)
{
    if (abuf->length > abuf->alloced)
	return 0;
    if (!abuf->data)
	return 0;
    /* There is no good way to check that all the memory that ABUF claims to
     * be allocated is in fact mapped and accessible to the running
     * process. */
    return 1;
}

inline static int
abrealloc (Autobuf *abuf, ssize_t new_alloced)
{
    if (new_alloced == 0)
	new_alloced = ABUF_CHUNK;

    if (abuf->alloced != new_alloced)
    {
	unsigned char *p;

	p = mem_realloc (abuf->data, new_alloced);
	if (!p)
	    return -1;

	abuf->alloced = new_alloced;
	abuf->data = p;
    }

    return 0;
}

inline static ssize_t
smallest_chunk (ssize_t len)
{
    len += ABUF_CHUNK - 1;
    len &= ~ (ABUF_CHUNK - 1);
    return len;
}

inline static void
try_shrink (Autobuf *abuf)
{
    if (abuf->alloced - abuf->length >= (ssize_t) ABUF_CHUNK)
    {
	ssize_t new_alloced = smallest_chunk (abuf->length);
	void *buf = abuf->data;

	/* This is an attempted shrinkage, so it's not a problem if the
	 * reallocation fails */
	if (mem_try_realloc (&buf, new_alloced) != -1)
	{
	    abuf->data = buf;
	    abuf->alloced = new_alloced;
	}
    }
}

inline static int
initialise (Autobuf *abuf)
{
    uchar__t *p;
    ssize_t alloced = ABUF_CHUNK;

    p = mem_alloc (alloced);
    if (!p)			/* if out of mem, try a minimal allocation */
    {
	alloced = 1;
	p = mem_alloc (alloced);
    }
    if (!p)
	return -1;
    else
    {
	abuf->data = p;
	abuf->alloced = alloced;
	abuf->length = 0;
    }
    return 0;
}

inline static void
finalise (Autobuf *abuf)
{
    mem_free (abuf->data);
}

inline static void
zero (Autobuf *abuf)
{
    abuf->length = 0;
    abrealloc (abuf, ABUF_CHUNK);	/* not a problem if the realloc fails */
}

inline static ssize_t
ssmax (ssize_t x, ssize_t y)
{
    return x > y ? x : y;
}

inline static int
compare (const Autobuf *x, const Autobuf *y)
{
    int i;

    i = memcmp (x->data, y->data, ssmax (x->length, y->length));
    if (i == 0 && x->length != y->length)
	return x->length - y->length;
    return i;
}

inline static int
addc_offset (Autobuf *abuf, int c, ssize_t offset)
{
    while (offset >= abuf->alloced)
	if (abrealloc (abuf, abuf->alloced + ABUF_CHUNK) == -1)
	    return -1;
    abuf->data[offset] = (uchar__t) c;
    return 0;
}

inline static int
copy_buf (Autobuf *abuf, const void *vbuf, ssize_t len)
{
    const unsigned char *buf = vbuf;

    if (len == 0)
	zero (abuf);
    else
    {
	int i = abrealloc (abuf, smallest_chunk (len));

	if (len > abuf->alloced && i == -1)
	    return -1;
	abuf->length = len;
	memcpy (abuf->data, buf, len);
    }

    return 0;
}

inline static int
cat_buf (Autobuf *abuf, const void *vbuf, ssize_t len)
{
    const unsigned char *buf = vbuf;

    if (len != 0)
    {
	int i = abrealloc (abuf, smallest_chunk (abuf->length + len));

	if (i == -1)
	    return -1;
	memcpy (abuf->data + abuf->length, buf, len);
	abuf->length += len;
    }

    return 0;
}

inline static int
insert_buf (Autobuf *abuf, ssize_t index, const uchar__t *buf, ssize_t len)
{
    if (index == abuf->length)
	return cat_buf (abuf, buf, len);

    if (len != 0)
    {
	ssize_t oldlen;
	int i = abrealloc (abuf, smallest_chunk (abuf->length + len));

	if (i == -1)
	    return -1;

	oldlen = abuf->length;
	abuf->length += len;
	memmove (abuf->data + index + len, abuf->data + index, oldlen - index);
	memcpy (abuf->data + index, buf, len);
    }

    return 0;
}

inline static int
slice (Autobuf *dest, const Autobuf *src, ssize_t index, ssize_t n)
{
    if (src != dest)
    {
	int i = abrealloc (dest, smallest_chunk (n));

	if (n > dest->alloced && i == -1)
	    /* If we tried to grow DEST but failed, then bail out now */
	    return -1;
	dest->length = n;
	memcpy (dest->data, src->data + index, n);
    }
    else			/* src and dest are the same object */
    {
	memmove (dest->data, dest->data + index, n);
	dest->length = n;
	try_shrink (dest);
    }

    return 0;
}

/*
 * Public functions
 */

int
abuf_valid_p (const Autobuf *abuf)
{
    assert (abuf);

    return abvalidp (abuf);
}

Autobuf *
abuf_create (void)
{
    Autobuf *abuf;
    int i;

    abuf = mem_alloc (sizeof (*abuf));
    if (abuf)
    {
	i = initialise (abuf);
	if (i == -1)
	{
	    mem_free (abuf);
	    abuf = 0;
	}
    }
    return abuf;
}

void
abuf_destroy (Autobuf *abuf)
{
    assert (abuf);

    finalise (abuf);
    mem_free (abuf);
}

uchar__t *
abuf_data (const Autobuf *abuf)
{
    assert (abuf);

    return abuf->data;
}

ssize_t
abuf_length (const Autobuf *abuf)
{
    assert (abuf);

    return abuf->length;
}

void
abuf_zero (Autobuf *abuf)
{
    assert (abuf);

    zero (abuf);
}

int
abuf_copy (Autobuf *dest, const Autobuf *src)
{
    assert (dest);
    assert (src);

    return copy_buf (dest, src->data, src->length);
}

int
abuf_copy_buf (Autobuf *abuf, const uchar__t *buf, ssize_t len)
{
    assert (abuf);
    assert (buf);
    assert (len >= 0);

    return copy_buf (abuf, buf, len);
}

int
abuf_copy_astr (Autobuf *abuf, const Autostr *str)
{
    assert (abuf);
    assert (str);

    return copy_buf (abuf, str->s, str->length);
}

int
abuf_copy_s (Autobuf *abuf, const char *s)
{
    assert (abuf);
    assert (s);

    return copy_buf (abuf, s, strlen (s));
}

int
abuf_copy_c (Autobuf *abuf, int c)
{
    assert (abuf);

    abuf->length = 1;
    abrealloc (abuf, smallest_chunk (abuf->length)); /* ignore realloc failure */
    abuf->data[0] = (uchar__t) c;
    return 0;
}

int
abuf_cat (Autobuf *dest, const Autobuf *src)
{
    assert (dest);
    assert (src);

    return cat_buf (dest, src->data, src->length);
}

int
abuf_cat_buf (Autobuf *abuf, const uchar__t *buf, ssize_t len)
{
    assert (abuf);
    assert (buf);
    assert (len >= 0);

    return cat_buf (abuf, buf, len);
}

int
abuf_cat_astr (Autobuf *abuf, const Autostr *str)
{
    assert (abuf);
    assert (str);

    return cat_buf (abuf, str->s, str->length);
}

int
abuf_cat_s (Autobuf *abuf, const char *s)
{
    assert (abuf);
    assert (s);

    return cat_buf (abuf, s, strlen (s));
}

int
abuf_cat_c (Autobuf *abuf, int ch)
{
    uchar__t c = ch;
    assert (abuf);

    return cat_buf (abuf, &c, 1);
}

int
abuf_insert (Autobuf *dest, ssize_t index, const Autobuf *src)
{
    assert (dest);
    assert (src);
    assert (index <= dest->length);
    assert (index >= 0);

    return insert_buf (dest, index, src->data, src->length);
}

int
abuf_insert_buf (Autobuf *abuf, ssize_t index, const uchar__t *buf, ssize_t len)
{
    assert (abuf);
    assert (buf);
    assert (index <= abuf->length);
    assert (index >= 0);
    assert (len >= 0);

    return insert_buf (abuf, index, buf, len);
}

int
abuf_insert_astr (Autobuf *abuf, ssize_t index, const Autostr *str)
{
    assert (abuf);
    assert (str);
    assert (index <= abuf->length);
    assert (index >= 0);

    return insert_buf (abuf, index, str->s, str->length);
}

int
abuf_insert_s (Autobuf *abuf, ssize_t index, const char *s)
{
    assert (abuf);
    assert (index <= abuf->length);
    assert (index >= 0);

    return insert_buf (abuf, index, s, strlen (s));
}

int
abuf_insert_c (Autobuf *abuf, ssize_t index, int ch)
{
    uchar__t c = ch;

    assert (abuf);
    assert (index <= abuf->length);
    assert (index >= 0);

    return insert_buf (abuf, index, &c, 1);
}

void
abuf_delete (Autobuf *abuf, ssize_t index, ssize_t n)
{
    uchar__t *src, *dest, *end;

    assert (abuf);
    assert (index < abuf->length);

    if (n == 0)
	return;
    if (n < 0)
	n = abuf->length - index;

    assert (index + n <= abuf->length);

    /* Move characters down if we deleted from the middle rather than
     * truncating at the end.  I'm still going for the explicit loop, as
     * opposed to a memmove.  It's a little less convenient here than in
     * str_delete, because I don't have a NUL to tell me when I've reached
     * the end, but nevertheless, I think it's much cleaner to work out the
     * address of the last character to copy than the number of characters
     * wanted.  RHA: you're still right. :) */
    dest = abuf->data + index;
    src = dest + n;
    end = abuf->data + abuf->length - 1;
    while (src <= end)
	*dest++ = *src++;

    abuf->length -= n;
    try_shrink (abuf);
}

int
abuf_equal (const Autobuf *x, const Autobuf *y)
{
    assert (x);
    assert (y);

    return compare (x, y) == 0;
}

int
abuf_cmp (const Autobuf *x, const Autobuf *y)
{
    assert (x);
    assert (y);

    return compare (x, y);
}

inline static ssize_t
terminate_get (Autobuf *abuf, ssize_t offset)
{
    abuf->length = offset;
    return offset;
}

ssize_t
abuf_getdelim (Autobuf *abuf, FILE *file, int delim)
{
    ssize_t offset;
    int c;

    assert (abuf);
    assert (file);

    offset = 0;
    zero (abuf);

    for (;;)
    {
	c = getc (file);

	if (delim >= 0 && c == (uchar__t) delim)
	    /* end of record: tidy up and return length */
	    return terminate_get (abuf, offset);

	if (c == EOF)
	{
	    if (offset == 0)
		/* If this is the first char read on this call, then
		 * return EOF. */
		return EOF;
	    /* Else `unexpected' EOF: tidy up and return length */
	    return terminate_get (abuf, offset);
	}

	/* A normal character (including NUL): add it to ABUF */
	if (addc_offset (abuf, c, offset) == -1)
	{
	    terminate_get (abuf, offset);
	    errno = ENOMEM;
	    return EOF;
	}
	offset++;
    }
}

ssize_t
abuf_fread (Autobuf *abuf, FILE *file, ssize_t n)
{
    assert (abuf);
    assert (file);

    if (n == 0)			/* do nothing, but clear the buffer */
    {
	zero (abuf);
	return 0;
    }

    if (n < 0)			/* read as much as currently fits */
	n = abuf->length;
    else			/* N > 0: read N bytes */
    {
	int i = abrealloc (abuf, smallest_chunk (n));
	if (n > abuf->alloced && i == -1)
	{
	    errno = ENOMEM;
	    return EOF;
	}
	abuf->length = n;
    }

    return fread (abuf->data, 1, n, file);
}

ssize_t
abuf_fwrite (const Autobuf *abuf, FILE *file, ssize_t index, ssize_t n)
{
    assert (abuf);
    assert (file);
    assert (index >= 0);
    assert (index < abuf->length);
    assert (index + n <= abuf->length);

    if (n < 0)
	n = abuf->length - index;
    return fwrite (abuf->data + index, 1, n, file);
}

int
abuf_set (Autobuf *abuf, ssize_t n, int c)
{
    assert (abuf);
    assert (n >= 0);

    if (n == 0)
	zero (abuf);
    else
    {
	int i = abrealloc (abuf, smallest_chunk (n));

	if (n > abuf->alloced && i == -1)
	    return -1;
	abuf->length = n;
	memset (abuf->data, c, n);
    }

    return 0;
}

inline static const uchar__t *
find_mem (const uchar__t *haystack, ssize_t haystack_len,
	  const uchar__t *needle, ssize_t needle_len)
{
    const uchar__t *p;
    const uchar__t *const last = haystack + haystack_len - needle_len;

    for (p = haystack; p <= last; p++)
	if (*p == *needle
	    && memcmp (p + 1, needle + 1, needle_len - 1) == 0)
	    return p;

    return 0;
}

ssize_t
abuf_find (const Autobuf *abuf, ssize_t index, const Autobuf *target)
{
    const uchar__t *p;

    assert (abuf);
    assert (target);

    if (target->length == 0)
	return -1;

    index++;
    if (index < 0)
	index = 0;

    assert (index < abuf->length);

    p = find_mem (abuf->data + index, abuf->length - index, target->data, target->length);
    return p ? p - abuf->data : -1;
}

ssize_t
abuf_find_buf (const Autobuf *abuf, ssize_t index, const uchar__t *buf, ssize_t len)
{
    const uchar__t *p;

    assert (abuf);
    assert (buf);
    assert (len >= 0);

    if (len == 0)
	return -1;

    index++;
    if (index < 0)
	index = 0;

    assert (index < abuf->length);

    p = find_mem (abuf->data + index, abuf->length - index, buf, len);
    return p ? p - abuf->data : -1;
}

ssize_t
abuf_find_astr (const Autobuf *abuf, ssize_t index, const Autostr *astr)
{
    const uchar__t *p;
    ssize_t target_len;

    assert (abuf);
    assert (astr);

    target_len = astr->length;

    if (target_len == 0)
	return -1;

    index++;
    if (index < 0)
	index = 0;

    assert (index < abuf->length);

    p = find_mem (abuf->data + index, abuf->length - index, astr->s, target_len);
    return p ? p - abuf->data : -1;
}

ssize_t
abuf_find_s (const Autobuf *abuf, ssize_t index, const char *s)
{
    const uchar__t *p;
    ssize_t target_len;

    assert (abuf);
    assert (s);

    target_len = strlen (s);

    if (target_len == 0)
	return -1;

    index++;
    if (index < 0)
	index = 0;

    assert (index < abuf->length);

    p = find_mem (abuf->data + index, abuf->length - index, s, target_len);
    return p ? p - abuf->data : -1;
}

ssize_t
abuf_find_c (const Autobuf *abuf, ssize_t index, int c)
{
    const uchar__t *p;

    assert (abuf);

    index++;
    if (index < 0)
	index = 0;

    assert (index <= abuf->length);

    p = memchr (abuf->data + index, c, abuf->length - index);
    return p ? p - abuf->data : -1;
}

ssize_t
abuf_rfind_c (const Autobuf *abuf, ssize_t last, int c)
{
    const uchar__t *p;

    assert (abuf);
    assert (last < abuf->length);

    if (last < 0)
	last = abuf->length;

    for (p = abuf->data + last - 1;  p >= abuf->data;  p--)
	if (*p == (uchar__t) c)
	    return p - abuf->data;

    return -1;
}

int
abuf_slice (Autobuf *dest, const Autobuf *src, ssize_t index, ssize_t n)
{
    assert (dest);
    assert (src);
    assert (index < src->length);
    assert (index >= 0);

    if (n < 0)
	n = src->length - index;

    assert (index + n <= src->length);

    return slice (dest, src, index, n);
}

int
abuf_slice_i (Autobuf *dest, const Autobuf *src, ssize_t start, ssize_t end)
{
    ssize_t n;

    assert (dest);
    assert (src);
    assert (start < src->length);
    assert (start >= 0);

    if (end < 0)
	n = src->length - start;
    else
    {
	assert (end <= src->length); /* `<=' because we stop before that index */
	assert (end >= start);
	n = end - start;
    }

    return slice (dest, src, start, n);
}
