/* autostr-priv.h -- header for string-private static functions in libretto
 *
 * Aaron Crane <aaronc@pobox.com>
 * 11 August 1997
 * 12 February 1998
 * 26 February 1998
 *
 * This file is part of Libretto, a library of useful functions.  Libretto is
 * Copyright  1996, 1997, 1998 Aaron Crane <aaronc@pobox.com>.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>

/* A few implementation notes:
 *
 * A Autostr with a null .s member is invalid.  On initialise, we allocate
 * at least one char (the terminating NUL).  astr_zero() returns to that
 * state.
 */

/* must be a power of two; see smallest_chunk() */
#define ASTR_CHUNK	(64u)

/*
 * Some inlined helper functions.
 *
 * All of them assume that the string is non-NULL and valid.  In general,
 * they tend not to perform sanity checks on what they do, so be careful.
 */

#define unufun __attribute__ ((unused))
inline static int s_validp (const Autostr *str) unufun;
inline static int s_realloc (Autostr *, ssize_t) unufun;
inline static ssize_t s_chunkof (ssize_t) unufun;
inline static void s_shrink (Autostr *) unufun;
inline static int s_init (Autostr *, ssize_t) unufun;
inline static void s_final (Autostr *) unufun;
inline static int s_addc (Autostr *, int, ssize_t) unufun;
inline static void s_zero (Autostr *) unufun;
inline static ssize_t s_terminate (Autostr *, ssize_t) unufun;
inline static int s_copy_s (Autostr *, const char *, ssize_t) unufun;
inline static int s_cat_s (Autostr *, const char *, ssize_t) unufun;
inline static int s_insert_s (Autostr *, ssize_t, const char *, ssize_t) unufun;
inline static int s_slice (Autostr *, const Autostr *, ssize_t, ssize_t) unufun;
#undef unufun

inline static int
s_validp (const Autostr *str)
{
    if (str->length >= str->alloced)
	return 0;
    if (str->length != (ssize_t) strlen (str->s))
	return 0;
    return 1;
}

/* WARNING: this function does not expect to be passed a valid Autostr.
 * Use it only if you know exactly what you are doing. */
inline static int
s_realloc (Autostr *str, ssize_t new_alloced)
{
    if (str->alloced != new_alloced)
    {
	char *p = mem_realloc (str->s, new_alloced);

	if (!p)
	    return -1;

	str->alloced = new_alloced;
	str->s = p;
    }

    return 0;
}

inline static ssize_t
s_chunkof (ssize_t len)
{
    len += ASTR_CHUNK - 1;
    len &= ~ (ASTR_CHUNK - 1);
    return len;
}

inline static void
s_shrink (Autostr *astr)
{
    if (astr->alloced - astr->length >= (ssize_t) ASTR_CHUNK)
    {
	ssize_t new_alloced = s_chunkof (astr->length + 1);
	void *buf = astr->s;

	/* This is an attempted shrinkage, so it's not a problem if the
	 * reallocation fails */
	if (mem_try_realloc (&buf, new_alloced) != -1)
	{
	    astr->s = buf;
	    astr->alloced = new_alloced;
	}
    }
}

	
/* Caller must ensure that SIZE > 0 */
inline static int
s_init (Autostr *str, ssize_t size)
{
    char *p;

    p = mem_alloc (size);
    if (!p)
    {
	size = 1;
	p = mem_alloc (size);
    }

    if (!p)
	return -1;
    else
    {
	str->s = p;
	str->alloced = size;
	str->length = 0;
	str->s[0] = 0;
    }

    return 0;
}

inline static void
s_final (Autostr *str)
{
    mem_free (str->s);
}

/* WARNING: this function does not maintain a correct Autostr; that is the
 * responsibility of the caller.  For example, it does not NUL-terminate,
 * nor does it set STR->length. */
inline static int
s_addc (Autostr *str, int c, ssize_t offset)
{
    while (offset >= str->alloced - 1)
	if (s_realloc (str, str->alloced + ASTR_CHUNK) == -1)
	    return -1;
    str->s[offset] = (char) c;
    return 0;
}

inline static void
s_zero (Autostr *astr)
{
    astr->length = 0;
    astr->s[0] = 0;
    s_shrink (astr);
}

inline static ssize_t
s_terminate (Autostr *str, ssize_t offset)
{
    /* XXX: I *think* that STR already has room for its terminating null
     * (assuming that we've been using s_addc() to build it up).  Of course,
     * if I'm wrong, then everything is horrifically broken... but the test
     * suite completes successfully. */
    str->s[offset] = 0;
    str->length = offset;
    return offset;
}

inline static int
s_copy_s (Autostr *str, const char *s, ssize_t len)
{
    if (len == 0)
	s_zero (str);
    else
    {
	int i = s_realloc (str, s_chunkof (len + 1));

	if (len >= str->alloced && i == -1)
	    return -1;
	str->length = len;
	strcpy (str->s, s);
    }

    return 0;
}

inline static int
s_cat_s (Autostr *str, const char *s, ssize_t len)
{
    if (len != 0)
    {
	int i = s_realloc (str, s_chunkof (str->length + len + 1));

	if (i == -1)
	    return -1;
	strcpy (str->s + str->length, s);
	str->length += len;
    }

    return 0;
}

inline static int
s_insert_s (Autostr *str, ssize_t index, const char *s, ssize_t len)
{
    if (index == str->length)
	return s_cat_s (str, s, len);

    if (len != 0)
    {
	ssize_t oldlen;
	int i = s_realloc (str, s_chunkof (str->length + len + 1));

	if (i == -1)
	    return -1;

	oldlen = str->length;
	memmove (str->s + index + len, str->s + index, oldlen - index + 1);
	memcpy (str->s + index, s, len);
	str->length += len;
    }

    return 0;
}

inline static int
s_slice (Autostr *dest, const Autostr *src, ssize_t index, ssize_t n)
{
    if (src != dest)
    {
	int i = s_realloc (dest, s_chunkof (n + 1));

	if (n > dest->alloced && i == -1)
	    /* If realloc failed on a grow, then bail out now */
	    return -1;

	dest->length = n;
	strncpy (dest->s, src->s + index, n);
	dest->s[n] = 0;
    }
    else			/* SRC and DEST are the same object */
    {
	dest->length = n;
	memmove (dest->s, dest->s + index, n);
	dest->s[n] = 0;
	s_shrink (dest);
    }

    return 0;
}
