/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2009 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
/*! \headerfile libpwmd.h
 *
 * libpwmd is a library making it easy for applications to use the pwmd
 * server. Pwmd version 2.0 or later is required; either locally or remotely. 
 */

/*! \section threads Multiple Threads
 *
 * \ref libpwmd should be thread-safe on a per handle bases. Meaning that only
 * one thread should access a \ref pwm_t handle at a time.
 *
 * There is a limitation that when compiled with \ref pth(3) support SSH
 * connections cannot make use of the \ref pth(3) IO functions and therefore
 * will block while waiting on IO operations in other threads (as of libssh2
 * 1.2.8).
 */

/*! \section ssh SSH Details
 *
 * A remote connection to a pwmd server is possible by using an SSH channel
 * which spawns a shell and executes a proxy server that connects to the pwmd
 * local Unix Domain Socket. Authentication is done by using SSH public key
 * (see \ref ssh-keygen(1)) authentication and verifying the host key against
 * a local OpenSSH known hosts formatted file.
 *
 * The servers public key can be had by using \ref pwmd_get_hostkey() and
 * storing the result in a file or done automatially by using a callback
 * function \ref pwmd_knownhost_cb_t while connecting to the unknown host.
 *
 * On the server side you'll need a proxy server to connect to the real pwmd
 * server. Here's an example \ref authorized_keys(5) entry that will do this.
 * The \a public_key portion should be the same as the contents of the
 * generated \ref ssh-keygen(1) \a identity.pub file which is passed as a
 * parameter to the SSH connection functions:
 *
 * \code
 * command="socat UNIX-CONNECT:$HOME/.pwmd/socket -" <public_key> ...
 * \endcode
 *
 * \note Only an SSH identity without a passphrase is supported. For now
 * anyway. This is a limitation of libssh2 (version 1.1 as of this writing).
 *
 * \version 6.0.3
 * The first version to use the OpenSSH known hosts file format exclusively.
 * Earlier versions used only an SHA1 hash of the host key.
 *
 * \x11
 */

/*! \section pinentry Pinentry Details
 *
 * \ref pinentry(1) is a program that prompts the user for input which is
 * normally a passphrase or a confirmation. libpwmd can use this program
 * either locally (X11 forwarding is not yet supported) or have the pwmd
 * server use it's pinentry to retrieve a passphrase when needed. How this is
 * done depends what function gets called and whether the pwmd connection is
 * over an SSH channel.
 *
 * There are a few options that tell pinentry how and where to prompt for a
 * needed passphrase. See the \ref pwmd_option_t section for details. These
 * options are not sent (when using pwmd's pinentry, not the local one) until
 * the pinentry is needed.
 *
 * If using a local pinentry by calling \ref pwmd_open2(), \ref pwmd_save2(),
 * \ref pwmd_open_async2() or pwmd_save_async2(), libpwmd will send the
 * command "SET ENABLE_PINENTRY=0" to the pwmd server. This is needed so pwmd
 * wont try to launch it's own pinentry on passphrase or confirmation failure.
 * So you may need to reset this option manually depending on your needs;
 * especially when changing pinentry methods when doing a save (the passphrase
 * may be set as empty since the remote pinentry is disabled!).
 *
 * Some pinentry options can also be specified in a local configuration file
 * \a "~/.pwmd/pinentry.conf". These options are initial values for each
 * pinentry invokation (not retries) and may be changed by setting the
 * appropriate \ref pwmd_option_t. Each option and value is separated with a
 * '=' on a single line. Unrecognized options are ignored. Here are the
 * recognized options:
 *
 * \param PATH The full path to the location of the pinentry binary.
 * \param DISPLAY The X11 display to use.
 * \param TTYNAME The full path to the tty that pinentry should prompt on.
 * \param TTYTYPE The terminal type of the tty which is required if DISPLAY is
 * not set.
 *
 * \filepath
 *
 * \note The initial values for the pinentry TTY, TERM and DISPLAY are set
 * during \ref pwmd_new() depending on the current environment. They may need
 * to be reset as needed.
 *
 * \note After establishing an SSH connection, the pwmd pinentry is disabled
 * by sending the command "SET ENABLE_PINENTRY=0". This is needed because
 * there currently isn't a way to have the remote pinentry use the local
 * display.  You must be careful to use either a local pinentry or set a
 * passphrase manually with \ref pwmd_setopt() when a passphrase is required
 * or needed.
 *
 * \x11
 *
 * \see \ref ssh
 */

/*! \section Errors
 *
 * libpwmd uses libgpg-error for all error codes.  Some are user defined
 * GPG_ERR_USER_N codes, but most are reused from the existing ones.  Error
 * codes can be described by using \ref pwmd_strerror(), or the thread-safe
 * \ref pwmd_strerror_r().
 *
 * \note libgpg-error normally returns an error code as a bitmask of an error
 * source and the actual error code. In order to simplify the result codes,
 * libpwmd strips any error source from the error code before returning it.
 */

/*! \file */
#ifndef LIBPWMD_H
#define LIBPWMD_H

#ifndef LIBPWMD_API
#define LIBPWMD_API
#endif

#include <gpg-error.h>
#include <stdarg.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \def LIBPWMD_VERSION
 * \brief Version information.
 *
 * The version of this library.
 */
#define LIBPWMD_VERSION 0x603


/*! \def LIBPWMD_VERSION_MAJOR
 * \brief Version information.
 *
 * The major release number of this library.
 */
#define LIBPWMD_VERSION_MAJOR 6


/*! \def LIBPWMD_VERSION_MINOR
 * \brief Version information.
 *
 * The minor release number of this library.
 */
#define LIBPWMD_VERSION_MINOR 0


/*! \def LIBPWMD_VERSION_PATCH
 * \brief Version information.
 *
 * The patch level of this library.
 */
#define LIBPWMD_VERSION_PATCH 3


/*! \def LIBPWMD_VERSION_STR
 * \brief Version information.
 *
 * A string representation of the version of this library.
 */
#define LIBPWMD_VERSION_STR "6.0.3"


/*! \brief Returns this version of libpwmd.
 *
 * As a string.
 * \return A string.
 */
LIBPWMD_API const char *pwmd_version();


struct pwm_s;
/*! \typedef pwm_t
 * \brief libpwmd handle.
 *
 * When a handle is mentioned in this documentation it is a pointer of this
 * type. A new handle is created with \ref pwmd_new().
 */
typedef struct pwm_s pwm_t;


/*! \typedef pwmd_async_t
 * \brief Asynchronous return value.
 *
 * The return code of \ref pwmd_process() which is used for all asynchronous
 * commands.
 */
typedef enum {
    /*! \internal */
    ASYNC_INIT,

    /*! \ref pwmd_process() should be called again. */
    ASYNC_PROCESS,

    /*! The command has completed. The result code should be checked for an
     * error. */
    ASYNC_DONE,
} pwmd_async_t;


/*! \typedef pwmd_ip_version_t
 * \brief IP protocol version for remote SSH connections.
 *
 * The value of the option \ref PWMD_OPTION_IP_VERSION which is set with \ref
 * pwmd_setopt().
 */
typedef enum {
    /*! Try both IPv4 and IPv6 addresses. Note that IPv6 is tried first and
     * that \ref PWMD_IP_ANY only affects a hostname and not an IP address in
     * the address specification. */
    PWMD_IP_ANY,

    /*! Only try IPv4. */
    PWMD_IPV4,

    /*! Only try IPv6. */
    PWMD_IPV6
} pwmd_ip_version_t;


/*! \def PWMD_FD_READABLE
 * \brief For use with \ref pwmd_get_fds().
 *
 * Set when the file descriptor is readable.
 */
#define PWMD_FD_READABLE	0x01


/*! \def PWMD_FD_WRITABLE
 * \brief For use with \ref pwmd_get_fds().
 *
 * Set when the file descriptor is writable.
 */
#define PWMD_FD_WRITABLE	0x02


/*! For use with \ref pwmd_get_fds(). */
typedef struct {
    /*! The file descriptor. */
    int fd;

    /*! A bitmask of \ref PWMD_FD_READABLE and \ref PWMD_FD_WRITABLE. */
    unsigned flags;
} pwmd_fd_t;


/*! \typedef pwmd_socket_t
 * \brief The type of socket a handle is connected to.
 *
 * For use with \ref pwmd_socket_type().
 */
typedef enum {
    /*! A local domain socket. */
    PWMD_SOCKET_LOCAL,

    /*! An SSH connection over a TCP socket. */
    PWMD_SOCKET_SSH
} pwmd_socket_t;


/*! \typedef pwmd_pinentry_t
 * \brief Local pinentry commands and not pwmd pinentry.
 *
 * For use with \ref pwmd_getpin().
 */
typedef enum {
    /*! When opening a file. */
    PWMD_PINENTRY_OPEN,

    /*! When opening a file failed. */
    PWMD_PINENTRY_OPEN_FAILED,

    /*! When saving a file. */
    PWMD_PINENTRY_SAVE,

    /*! For passphrase confirmation. */
    PWMD_PINENTRY_SAVE_CONFIRM,

    /*! For confirmation of a user-defined prompt. */
    PWMD_PINENTRY_CONFIRM,

    /*! For the default or user defined string set with \ref
     * PWMD_OPTION_PINENTRY_DESC. */
    PWMD_PINENTRY_DEFAULT,

    /*! To terminate the pinentry process created with \ref pwmd_getpin(). */
    PWMD_PINENTRY_CLOSE
} pwmd_pinentry_t;


/*! \typedef pwmd_passphrase_cb_t
 * \brief Custom passphrase retrieval function.
 *
 * The value of the option \ref PWMD_OPTION_PASSPHRASE_CB which is set with
 * \ref pwmd_setopt().
 *
 * \param user A user data pointer which is set with \ref
 * PWMD_OPTION_PASSPHRASE_DATA.
 * \param[out] passphrase The passphrase which may be an empty string or NULL.
 * It is not modified by libpwmd but must remain allocated for as long as it
 * is needed.
 * \return 0 on success or an error code which will cause a command to fail.
 */
typedef gpg_error_t (*pwmd_passphrase_cb_t)(void *user, char **passphrase);


/*! \typedef pwmd_status_cb_t
 * \brief Process status messages.
 *
 * The value of the option \ref PWMD_OPTION_STATUS_CB which is set with \ref
 * pwmd_setopt().
 *
 * \param user A user data pointer which is set with \ref
 * PWMD_OPTION_STATUS_DATA.
 * \param line The status message line received from the server.
 * \return 0 on success or an error code which will cause a command to fail.
 */
typedef int (*pwmd_status_cb_t)(void *user, const char *line);


/*! \typedef pwmd_inquire_cb_t
 * \brief Send data to the pwmd server.
 *
 * This is a callback function that gets passed to \ref pwmd_inquire(). It is
 * used for sending data to the server for commands that need to reply to an
 * INQUIRE server response (STORE and IMPORT). The reason for this callback is
 * to let the client send as many bytes as it wants rather than the entire
 * chunk at once.  It gets called during an internal \ref assuan_transact()
 * from an internal inquire callback function which in turn calls this
 * function by looping over its return value.
 *
 * \param user The user data pointer passed to \ref pwmd_inquire().
 * \param cmd The same as the \a cmd argument to \ref pwmd_inquire().
 * \param rc The result of the last internal call to \ref assuan_send_data()
 * which did the sending of the data to the pwmd server. On the first call to
 * this callback it's value will always be 0 since no data has been sent yet.
 * \param[out] data The next chunk of data to send or NULL.
 * \param[out] len The length of \a data or 0.
 *
 * \retval 0 There is more data to be sent.
 * \retval GPG_ERR_CANCELED Cancel the current inquire.
 * \retval GPG_ERR_EOF No need to call this function again, the current
 * \a line is the last to send.
 * \retval code Any other error code which will terminate the INQUIRE.
 *
 * \note The sent data is processed line-per-line. The line is either newline
 * terminated or is buffered until ASSUAN_LINELENGTH bytes have been
 * allocated. Any remaining bytes are sent after the INQUIRE has finished.
 */
typedef gpg_error_t (*pwmd_inquire_cb_t)(void *user, const char *cmd,
	gpg_error_t rc, char **data, size_t *len);


/*! \typedef pwmd_knownhost_cb_t
 * \brief Verify a remote SSH connection.
 *
 * When \ref PWMD_OPTION_KNOWNHOST_CB is set and a the current connections
 * host key was not found in the known hosts file, then this callback function
 * can be used to confirm the addition of the new host key to the known_hosts
 * file.
 *
 * \param user User data which was set with \ref PWMD_OPTION_KNOWNHOST_DATA.
 * \param host The hostname as passed to the connecting function.
 * \param key The raw host key. Note that this differs from the format
 * returned from \ref pwmd_get_hostkey().
 * \param len The host key length.
 * \retval 0 Add the host key to the known hosts file.
 * \retval GPG_ERR_NOT_CONFIRMED Do not add the host key and abort the
 * connection.
 *
 * \note If the known hosts file cannot be modified do to filesystem
 * restrictions when trying to add the new host key, no error is returned.
 * Instead the host key is added to the current connections host key cache and
 * the connection is considered verified.
 *
 * \see \ref ssh
 */
typedef gpg_error_t (*pwmd_knownhost_cb_t)(void *user, const char *host,
	const char *key, size_t len);


/*! \enum pwmd_option_t
 * \brief libpwmd options.
 *
 * Options are set with \ref pwmd_setopt().
 *
 * \filepath
 */
typedef enum {
    /*! A custom passphrase retrieval function which, when set, will be used
     * instead of \ref pinentry(1). This function will not be used if opening
     * a file and the passphrase is cached on the server or the file is a new
     * one. The value of this option should be a \ref pwmd_passphrase_cb_t.
     *
     * \note An empty string as the passphrase is allowed.
     */
    PWMD_OPTION_PASSPHRASE_CB,

    /*! User supplied data which is passed to the custom passphrase function.
     * */
    PWMD_OPTION_PASSPHRASE_DATA,

    /*! A string to use as the passphrase when doing an open or save. When not
     * NULL, this option has precedence over \ref PWMD_OPTION_PASSPHRASE_CB.
     *
     * \note An empty string as the passphrase is allowed.
     */
    PWMD_OPTION_PASSPHRASE,

    /*! An integer value that specifies the number of tries before \ref
     * pinentry(1) will give up when opening a file with the wrong supplied
     * passphrase. The default is 3.
     *
     * \note This option has no effect when trying to save a file. The user
     * must either cancel the pinentry causing the save to fail or enter the
     * correct passphrase during passphrase confirmation.
     */
    PWMD_OPTION_PINENTRY_TRIES,

    /*! A character string value which specifies the full path of the \ref
     * pinentry(1) binary. For the local \ref pinentry(1) method, the default
     * is specified at compile time.
     * 
     * \see \ref pinentry
     */
    PWMD_OPTION_PINENTRY_PATH,

    /*! A value which specifies the full path to the TTY that \ref pinentry(1)
     * will use to prompt on. When set and no DISPLAY is available, \ref
     * PWMD_OPTION_PINENTRY_TERM must also be set.
     *
     * \see \ref pinentry
     */
    PWMD_OPTION_PINENTRY_TTY,

    /*! A value which specifies the terminal type (e.g., vt100) that \ref
     * pinentry(1) will use when no X11 display is available.
     *
     * \see \ref pinentry
     */
    PWMD_OPTION_PINENTRY_TERM,

    /*! A value which specifies the X11 display that \ref pinentry(1) will
     * use.
     *
     * \x11
     *
     * \see \ref pinentry
     */
    PWMD_OPTION_PINENTRY_DISPLAY,

    /*! A character string that \ref pinentry(1) will use in it's dialog
     * window.
     */
    PWMD_OPTION_PINENTRY_TITLE,

    /*! \copydoc PWMD_OPTION_PINENTRY_TITLE */
    PWMD_OPTION_PINENTRY_PROMPT,

    /*! \copydoc PWMD_OPTION_PINENTRY_TITLE */
    PWMD_OPTION_PINENTRY_DESC,

    /*! For \ref pinentry(1) localization. */
    PWMD_OPTION_PINENTRY_LC_CTYPE,

    /*! \copydoc PWMD_OPTION_PINENTRY_LC_CTYPE */
    PWMD_OPTION_PINENTRY_LC_MESSAGES,

    /*! An integer value that specifies the number of seconds \ref pinentry(1)
     * will wait for input before timing out and aborting the current command.
     * If 0, then no timeout will be used. The default is 30.
     */
    PWMD_OPTION_PINENTRY_TIMEOUT,

    /*! A function of type \ref pwmd_status_cb_t that will process status
     * messages received from the pwmd server.
     */
    PWMD_OPTION_STATUS_CB,

    /*! A user data pointer which is passed to the status message function. */
    PWMD_OPTION_STATUS_DATA,

    /*! The IP version of type \ref pwmd_ip_version_t that \ref
     * pwmd_ssh_connect() and \ref pwmd_ssh_connect_async() will use when
     * connecting to the remote pwmd server. The default is \ref PWMD_IP_ANY.
     *
     * \pre_conn_req
     */
    PWMD_OPTION_IP_VERSION,

    /*! A function to confirm an unknown host hash which wasn't found in the
     * known hosts file.
     *
     * \see \ref ssh
     */
    PWMD_OPTION_KNOWNHOST_CB,

    /*! User supplied data which is passed to the known host function.
     *
     * \see \ref ssh
     */
    PWMD_OPTION_KNOWNHOST_DATA,

    /*! When the total number of bytes to be sent via an INQUIRE is known,
     * this should be set so XFER status messages can be parsed correctly.
     * When not known or unset, 0 is used as the total argument to the XFER
     * status message. This option should be set before each call to
     * \ref pwmd_inquire().
     *
     * \note During the INQUIRE, PWMD_OPTION_STATUS_CB is called after every
     * ASSUAN_LINELENGTH bytes have been successfully transferred.
     *
     * \note This is a libpwmd feature. pwmd itself does not send XFER status
     * messages during an INQUIRE. Status messages can be parsed only when
     * PWMD_OPTION_STATUS_CB is set.
     */
    PWMD_OPTION_INQUIRE_TOTAL,

    /*! When set to 1, lock the file mutex after opening a file as if the LOCK
     * command had been sent.
     */
    PWMD_OPTION_LOCK_ON_OPEN,

    /*! A long integer specifying the number of iterations to encrypt with.
     * When -1 then the next save operation will use the current iteration
     * setting and no command will be sent to alter it.
     */
    PWMD_OPTION_ITERATIONS,

    /*! A string specifying the cipher to use to encrypt with. See the pwmd(1)
     * manual page for available ciphers.
     */
    PWMD_OPTION_CIPHER,

    /*! When 1, tell pwmd that the passphrase is Base64 encoded. pwmd will
     * decode the passphrase before encryption and decryption. */
    PWMD_OPTION_BASE64,

    /*! Use ssh-agent to retrieve private key to authenticate. */
    PWMD_OPTION_SSH_AGENT,
} pwmd_option_t;


/*! \brief Initialize the library.
 *
 * This function must be the first function called in the library before any
 * others. It sets up the memory allocators and internationalization among
 * other things.
 *
 * \return 0 on success or an error code.
 */
LIBPWMD_API gpg_error_t pwmd_init(void);


/*! \brief Creates a new handle.
 *
 * Creates a new handle for use with the other functions.
 *
 * \param name If not NULL, the name of the application. The application name
 * is sent to the pwmd server after successfully connecting.
 *
 * \return a new handle or NULL if there was not enough memory.
 */
LIBPWMD_API pwm_t *pwmd_new(const char *name);


/*! \brief Connect to a local pwmd server.
 *
 * Connects to a local unix domain socket.
 *
 * \param pwm A handle.
 * \param path The socket path to connect to. If NULL, then a default of
 * \a "~/.pwmd/socket" will be used.
 * \return 0 on success or an error code.
 * \filepath
 * \see pwmd_ssh_connect(), pwmd_ssh_connect_async(), pwmd_disconnect()
 */
LIBPWMD_API gpg_error_t pwmd_connect(pwm_t *pwm, const char *path);


/*! \brief Establish a remote connection to a pwmd server.
 *
 * Connects to a pwmd server over an SSH channel.
 *
 * \param pwm A handle.
 * \param host The hostname to connect to or NULL to resume a connection
 * previously started with \ref pwmd_get_hostkey().
 * \param port The port or -1 for the default of 22.
 * \param identity The SSH identity file to use for authentication. This
 * should specify the private key. The public key is assumed to be \a
 * identity.pub. This parameter is not used if \ref PWMD_OPTION_SSH_AGENT is
 * set.
 * \param user The username on the SSH server to login as. If NULL then
 * invoking username will be used.
 * \param known_hosts An OpenSSH known hosts formatted file containing public
 * SSH server hashes which may be obtained with \ref pwmd_get_hostkey() or via
 * \ref pwmd_knownhost_cb_t during a connection. If NULL, the default of
 * \a "~/.ssh/known_hosts" will be used.
 * \return 0 on success or an error code.
 * \filepath
 * \see pwmd_ssh_connect_async(), \ref PWMD_OPTION_IP_VERSION,
 * pwmd_disconnect(), \ref ssh
 */
LIBPWMD_API gpg_error_t pwmd_ssh_connect(pwm_t *pwm, const char *host,
	int port, const char *identity, const char *user, const char
	*known_hosts);


/*! \brief Establish a remote connection to a pwmd server (asynchronously).
 *
 * This is a variant of \ref pwmd_ssh_connect() that will not block while doing
 * DNS lookups or while connecting. 
 *
 * \process
 *
 * \param pwm A handle.
 * \param host The hostname to connect to or NULL to resume a connection
 * previously started with \ref pwmd_get_hostkey().
 * \param port The port or -1 for the default of 22.
 * \param identity The SSH identity file to use for authentication. This
 * should specify the private key. The public key is assumed to be \a
 * identity.pub. This parameter is not used if \ref PWMD_OPTION_SSH_AGENT is
 * set.
 * \param user The username on the SSH server to login as. If NULL, the
 * invoking username will be used.
 * \param known_hosts An OpenSSH known hosts formatted file containing public
 * SSH server hashes which may be obtained with \ref pwmd_get_hostkey() or via
 * \ref pwmd_knownhost_cb_t during a connection. If NULL, the default of
 * \a "~/.ssh/known_hosts" will be used.
 * \return 0 on success or an error code.
 * \filepath
 * \see pwmd_process(), \ref PWMD_OPTION_IP_VERSION, pwmd_disconnect(),
 * \ref ssh
 */
LIBPWMD_API gpg_error_t pwmd_ssh_connect_async(pwm_t *pwm, const char *host,
	int port, const char *identity, const char *user, const char
	*known_hosts);


/*! \brief Establish a connection by parsing a URL.
 *
 * This allows for connecting to a pwmd server by parsing the given URL
 * string. Whether the connection is to a remote or local server depends on
 * the contents:
 * \code
 * file://[path/to/local/socket]
 *
 * or
 *
 * ssh[46]://[username@]hostname[:port][,identity,known_hosts]
 * \endcode
 *
 * The parameters in square brackets are optional and if not specified then
 * defaults will be used. If neither socket specification is matched, the
 * \a url is assumed to be a file://.
 *
 * \param pwm A handle.
 * \param url The string to parse.
 * \filepath
 * \return 0 on success or an error code.
 * \see \ref pwmd_socket_type(), pwmd_disconnect()
 */
LIBPWMD_API gpg_error_t pwmd_connect_url(pwm_t *pwm, const char *url);


/*! \brief Establish a connection by parsing a URL (asynchronously).
 *
 * This allows for connecting to a pwmd server by parsing the given URL
 * string. Whether the connection is to a remote or local server depends on
 * the contents:
 * \code
 * file://[path/to/local/socket]
 *
 * or
 *
 * ssh[46]://[username@]hostname[:port][,identity,known_hosts]
 * \endcode
 *
 * The parameters in square brackets are optional and if not specified then
 * defaults will be used. If neither socket specification is matched, the
 * \a url is assumed to be a file://.
 *
 * \process
 *
 * \param pwm A handle.
 * \param url The string to parse.
 * \filepath
 * \return 0 on success or an error code.
 * \see \ref pwmd_socket_type(), pwmd_disconnect()
 */
LIBPWMD_API gpg_error_t pwmd_connect_url_async(pwm_t *pwm, const char *url);


/*! \brief Retrieve a remote SSH public host key.
 *
 * This key is needed for host verification of the remote pwmd server. You
 * should be sure that the remote host is really the host that your wanting to
 * connect to and not subject to a man-in-the-middle attack.
 *
 * \param pwm A handle.
 * \param host The hostname or IP to connect to.
 * \param port The port or -1 for the default of 22.
 * \param[out] result An OpenSSH known hosts formatted string containing the
 * servers public key which should be freed with \ref pwmd_free(). If the
 * \a host was a hostname then two newline separated known host entries will
 * be returned; one for the hostname and one for the resolved IP address. The
 * IP known host entry will always be the second in the string.
 * \return 0 on success or an error code.
 *
 * \version 6.0.3
 * The connection is kept open but not verified after returning. It can be
 * resumed from one of the SSH connection functions.
 *
 * \see pwmd_get_hostkey_async(), pwmd_ssh_connect(), \ref ssh
 */
LIBPWMD_API gpg_error_t pwmd_get_hostkey(pwm_t *pwm, const char *host, int port,
	char **result);


/*! \brief Retrieve a remote SSH host key (asynchronously).
 *
 * This key is needed for host verification of the remote pwmd server. You
 * should be sure that the remote host is really the host that your wanting to
 * connect to and not subject to a man-in-the-middle attack.
 *
 * \process
 *
 * \param pwm A handle.
 * \param host The hostname or IP to connect to.
 * \param port The port or -1 for the default of 22.
 * \return 0 on success or an error code. The result is obtained from \ref
 * pwmd_process() should be freed with \ref pwmd_free(). It has the same
 * format as the result from \ref pwmd_get_hostkey().
 *
 * \version 6.0.3
 * The connection is kept open but not verified after returning. It can be
 * resumed from one of the SSH connection functions.
 *
 * \see pwmd_get_hostkey(), pwmd_ssh_connect_async(), \ref pwmd_process(),
 * \ref ssh
 */
LIBPWMD_API gpg_error_t pwmd_get_hostkey_async(pwm_t *pwm, const char *host,
	int port);


/*! \brief Get the associated file descriptor(s) for a handle.
 *
 * This function lets the application manually poll the available file
 * descriptors for the specified handle. It should be called after each
 * asynchronous function call and after each call to \ref pwmd_process() since
 * the polled file descriptors may have been closed since the previous call.
 *
 * After returning, \a n_fds is set to the number of available file
 * descriptors which are stored in \a fds. The .flags member of \ref pwmd_fd_t
 * specifies what can be monitored and is a bitmask of \ref PWMD_FD_READABLE
 * and \ref PWMD_FD_WRITABLE. When ready, \ref pwmd_process() should be
 * called.
 *
 * \param pwm A handle.
 * \param[out] fds Set to the file descriptor(s) of the associated handle.
 * \param[out] n_fds Initially the size of \a fds then updated to the number
 * of available file descriptors which are stored in \a fds.
 * \retval 0 on success or an error code.
 * \retval GPG_ERR_ERANGE There are more file descriptors than the amount
 * specified in \a n_fds. \a fds and \a n_fds are still usable though.
 * \see pwmd_process()
 */
LIBPWMD_API gpg_error_t pwmd_get_fds(pwm_t *pwm, pwmd_fd_t *fds, int *n_fds);


/*! \brief Check for a unparsed buffered line.
 *
 * A buffered line is a line that was read from the server but has not yet
 * been processed. This function determines if there is such a line.
 *
 * \param pwm A handle.
 * \retval 0 if there is a pending line.
 * \retval GPG_ERR_NO_DATA if there is no pending line.
 * \see pwmd_process()
 */
LIBPWMD_API gpg_error_t pwmd_pending_line(pwm_t *pwm);


/*! \brief Set handle options.
 *
 * See \ref pwmd_option_t for option specific details.
 *
 * \param pwm A handle.
 * \param opt The option and following value.
 * \return 0 on success or an error code.
 */
LIBPWMD_API gpg_error_t pwmd_setopt(pwm_t *pwm, pwmd_option_t opt, ...);


/*! \brief Launch a local pinentry.
 *
 * Does not send any command to the server. Maybe useful if a passphrase is
 * needed before opening a file over a remote connection. This passphrase can
 * then be set with \ref pwmd_setopt().
 *
 * This function may also be used to display a user confirmation dialog with
 * pinentry when \a which is \ref PWMD_PINENTRY_CONFIRM. The text to prompt
 * with is set with \ref PWMD_OPTION_PINENTRY_TITLE.
 *
 * \param pwm A handle.
 * \param filename The filename to use in the pinentry dialog strings.
 * \param[out] result The entered value in the pinentry dialog which should be
 * freed with \ref pwmd_free().
 * \param which Determines the default strings shown in the pinentry
 * dialog. \ref pwmd_setopt() may also be used to override the defaults. In
 * this case \ref PWMD_PINENTRY_DEFAULT should be used. \ref
 * PWMD_PINENTRY_CLOSE should be used to terminate the pinentry process when
 * the pinentry is no longer needed.
 *
 * \return 0 on success or an error.
 */
LIBPWMD_API gpg_error_t pwmd_getpin(pwm_t *pwm, const char *filename,
	char **result, pwmd_pinentry_t which);


/*! \brief Open a file on the pwmd server.
 *
 * This will send the OPEN command to the server.
 *
 * \param pwm A handle.
 * \param filename The filename to open. The \a filename is not a full path
 * but the data file only.
 * \return 0 on success or an error code.
 * \see \ref pinentry
 */
LIBPWMD_API gpg_error_t pwmd_open(pwm_t *pwm, const char *filename);


/*! \brief Open a file on the pwmd server by using a server inquire.
 *
 * This is a convenience function to the OPEN command using \ref
 * pwmd_inquire() in that it passes the OPEN options that were set with \ref
 * pwmd_setopt() for you. It uses the specified callback function to retreive
 * the passphrase.
 *
 * \note Only the passphrase should be sent when using this function and not
 * any filename argument. That argument will be sent automatically when using
 * this function.
 *
 * \note Pinentry is disabled when using this function.
 *
 * \param pwm A handle.
 * \param filename The filename to open. The \a filename is not a full path
 * but the data file only.
 * \param func A callback function of type \ref pwmd_inquire_cb_t which sets
 * the passphrase to be sent. This acts as a normal \ref pwmd_inquire() would.
 * \param user A user data pointer passed to the callback function \a func.
 * \return 0 on success or an error code.
 * \see \ref PWMD_OPTION_BASE64
 */
LIBPWMD_API gpg_error_t pwmd_open_inquire(pwm_t *pwm, const char *filename,
	pwmd_inquire_cb_t func, void *user);


/*! \brief Open a file on the pwmd server using a local pinentry.
 *
 * This will send the OPEN command to the server like \ref pwmd_open() but
 * will use the local pinentry and not pwmd's pinentry.
 *
 * \sigalrm
 *
 * \note This pinentry method is not thread safe. It needs to set a couple of
 * global variables for the pinentry timeout to work properly.
 *
 * \param pwm A handle.
 * \param filename The filename to open. The \a filename is not a full path
 * but the data file only.
 * \return 0 on success or an error code.
 * \retval GPG_ERR_PIN_BLOCKED Another handle is using the local pinentry.
 * \see \ref pinentry
 */
LIBPWMD_API gpg_error_t pwmd_open2(pwm_t *pwm, const char *filename);


/*! \brief Open a file on the pwmd server asynchronously (fork method).
 *
 * This is kind of a hybrid of \ref pwmd_open2() and \ref pwmd_open_async().
 * It will use the local pinentry asynchronously and also do the OPEN command
 * asynchronously.
 *
 * \process
 *
 * \sigalrm
 *
 * \param pwm A handle.
 * \param filename The filename to open. The \a filename is not a full path
 * but the data file only.
 * \return 0 on success or an error code.
 * \see pwmd_process(), \ref pinentry
 */
LIBPWMD_API gpg_error_t pwmd_open_async2(pwm_t *pwm, const char *filename);


/*! \brief Open a file on the pwmd server (asynchronously).
 *
 * This will send the OPEN command to the pwmd server. The difference from
 * \ref pwmd_open() is that it will not block if a pinentry is needed for
 * passphrase input.
 *
 * \process
 *
 * \param pwm A handle.
 * \param filename The filename to open. The \a filename is not a full path
 * but the data file only.
 * \return 0 on success or an error code.
 * \see pwmd_process(), \ref pinentry
 */
LIBPWMD_API gpg_error_t pwmd_open_async(pwm_t *pwm, const char *filename);


/*! \brief Process an asynchronous function.
 *
 * After an asynchronous function has been called and has returned
 * successfully, this function must be called to process the command and
 * retrieve the result and return value.
 *
 * This function may also be called when not in a command to check for pending
 * status messages sent from the server or to process a pending line.
 *
 * \param pwm A handle.
 * \param[out] rc Set to the return code of the original command after
 * ASYNC_DONE has been returned. This value must be checked to determine if
 * the command succeeded.
 * \param[out] result Set to the result of the command when \a rc is 0. Note
 * that not all commands return a result.
 * \retval ASYNC_DONE The command has completed. \a rc should be checked to
 * determine if the command was successful or not.
 * \retval ASYNC_PROCESS The command is still running and this function should
 * be called again.
 * \see pwmd_get_fds(), pwmd_pending_line()
 */
LIBPWMD_API pwmd_async_t pwmd_process(pwm_t *pwm, gpg_error_t *rc,
	char **result);


/*! \brief Save a file on the pwmd server.
 *
 * This will send the SAVE command.
 *
 * \param pwm A handle.
 * \return 0 on success or an error code.
 * \see \ref pinentry
 */
LIBPWMD_API gpg_error_t pwmd_save(pwm_t *pwm);


/*! \brief Save a file on the pwmd server by using a server inquire.
 *
 * This is a convenience function to the SAVE command using \ref
 * pwmd_inquire() in that it passes the SAVE options that were set with \ref
 * pwmd_setopt() for you. It uses the specified callback function to retreive
 * the passphrase.
 *
 * \note Pinentry is disabled when using this function.
 *
 * \param pwm A handle.
 * \param func A callback function of type \ref pwmd_inquire_cb_t which sets
 * the passphrase to be sent. This acts as a normal \ref pwmd_inquire() would.
 * \param user A user data pointer passed to the callback function \a func.
 * \return 0 on success or an error code.
 * \see \ref PWMD_OPTION_BASE64
 */
LIBPWMD_API gpg_error_t pwmd_save_inquire(pwm_t *pwm, pwmd_inquire_cb_t func,
	void *user);


/*! \brief Save a file on the pwmd server using the local pinentry.
 *
 * This will send the SAVE command like \ref pwmd_save() but will use a local
 * pinentry and not pwmd's pinentry.
 *
 * \param pwm A handle.
 * \return 0 on success or an error code.
 * \see \ref pinentry
 */
LIBPWMD_API gpg_error_t pwmd_save2(pwm_t *pwm);


/*! \brief Save a file on the pwmd server asynchronously (fork method).
 *
 * This is kind of a hybrid of \ref pwmd_save2() and \ref pwmd_save_async().
 * It will use the local pinentry asynchronously and also do the SAVE command
 * asynchronously.
 *
 * \process
 *
 * \param pwm A handle.
 * \return 0 on success or an error code.
 * \see pwmd_process(), \ref pinentry
 */
LIBPWMD_API gpg_error_t pwmd_save_async2(pwm_t *pwm);


/*! \brief Save changes to a file on the pwmd server (asynchronously).
 *
 * This will send the SAVE command to the pwmd server. The difference from
 * \ref pwmd_save() is that it will not block if a pinentry is needed for
 * passphrase input.
 *
 * \process
 *
 * \param pwm A handle.
 * \return 0 on success or an error code.
 * \see pwmd_process(), \ref pinentry
 */
LIBPWMD_API gpg_error_t pwmd_save_async(pwm_t *pwm);


/*! \brief Send a command to the pwmd server.
 *
 * Sends a command to the pwmd server. You should avoid sending the BYE
 * command here because the assuan context will be freed and bad things will
 * happen. Use \ref pwmd_close() instead. For commands that use an INQUIRE to
 * send data to the server (STORE and IMPORT), \ref pwmd_inquire() must be
 * used and not this function.
 *
 * \param pwm A handle.
 * \param[out] result The result of the command when successful which must be
 * freed with \ref pwmd_free().
 * \param cmd The command to send and any following arguments.
 * \return 0 on success or an error code.
 *
 * \note Not all commands return a result.
 */
LIBPWMD_API gpg_error_t pwmd_command(pwm_t *pwm, char **result,
	const char *cmd, ...);


/*! \brief Send a command to the pwmd server.
 *
 * Like \ref pwmd_command() but uses an argument pointer instead.
 *
 * \param pwm A handle.
 * \param[out] result The result of the command when successful which must be
 * freed with \ref pwmd_free().
 * \param cmd The command to send.
 * \param ap The arguments to \a cmd.
 * \return 0 on success or an error code.
 *
 * \note Not all commands return a result.
 */
LIBPWMD_API gpg_error_t pwmd_command_ap(pwm_t *pwm, char **result,
	const char *cmd, va_list ap);


/*! \brief Send data to a pwmd server.
 *
 * This lets commands that use an INQUIRE (STORE and IMPORT) send the data
 * to the server. Use this function rather than \ref pwmd_command() for these
 * pwmd commands.
 *
 * \param pwm A handle.
 * \param cmd The \ref pwmd(1) command to send including any options.
 * \param func A callback function of type \ref pwmd_inquire_cb_t which sets
 * the data to be sent.
 * \param user A user data pointer passed to the callback function \a func.
 * \return 0 on success or an error code.
 *
 * \see pwmd_inquire_cb_t
 */
LIBPWMD_API gpg_error_t pwmd_inquire(pwm_t *pwm, const char *cmd,
	pwmd_inquire_cb_t func, void *user);


/*! \brief Close a connection to the pwmd server.
 *
 * This will close the connection but keep any previously set options for the
 * specified handle.
 *
 * \param pwm A handle.
 * \return 0 on success or an error code.
 * \see \ref pwmd_close()
 */
LIBPWMD_API gpg_error_t pwmd_disconnect(pwm_t *pwm);


/*! \brief Close a handle.
 *
 * This will close the connection to a pwmd server and free any resources
 * associated with it.
 *
 * \param pwm A handle.
 * \return Nothing.
 * \see \ref pwmd_disconnect(), \ref pwmd_new()
 */
LIBPWMD_API void pwmd_close(pwm_t *pwm);


/*! \brief The type of connection a handle has.
 *
 * Useful when you want to know what kind of connection a handle has.
 *
 * \param pwm A handle.
 * \param[out] type The type of socket.
 * \return 0 on success or an error code.
 * \see pwmd_connect_url()
 */
LIBPWMD_API gpg_error_t pwmd_socket_type(pwm_t *pwm, pwmd_socket_t *type);


/*! \brief Free a previously allocated pointer.
 *
 * Use this function to free resources allocated by the other libpwmd memory
 * functions. Do not use it to free allocations made by other allocators.
 *
 * The difference between the standard free() and this function is that
 * this one will zero out the contents of the pointer before freeing it.
 *
 * \param ptr The pointer to deallocate.
 * \return Nothing.
 * \see pwmd_malloc(), pwmd_calloc(), pwmd_realloc(), pwmd_strdup(),
 * pwmd_process(), pwmd_command()
 */
LIBPWMD_API void pwmd_free(void *ptr);


/*! \brief A wrapper around malloc.
 *
 * Like malloc(), but lets libpwmd keep track of the pointer.
 *
 * \param size The number of bytes to allocate.
 * \return A newly allocated pointer or NULL if there wasn't enough memory.
 * \see malloc(3), pwmd_free()
 */
LIBPWMD_API void *pwmd_malloc(size_t size);


/*! \brief A wrapper around calloc().
 *
 * Like calloc(), but lets libpwmd keep track of the pointer.
 *
 * \param nmemb The number of elements to allocate.
 * \param size The number of bytes to allocate.
 * \return A newly allocated pointer or NULL if there wasn't enough memory.
 * \see calloc(3), pwmd_free()
 */
LIBPWMD_API void *pwmd_calloc(size_t nmemb, size_t size);


/*! \brief A wrapper around realloc().
 *
 * Like realloc(), but lets libpwmd keep track of the pointer.
 *
 * \note This function will try and allocate the entire \a size before freeing
 * the original pointer and returning the new one.
 *
 * \param ptr The pointer to reallocate.
 * \param size The new number of bytes to allocate.
 * \return A newly allocated pointer or NULL if there wasn't enough memory.
 * \see realloc(3), pwmd_free()
 */
LIBPWMD_API void *pwmd_realloc(void *ptr, size_t size);


/*! \brief A wrapper around strdup().
 *
 * Like strdup(), but lets libpwmd keep track of the pointer.
 *
 * \param str The string to duplicate.
 * \return A newly allocated character pointer or NULL if there wasn't
 * enough memory.
 * \see strdup(3), pwmd_free()
 */
LIBPWMD_API char *pwmd_strdup(const char *str);


/*! \brief Duplicate a formatted string.
 *
 * Like \ref sprintf(3) but returns an allocated string.
 *
 * \param fmt The formatted string and any following arguments.
 * \return A newly allocated character pointer or NULL if there wasn't
 * enough memory.
 * \see pwmd_free()
 */
LIBPWMD_API char *pwmd_strdup_printf(const char *fmt, ...);


/*! \def EPWMD_NO_FILE
 * \hideinitializer
 * \brief No data file has been opened.
 *
 * Some commands don't require an open data file but most do.
 */
#define EPWMD_NO_FILE		GPG_ERR_USER_1


/*! \def EPWMD_LIBXML_ERROR
 * \hideinitializer
 * \brief libxml2 error.
 *
 * This can be a memory allocation error or a parse error. The details of the
 * error cannot be obtained with libpwmd. You'd have to connect to the pwmd
 * socket and do the command directly to get the actual error.
 */
#define EPWMD_LIBXML_ERROR	GPG_ERR_USER_2


/*! \def EPWMD_FILE_MODIFIED
 * \hideinitializer
 * \brief The data file has been modified.
 *
 * Rather than process the next command this error is returned to prevent
 * overwriting new data which may have been saved by another client.
 */
#define EPWMD_FILE_MODIFIED	GPG_ERR_USER_3


/*! \def EPWMD_MAX
 * \hideinitializer
 * \if cond1
 * \internal
 * \endif
 * \brief libgpg-error error code offset.
 *
 * If you use your own libgpg-error codes then this should be the base of
 * them.
 */
#define EPWMD_MAX		GPG_ERR_USER_4


/*! \brief Return a description of an error code.
 *
 * \param code The error code to describe.
 * \return A character description of the error code.
 * \see pwmd_strerror_r()
 */
LIBPWMD_API const char *pwmd_strerror(gpg_error_t code);


/*! \brief Return a description of an error code (thread-safe).
 *
 * This is a thread-safe version of \ref pwmd_strerror().
 *
 * \param code The error code to describe.
 * \param[out] buf An allocated buffer to hold the error description.
 * \param size The size of the allocated buffer \a buf.
 *
 * \retval 0 Success.
 * \retval ERANGE \a size was not large enough to hold the entire description
 * and \a buf is set to the truncated error string.
 */
LIBPWMD_API int pwmd_strerror_r(gpg_error_t code, char *buf, size_t size);

#ifdef __cplusplus
}
#endif

#endif
