/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2007-2009 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <err.h>
#include <errno.h>
#include <string.h>
#include <libpwmd.h>
#include <assuan.h>
#ifdef DEBUG
#include <sys/select.h>
#endif
#include <fcntl.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#ifdef HAVE_GETOPT_LONG
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#else
#include "getopt_long.h"
#endif

#include "gettext.h"
#define N_(msgid)	gettext(msgid)

#include "mem.h"

#define DEFAULT_PORT	22
pwm_t *pwm;

static void show_error(gpg_error_t error)
{
    fprintf(stderr, "ERR %i %s\n", gpg_err_code(error), pwmd_strerror(error));
}

static void usage(const char *pn, int status)
{
    fprintf(status == EXIT_FAILURE ? stderr : stdout, N_(
	    "Read a PWMD protocol command from standard input.\n\n"
	    "Usage: pwmc [options] [file]\n"
#ifdef DEBUG
	    "   --debug <N>\n"
	    "       pinentry method (0=pwmd, 1=libpwmd, 2=pwmd async, "
		    "3=libpwmd async)\n"
#endif
	    "\n"
#ifdef WITH_TCP
	    "   --host, -h <hostname>\n"
	    "       connect to the specified hostname\n"
	    "\n"
	    "   --port\n"
	    "       alterate port (22)\n"
	    "\n"
	    "   --user\n"
	    "       SSH username (default is the invoking user)\n"
	    "\n"
	    "   --identity, -i <filename>\n"
	    "       SSH identity file\n"
	    "\n"
	    "   --known-hosts, -k <filename>\n"
	    "       known host's file (for server validation)\n"
	    "\n"
	    "   --get-hostkey, -g\n"
	    "       retrieve the remote SSH host key and exit\n"
	    "\n"
	    "   --ipv4, -4\n"
	    "       try connecting via IPv4 only\n"
	    "\n"
	    "   --ipv6, -6\n"
	    "       try connecting via IPv6 only\n"
	    "\n"
#endif
            "   --url <string>\n"
	    "       a url string to parse\n"
	    "\n"
	    "   --no-status\n"
	    "       disable showing of status messages from the server\n"
	    "\n"
	    "   --name, -n <string>\n"
	    "       set the client name\n"
	    "\n"
	    "   --socket <filename>\n"
	    "       local socket to connect to (~/.pwmd/socket)\n"
	    "\n"
	    "   --passphrase, -P <string>\n"
	    "       passphrase to use (disables pinentry use)\n"
	    "\n"
	    "   --timeout <seconds>\n"
	    "       pinentry timeout\n"
	    "\n"
	    "   --tries <N>\n"
	    "       number of pinentry tries before failing (3)\n"
	    "\n"
	    "   --pinentry <path>\n"
	    "       the full path to the pinentry binary (server default)\n"
	    "\n"
	    "   --ttyname, -y <path>\n"
	    "       tty that pinentry will use\n"
	    "\n"
	    "   --ttytype, -t <string>\n"
	    "       pinentry terminal type (default is TERM)\n"
	    "\n"
	    "   --display, -d\n"
	    "       pinentry display (default is DISPLAY)\n"
	    "\n"
	    "   --lc-ctype <string>\n"
	    "       locale setting for pinentry\n"
	    "\n"
	    "   --lc-messages <string>\n"
	    "       locale setting for pinentry\n"
	    "\n"

	    "   --local-pinentry\n"
	    "       force using a local pinentry\n"
	    "\n"
	    "   --output-fd <FD>\n"
	    "       redirect command output to the specified file descriptor\n"
	    "\n"
	    "   --inquire-fd <FD>\n"
	    "       read inquire data from the specified file descriptor\n"
	    "\n"
	    "   --save, -S\n"
	    "       send the SAVE command before exiting\n"
	    "\n"
	    "   --force-save\n"
	    "       like --save, but ask for a passphrase\n"
	    "\n"
	    "   --iterations, -I <N>\n"
	    "       encrypt with the specified number of iterations when saving\n"
	    "\n"
	    "   --version\n"
	    "   --help\n"));
    fprintf(status == EXIT_FAILURE ? stderr : stdout, N_(
		"\n"
		"A url string (specified with --url) may be in the form of:\n"
		"    file://[path/to/socket]\n"
#ifdef WITH_TCP
		"    ssh[46]://[username@]hostname[:port],identity,known_hosts\n"
#endif
		));
    exit(status);
}

struct inquire_s {
    FILE *fp;
    char *data;
};

static gpg_error_t do_inquire(void *data, const char *keyword, gpg_error_t rc,
	char **result, size_t *result_len)
{
    int c;
    static char buf[ASSUAN_LINELENGTH];
    char *p;
    size_t len = 0;
    struct inquire_s *inq = (struct inquire_s *)data;

    if (rc) {
	memset(buf, 0, sizeof(buf));
	return rc;
    }

    buf[0] = 0;
    p = buf;

    if (inq->data) {
	snprintf(buf, sizeof(buf), "%s", inq->data);
	pwmd_free(inq->data);
	inq->data = NULL;
	len = strlen(buf);
	p = buf + len;
    }

    while ((c = fgetc(inq->fp)) != EOF) {
	if (len == sizeof(buf)) {
	    ungetc(c, inq->fp);
	    break;
	}

	*p++ = c;
	len++;
    }

    if (!buf[0]) {
	memset(buf, 0, sizeof(buf));
	return GPG_ERR_EOF;
    }

    *result = buf;
    *result_len = len;
    return 0;
}

static int status_msg_cb(void *data, const char *line)
{
    fprintf(stderr, "%s\n", line);
    return 0;
}

static gpg_error_t process_cmd(pwm_t *pwm, char **result, int input)
{
    gpg_error_t rc;
    pwmd_async_t s;

    do {
	int i, n;
	fd_set rfds;
	int nfds = 5;
	pwmd_fd_t pfds[nfds];

	FD_ZERO(&rfds);
	rc = pwmd_get_fds(pwm, pfds, &nfds);

	if (rc)
	    return rc;

	if (!nfds)
	    return 0;

	for (i = 0, n = 0; i < nfds; i++) {
	    FD_SET(pfds[i].fd, &rfds);
	    n = pfds[i].fd > n ? pfds[i].fd : n;
	}

	if (input)
	    FD_SET(STDIN_FILENO, &rfds);

	nfds = select(n+1, &rfds, NULL, NULL, NULL);

	if (nfds == -1) {
	    rc = gpg_error_from_errno(errno);
	    return rc;
	}

	if (input && FD_ISSET(STDIN_FILENO, &rfds))
	    return 0;

	s = pwmd_process(pwm, &rc, result);
    } while (s == ASYNC_PROCESS);

    return rc;
}

int main(int argc, char *argv[])
{
    int opt;
    char *password = NULL;
    char *filename = NULL;
    char *socketpath = NULL;
    char command[ASSUAN_LINELENGTH], *p;
    int ret = EXIT_SUCCESS;
    gpg_error_t error;
    char *result = NULL;
    int save = 0;
    char *pinentry_path = NULL;
    char *display = NULL, *tty = NULL, *ttytype = NULL, *lcctype = NULL,
	 *lcmessages = NULL;
    int outfd = STDOUT_FILENO;
    FILE *outfp = stdout;
    int inquirefd = STDIN_FILENO;
    FILE *inquirefp = stdin;
    int show_status = 1;
    char *clientname = "pwmc";
    char *inquire = NULL;
    long iter = -2;
    int have_iter = 0;
    int timeout = 0;
    int local_pin = 0;
    int force_save = 0;
#ifdef WITH_TCP
    char *host = NULL;
    int port = DEFAULT_PORT;
    char *username = NULL;
    char *ident = NULL;
    char *known_hosts = NULL;
    int get = 0;
    int prot = PWMD_IP_ANY;
#endif
    int tries = 0;
    int local_tries = 3;
    int try = 0;
    pwmd_socket_t s;
#ifdef DEBUG
    int method = 0;
    fd_set rfds;
#endif
    char *url_string = NULL;
    /* The order is important. */
    enum {
#ifdef DEBUG
	OPT_DEBUG,
#endif
#ifdef WITH_TCP
	OPT_HOST, OPT_PORT, OPT_IDENTITY, OPT_KNOWN_HOSTS, OPT_USER,
	OPT_GET_HOSTKEY, OPT_IPV4, OPT_IPV6,
#endif
	OPT_URL, OPT_LOCAL, OPT_FORCE_SAVE, OPT_TTYNAME, OPT_TTYTYPE,
	OPT_DISPLAY, OPT_LC_CTYPE, OPT_LC_MESSAGES, OPT_TIMEOUT, OPT_TRIES,
	OPT_PINENTRY, OPT_PASSPHRASE, OPT_SOCKET, OPT_SAVE, OPT_ITERATIONS,
	OPT_OUTPUT_FD, OPT_INQUIRE_FD, OPT_NO_STATUS, OPT_NAME, OPT_VERSION,
	OPT_HELP,
    };
    const struct option long_opts[] = {
#ifdef DEBUG
        { "debug", 1, 0, 0 },
#endif
#ifdef WITH_TCP
	{ "host", 1, 0, 'h' },
	{ "port", 1, 0, 'p' },
	{ "identity", 1, 0, 'i' },
	{ "known-hosts", 1, 0, 'k' },
	{ "user", 1, 0, 'u' },
	{ "get-hostkey", 0, 0, 'g' },
	{ "ipv4", 0, 0, '4' },
	{ "ipv6", 0, 0, '6' },
#endif
	{ "url", 1, 0, 0 },
	{ "local-pinentry", 0, 0 },
	{ "force-save", 0, 0 },
	{ "ttyname", 1, 0, 'y' },
	{ "ttytype", 1, 0, 't' },
	{ "display", 1, 0, 'd' },
	{ "lc-ctype", 1, 0, 0 },
	{ "lc-messages", 1, 0, 0 },
	{ "timeout", 1, 0, 0 },
	{ "tries", 1, 0, 0 },
	{ "pinentry", 1, 0, 0 },
	{ "passphrase", 1, 0, 'P' },
	{ "socket", 1, 0, 0 },
	{ "save", 0, 0, 'S' },
	{ "iterations", 1, 0, 'I' },
	{ "output-fd", 1, 0, 0 },
	{ "inquire-fd", 1, 0, 0 },
	{ "no-status", 0, 0, 0 },
	{ "name", 1, 0, 'n' },
	{ "version", 0, 0, 0 },
	{ "help", 0, 0, 0 },
	{ 0, 0, 0, 0}
    };
#ifdef WITH_TCP
    const char *optstring = "46h:p:i:k:u:gy:t:d:P:I:Sn:";
#else
    const char *optstring = "y:t:d:P:I:Sn:";
#endif
    int opt_index = 0;

#ifdef ENABLE_NLS
    setlocale(LC_ALL, "");
    bindtextdomain("libpwmd", LOCALEDIR);
#endif

    while ((opt = getopt_long(argc, argv, optstring, long_opts, &opt_index)) != -1) {
	switch (opt) {
	    /* Handle long options without a short option part. */
	    case 0:
		switch (opt_index) {
#ifdef DEBUG
		    case OPT_DEBUG:
			method = atoi(optarg);

			if (method > 3)
			    method = 3;
			break;
#endif
		    case OPT_URL:
			url_string = optarg;
			break;
		    case OPT_LOCAL:
			local_pin = 1;
			break;
		    case OPT_FORCE_SAVE:
			save = force_save = 1;
			break;
		    case OPT_LC_CTYPE:
			lcctype = pwmd_strdup(optarg);
			break;
		    case OPT_LC_MESSAGES:
			lcmessages = pwmd_strdup(optarg);
			break;
		    case OPT_TIMEOUT:
			timeout = atoi(optarg);
			break;
		    case OPT_TRIES:
			tries = atoi(optarg);
			local_tries = tries;
			break;
		    case OPT_SOCKET:
			socketpath = pwmd_strdup(optarg);
			break;
		    case OPT_INQUIRE_FD:
			inquirefd = atoi(optarg);
			inquirefp = fdopen(inquirefd, "r");

			if (!inquirefp) {
			    pwmd_free(password);
			    err(EXIT_FAILURE, "%i", inquirefd);
			}
			break;
		    case OPT_OUTPUT_FD:
			outfd = atoi(optarg);
			outfp = fdopen(outfd, "w");

			if (!outfp) {
			    pwmd_free(password);
			    err(EXIT_FAILURE, "%i", outfd);
			}
			break;
		    case OPT_NO_STATUS:
			show_status = 0;
			break;
		    case OPT_VERSION:
			pwmd_free(password);
			printf("%s (pwmc)\n%s\n\n"
				"Compile-time features:\n"
#ifdef WITH_TCP
				"+SSH "
#else
				"-SSH "
#endif
#ifdef WITH_PINENTRY
				"+PINENTRY "
#else
				"-PINENTRY "
#endif
#ifdef WITH_QUALITY
				"+QUALITY "
#else
				"-QUALITY "
#endif
#ifdef MEM_DEBUG
				"+MEM_DEBUG "
#else
				"-MEM_DEBUG "
#endif
				"\n"
				, PACKAGE_STRING, PACKAGE_BUGREPORT);
			exit(EXIT_SUCCESS);
		    case OPT_PINENTRY:
			pinentry_path = optarg;
			break;
		    case OPT_HELP:
			usage(argv[0], EXIT_SUCCESS);
		    default:
			usage(argv[0], EXIT_FAILURE);
		}

		break;
#ifdef WITH_TCP
	    case '4':
		prot = PWMD_IPV4;
		break;
	    case '6':
		prot = PWMD_IPV6;
		break;
	    case 'h':
		host = pwmd_strdup(optarg);
		break;
	    case 'p':
		port = atoi(optarg);
		break;
	    case 'i':
		ident = pwmd_strdup(optarg);
		break;
	    case 'u':
		username = pwmd_strdup(optarg);
		break;
	    case 'k':
		known_hosts = pwmd_strdup(optarg);
		break;
	    case 'g':
		get = 1;
		break;
#endif
	    case 'y':
		tty = optarg;
		break;
	    case 't':
		ttytype = optarg;
		break;
	    case 'd':
		display = optarg;
		break;
	    case 'S':
		save = 1;
		break;
	    case 'I':
		iter = strtol(optarg, NULL, 10);
		have_iter = 1;
		break;
	    case 'P':
		password = pwmd_strdup(optarg);
		memset(optarg, 0, strlen(optarg));
		break;
	    case 'n':
		clientname = optarg;
		break;
	    default:
		pwmd_free(password);
		usage(argv[0], EXIT_FAILURE);
	}
    }

#ifdef DEBUG
    if (!url_string) {
#endif
#ifdef WITH_TCP
    if (host && !get && (!known_hosts || !ident)) {
	pwmd_free(password);
	usage(argv[0], EXIT_FAILURE);
    }

    if (get && !host) {
	pwmd_free(password);
	usage(argv[0], EXIT_FAILURE);
    }
#endif
#ifdef DEBUG
    }
#endif

    filename = argv[optind];
    pwmd_init();
    pwm = pwmd_new(clientname);
#ifdef DEBUG
    FD_ZERO(&rfds);
#endif
    
#ifdef WITH_TCP
    if (host) {
	if (prot != PWMD_IP_ANY) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_IP_VERSION, prot);

	    if (error)
		goto done;
	}

#ifdef DEBUG
	if (method >= 2) {
	    if (get) {
		char *hostkey;

		error = pwmd_get_hostkey_async(pwm, host, port);

		if (error)
		    errx(EXIT_FAILURE, "%s: %s", host, pwmd_strerror(error));

		error = process_cmd(pwm, &hostkey, 0);

		if (error)
		    goto done;

		printf("%s\n", hostkey);
		pwmd_free(hostkey);
		pwmd_free(password);
		pwmd_close(pwm);
		exit(EXIT_SUCCESS);
	    }

	    if (url_string)
		error = pwmd_connect_url_async(pwm, url_string);
	    else
		error = pwmd_ssh_connect_async(pwm, host, port, ident, username,
			known_hosts);

	    if (error)
		goto done;

	    error = process_cmd(pwm, NULL, 0);

	    if (error)
		goto done;
	}
	else {
#endif
	    if (get) {
		char *hostkey;

		error = pwmd_get_hostkey(pwm, host, port, &hostkey);

		if (error)
		    goto done;

		printf("%s\n", hostkey);
		pwmd_free(hostkey);
		pwmd_free(password);
		pwmd_close(pwm);
		exit(EXIT_SUCCESS);
	    }

	    if (url_string)
		error = pwmd_connect_url(pwm, url_string);
	    else
		error = pwmd_ssh_connect(pwm, host, port, ident, username, known_hosts);

	    if (error)
		goto done;
#ifdef DEBUG
	}
#endif
    }
    else {
#endif
	if (url_string)
	    error = pwmd_connect_url(pwm, url_string);
	else
	    error = pwmd_connect(pwm, socketpath);

	if (error)
	    goto done;
#ifdef WITH_TCP
    }
#endif

    error = pwmd_socket_type(pwm, &s);

    if (error)
	goto done;

    if (s == PWMD_SOCKET_SSH && force_save && !local_pin) {
	error = GPG_ERR_WRONG_KEY_USAGE;
	goto done;
    }

    if (have_iter) {
	error = pwmd_command(pwm, &result, "VERSION");

	if (error)
	    goto done;

	pwmd_free(result);

	if (iter < 0) {
	    pwmd_free(password);
	    pwmd_close(pwm);
	    usage(argv[0], EXIT_FAILURE);
	}
    }

    if (timeout > 0) {
	error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TIMEOUT, timeout);

	if (error)
	    goto done;
    }

    if (local_pin && filename) {
	error = pwmd_command(pwm, NULL, "ISCACHED %s", filename);

	if (error == GPG_ERR_NOT_FOUND) {
local_password:
	    if (try++ == local_tries)
		goto done;

	    if (password)
		pwmd_free(password);

	    password = NULL;

	    if (try > 1)
		error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TIMEOUT, 0);

	    error = pwmd_getpin(pwm, filename, &password,
		    try > 1 ? PWMD_PINENTRY_OPEN_FAILED : PWMD_PINENTRY_OPEN);

	    if (error)
		goto done;

	    error = pwmd_setopt(pwm, PWMD_OPTION_PASSPHRASE, password);

	    if (error)
		goto done;

	    if (try > 1)
		goto do_open;
	}
	else if (error && error != GPG_ERR_ENOENT)
	    goto done;
    }

    if (password) {
	error = pwmd_setopt(pwm, PWMD_OPTION_PASSPHRASE, password);

	if (error)
	    goto done;
    }
    else {
	if (pinentry_path) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_PATH, pinentry_path);

	    if (error)
		goto done;
	}

	if (display) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_DISPLAY, display);

	    if (error)
		goto done;
	}

	if (tty) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TTY, tty);

	    if (error)
		goto done;
	}

	if (ttytype) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TERM, ttytype);

	    if (error)
		goto done;
	}

	if (lcctype) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_LC_CTYPE, lcctype);

	    if (error)
		goto done;
	}

	if (lcmessages) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_LC_MESSAGES,
		    lcmessages);

	    if (error)
		goto done;
	}

	if (tries > 0) {
	    error = pwmd_setopt(pwm, PWMD_OPTION_PINENTRY_TRIES, tries);

	    if (error)
		goto done;
	}
    }

    if (show_status) {
	error = pwmd_setopt(pwm, PWMD_OPTION_STATUS_CB, status_msg_cb);

	if (error)
	    goto done;
    }

do_open:
    if (filename) {
#ifdef DEBUG
	switch (method) {
	    case 0:
		error = pwmd_open(pwm, filename);
		break;
	    case 1:
		error = pwmd_open2(pwm, filename);
		break;
	    case 2:
		error = pwmd_open_async(pwm, filename);

		break;
	    case 3:
		error = pwmd_open_async2(pwm, filename);
		break;
	}

	if (error && local_pin && error == GPG_ERR_INV_PASSPHRASE)
	    goto local_password;

	if (error)
	    goto done;

	if (method >= 2)
	    error = process_cmd(pwm, &result, 0);
#else
	error = pwmd_open(pwm, filename);

	if (error && local_pin && error == GPG_ERR_INV_PASSPHRASE)
	    goto local_password;
#endif

	if (error)
	    goto done;
    }

    if (filename) {
	error = pwmd_command(pwm, &result, "LOCK");

	if (error)
	    goto done;
    }

    fcntl(STDIN_FILENO, F_SETFL, O_NONBLOCK);

    for (;;) {
	ssize_t n;

	error = process_cmd(pwm, NULL, 1);

	if (error)
	    goto done;

	n = read(STDIN_FILENO, command, sizeof(command));

	if (n == -1) {
	    if (errno == EAGAIN)
		continue;

	    error = gpg_error_from_errno(errno);
	    goto done;
	}
	else if (!n)
	    goto done;

	command[n] = 0;

	if (n && command[strlen(command)-1] == '\n')
	    command[strlen(command)-1] = 0;

	p = command;
	break;
    }

    if (!p || !*p)
	goto done;

    /*
     * This is a known INQUIRE command. We use pwmd_inquire() to send the
     * data from the do_inquire() callback function.
     */
    if (strncasecmp(p, "STORE ", 6) == 0) {
	p += 6;
	inquire = (char *)"STORE";
    }
    else if (strncasecmp(p, "IMPORT ", 7) == 0) {
	p += 7;
	inquire = (char *)"IMPORT";
    }

    if (inquire) {
	struct inquire_s *inq = (struct inquire_s *)pwmd_malloc(sizeof(struct inquire_s));

	if (!inq) {
	    error = gpg_error_from_errno(ENOMEM);
	    goto done;
	}

	inq->data = pwmd_strdup(p);
	inq->fp = inquirefp;
	error = pwmd_inquire(pwm, inquire, do_inquire, inq);
	pwmd_free(inq);
	goto done;
    }

    if (strcasecmp(p, "BYE") == 0)
	goto done;

    error = pwmd_command(pwm, &result, command);
    memset(command, 0, sizeof(command));

    if (error)
	goto done;

    if (result) {
	fwrite(result, 1, strlen(result), outfp);
	pwmd_free(result);
    }

done:
    memset(command, 0, sizeof(command));
    pwmd_free(password);
    password = NULL;

    if (!error && save && filename) {
	if (iter != -2) {
	    error = pwmd_command(pwm, &result, "SET ITERATIONS=%i", iter);

	    if (error)
		goto done;
	}

	if (local_pin) {
	    char *p1;
again:
	    if (!force_save) {
		error = pwmd_command(pwm, NULL, "ISCACHED %s", filename);

		if (error && error != GPG_ERR_NOT_FOUND &&
			error != GPG_ERR_ENOENT)
		    goto done;
		else if (!error)
		    goto do_save;
	    }

	    error = pwmd_getpin(pwm, filename, &p1, PWMD_PINENTRY_SAVE);

	    if (error)
		goto done;

	    error = pwmd_getpin(pwm, filename, &password,
		    PWMD_PINENTRY_SAVE_CONFIRM);

	    if (error) {
		pwmd_free(p1);
		goto done;
	    }

	    if ((p1 || password) && ((!p1 && password) || (!password && p1) ||
		    strcmp(p1, password))) {
		pwmd_free(p1);
		pwmd_free(password);
		password = NULL;
		goto again;
	    }

	    if (p1)
		pwmd_free(p1);

	    error = pwmd_setopt(pwm, PWMD_OPTION_PASSPHRASE, password);

	    if (password)
		pwmd_free(password);

	    if (error)
		goto done;
	}

	if (force_save) {
	    error = pwmd_command(pwm, NULL, "CLEARCACHE %s", filename);

	    if (error)
		goto done;

	    if (!local_pin) {
		error = pwmd_setopt(pwm, PWMD_OPTION_PASSPHRASE, NULL);

		if (error)
		    goto done;
	    }
	}

do_save:
#ifdef DEBUG
	switch (method) {
	    case 0:
		error = pwmd_save(pwm);
		break;
	    case 1:
		error = pwmd_save2(pwm);
		break;
	    case 2:
		error = pwmd_save_async(pwm);
		break;
	    case 3:
		error = pwmd_save_async2(pwm);
		break;
	}

	if (!error && method >= 2)
	    error = process_cmd(pwm, NULL, 0);

#else
	error = pwmd_save(pwm);
#endif
    }

    if (!error && filename)
	error = pwmd_command(pwm, &result, "UNLOCK");

    if (error) {
	show_error(error);
	ret = EXIT_FAILURE;
    }

    pwmd_close(pwm);

    if (socketpath)
	pwmd_free(socketpath);

    exit(ret);
}
