/* vim:tw=78:ts=8:sw=4:set ft=c:  */
/*
    Copyright (C) 2006-2009 Ben Kibbey <bjk@luxsci.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <err.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <signal.h>
#include <stdarg.h>
#include <string.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <pwd.h>
#include <time.h>
#include <sys/types.h>
#include <limits.h>
#include <sys/select.h>
#include <libpwmd.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_ASSUAN_H
#include <assuan.h>
#endif

#include "mem.h"
#include "misc.h"
#include "types.h"

#ifdef WITH_PINENTRY
#include "pinentry.h"
#endif

#ifdef WITH_TCP
#include "ssh.h"
#endif

static gpg_error_t send_pinentry_options(pwm_t *pwm);
static gpg_error_t _pwmd_process(pwm_t *pwm);
static gpg_error_t set_pinentry_retry(pwm_t *pwm);
static gpg_error_t get_custom_passphrase(pwm_t *pwm, char **result);

static const char *_pwmd_strerror(gpg_error_t e)
{
    gpg_err_code_t code = gpg_err_code(e);

    if (code >= GPG_ERR_USER_1 && code < gpg_err_code(EPWMD_MAX)) {
	switch (code) {
	    case GPG_ERR_USER_1:
		return N_("No file is open");
	    case GPG_ERR_USER_2:
		return N_("General LibXML error");
	    case GPG_ERR_USER_3:
		return N_("File modified");
	    default:
		return N_("Unknown error");
	}
    }

    return NULL;
}

const char *pwmd_strerror(gpg_error_t code)
{
    const char *p = _pwmd_strerror(code);

    return p ? p : gpg_strerror(code);
}

int pwmd_strerror_r(gpg_error_t code, char *buf, size_t size)
{
    const char *p = _pwmd_strerror(code);

    if (p) {
	snprintf(buf, size, "%s", p);

	if (strlen(p) > size)
	    return ERANGE;

	return 0;
    }

    return gpg_strerror_r(code, buf, size);
}

gpg_error_t pwmd_init()
{
    static int initialized;

    if (initialized)
	return 0;

#ifndef MEM_DEBUG
    _xmem_init();
#endif
#ifdef ENABLE_NLS
    bindtextdomain("libpwmd", LOCALEDIR);
#endif
    gpg_err_init();
    assuan_set_malloc_hooks(pwmd_malloc, pwmd_realloc, pwmd_free);
    assuan_set_assuan_err_source(GPG_ERR_SOURCE_DEFAULT);
    initialized = 1;
    return 0;
}

gpg_error_t _connect_finalize(pwm_t *pwm)
{
    int active[2];
    int n = assuan_get_active_fds(pwm->ctx, 0, active, N_ARRAY(active));
    gpg_error_t rc;

    if (n <= 0)
	return GPG_ERR_EBADFD;

    pwm->fd = active[0];
    fcntl(pwm->fd, F_SETFL, 0);
#ifdef WITH_PINENTRY
    pwm->pid = -1;
#endif
    assuan_set_pointer(pwm->ctx, pwm);

#ifdef WITH_TCP
    // Until X11 forwarding is supported, disable the remote pwmd pinentry.
    if (pwm->tcp_conn) {
	pwm->tcp_conn->state = SSH_NONE;
	rc = pwmd_command(pwm, NULL, "SET ENABLE_PINENTRY=0");

	if (rc)
	    return rc;
    }
#endif

    if (pwm->name) {
	rc = pwmd_command(pwm, NULL, "SET NAME=%s", pwm->name);

	if (rc)
	    return rc;
    }

    return 0;
}

static gpg_error_t _pwmd_connect_url(pwm_t *pwm, const char *url, int async)
{
    char *p = (char *)url;
    gpg_error_t rc;

    if (!pwm)
	return GPG_ERR_INV_ARG;

    if (!p || !strncmp(p, "file://", 7) || !strncmp(p, "local://", 8)) {
	if (p) {
	    if (!strncmp(p, "file://", 7))
		p += 7;
	    else
		p += 8;
	}

	goto connect_uds;
    }
    else if (!strncmp(p, "ssh://", 6) || !strncmp(p, "ssh6://", 7) ||
	    !strncmp(p, "ssh4://", 7)) {
#ifndef WITH_TCP
	return GPG_ERR_NOT_IMPLEMENTED;
#else
	char *host = NULL;
	int port = -1;
	char *identity = NULL;
	char *known_hosts = NULL;
	char *username = NULL;

	if (!strncmp(p, "ssh6://", 7)) {
	    rc = pwmd_setopt(pwm, PWMD_OPTION_IP_VERSION, PWMD_IPV6);
	    p += 7;
	}
	else if (!strncmp(p, "ssh4://", 7)) {
	    rc = pwmd_setopt(pwm, PWMD_OPTION_IP_VERSION, PWMD_IPV4);
	    p += 7;
	}
	else {
	    rc = pwmd_setopt(pwm, PWMD_OPTION_IP_VERSION, PWMD_IP_ANY);
	    p += 6;
	}

	if (rc)
	    return rc;

	rc = _parse_ssh_url(p, &host, &port, &username, &identity,
		&known_hosts);

	if (rc)
	    goto fail;

	if (async)
	    rc = pwmd_ssh_connect_async(pwm, host, port, identity, username,
		    known_hosts);
	else
	    rc = pwmd_ssh_connect(pwm, host, port, identity, username,
		    known_hosts);

fail:
	if (host)
	    pwmd_free(host);

	if (username)
	    pwmd_free(username);

	if (identity)
	    pwmd_free(identity);

	if (known_hosts)
	    pwmd_free(known_hosts);

	return rc;
#endif
    }

connect_uds:
    rc = pwmd_connect(pwm, p);
    pwm->state = ASYNC_DONE;
    return rc;
}

gpg_error_t pwmd_connect_url(pwm_t *pwm, const char *url)
{
    return _pwmd_connect_url(pwm, url, 0);
}

gpg_error_t pwmd_connect_url_async(pwm_t *pwm, const char *url)
{
    return _pwmd_connect_url(pwm, url, 1);
}

gpg_error_t pwmd_connect(pwm_t *pwm, const char *path)
{
    char *socketpath = NULL;
    assuan_context_t ctx;
    struct passwd pw;
    char *pwbuf;
    gpg_error_t rc;

    if (!pwm)
	return GPG_ERR_INV_ARG;

    pwbuf = _getpwuid(&pw);

    if (!pwbuf)
	return gpg_error_from_errno(errno);

    if (!path || !*path)
	socketpath = pwmd_strdup_printf("%s/.pwmd/socket", pw.pw_dir);
    else
	socketpath = _expand_homedir((char *)path, &pw);

    pwmd_free(pwbuf);

    if (!socketpath)
	return gpg_error_from_errno(ENOMEM);

    rc = assuan_socket_connect_ext(&ctx, socketpath, -1, 0);
    pwmd_free(socketpath);

    if (rc)
	return gpg_err_code(rc);

    pwm->ctx = ctx;
    return _connect_finalize(pwm);
}

static void disconnect(pwm_t *pwm)
{
    if (!pwm || !pwm->ctx)
	return;

    assuan_disconnect(pwm->ctx);
    pwm->ctx = NULL;
    pwm->fd = -1;
}

void pwmd_close(pwm_t *pwm)
{
    if (!pwm)
	return;

    disconnect(pwm);

    if (pwm->password)
	pwmd_free(pwm->password);

    if (pwm->title)
	pwmd_free(pwm->title);

    if (pwm->desc)
	pwmd_free(pwm->desc);

    if (pwm->prompt)
	pwmd_free(pwm->prompt);

    if (pwm->pinentry_tty)
	pwmd_free(pwm->pinentry_tty);

    if (pwm->pinentry_display)
	pwmd_free(pwm->pinentry_display);

    if (pwm->pinentry_term)
	pwmd_free(pwm->pinentry_term);

    if (pwm->lcctype)
	pwmd_free(pwm->lcctype);

    if (pwm->lcmessages)
	pwmd_free(pwm->lcmessages);

    if (pwm->filename)
	pwmd_free(pwm->filename);

    if (pwm->name)
	pwmd_free(pwm->name);

#ifdef WITH_TCP
    if (pwm->tcp_conn)
	_free_ssh_conn(pwm->tcp_conn);
#endif

#ifdef WITH_PINENTRY
    if (pwm->pctx)
	_pinentry_disconnect(pwm);
#endif

    pwmd_free(pwm);
}

gpg_error_t pwmd_ssh_connect_async(pwm_t *pwm, const char *host, int port,
	const char *identity, const char *user, const char *known_hosts)
{
#ifndef WITH_TCP
    return GPG_ERR_NOT_IMPLEMENTED;
#else
    return _do_pwmd_ssh_connect_async(pwm, host, port, identity, user,
	    known_hosts, ASYNC_CMD_CONNECT);
#endif
}

gpg_error_t pwmd_ssh_connect(pwm_t *pwm, const char *host, int port,
	const char *identity, const char *user, const char *known_hosts)
{
#ifndef WITH_TCP
    return GPG_ERR_NOT_IMPLEMENTED;
#else
    return _do_pwmd_ssh_connect(pwm, host, port, identity, user, known_hosts, 0);
#endif
}

gpg_error_t pwmd_get_hostkey(pwm_t *pwm, const char *host, int port,
	char **result)
{
#ifndef WITH_TCP
    return GPG_ERR_NOT_IMPLEMENTED;
#else
    char *hostkey;
    gpg_error_t rc;

    rc = _do_pwmd_ssh_connect(pwm, host, port, NULL, NULL, NULL, 1);

    if (rc)
	return rc;

    hostkey = pwmd_strdup(pwm->tcp_conn->hostkey);

    if (!hostkey)
	rc = gpg_error_from_errno(ENOMEM);

    *result = hostkey;
    return rc;
#endif
}

gpg_error_t pwmd_get_hostkey_async(pwm_t *pwm, const char *host, int port)
{
#ifndef WITH_TCP
    return GPG_ERR_NOT_IMPLEMENTED;
#else
    return _do_pwmd_ssh_connect_async(pwm, host, port, NULL, NULL, NULL,
	    ASYNC_CMD_HOSTKEY);
#endif
}

static int inquire_realloc_cb(void *data, const void *buffer, size_t len)
{
    membuf_t *mem = (membuf_t *)data;
    void *p;

    if (!buffer)
	return 0;

    if ((p = pwmd_realloc(mem->buf, mem->len + len)) == NULL)
	return gpg_error_from_errno(ENOMEM);

    mem->buf = p;
    memcpy((char *)mem->buf + mem->len, buffer, len);
    mem->len += len;
    return 0;
}

static int inquire_cb(void *data, const char *keyword)
{
    pwm_t *pwm = (pwm_t *)data;
    gpg_error_t rc = 0;
#ifdef WITH_LIBPTH
    int flags = pth_fdmode(pwm->fd, PTH_FDMODE_POLL);
#else
    int flags = fcntl(pwm->fd, F_GETFL);
#endif

    /* Shouldn't get this far without a callback. */
    if (!pwm->inquire_func)
	return GPG_ERR_INV_ARG;

    for (;;) {
	char *result = NULL;
	size_t len;
	gpg_error_t arc;

	rc = pwm->inquire_func(pwm->inquire_data, keyword, rc, &result, &len);
	rc = gpg_err_code(rc);

	if (rc == GPG_ERR_EOF || !rc) {
	    if (len <= 0 || !result) {
		rc = 0;
		break;
	    }

	    arc = assuan_send_data(pwm->ctx, result, len);
	    arc = gpg_err_code(arc);

	    if (rc == GPG_ERR_EOF) {
		rc = arc;
		break;
	    }

	    rc = arc;
	}
	else if (rc)
	    break;

	if (!rc) {
	    /* Set to non-blocking so _pwmd_process() can return. */
#ifdef WITH_LIBPTH
	    pth_fdmode(pwm->fd, PTH_FDMODE_NONBLOCK);
#else
	    fcntl(pwm->fd, F_SETFL, O_NONBLOCK);
#endif
	    rc = _pwmd_process(pwm);
#ifdef WITH_LIBPTH
	    pth_fdmode(pwm->fd, flags);
#else
	    fcntl(pwm->fd, F_SETFL, flags);
#endif
	}
    }

#ifdef WITH_LIBPTH
    pth_fdmode(pwm->fd, flags);
#else
    fcntl(pwm->fd, F_SETFL, flags);
#endif
    return gpg_err_code(rc);
}

static gpg_error_t do_nb_command(pwm_t *pwm, const char *cmd, ...)
{
    char *buf;
    gpg_error_t rc;
    va_list ap;
    int len;

    if (pwm->state == ASYNC_DONE)
	pwm->state = ASYNC_INIT;

    if (pwm->state != ASYNC_INIT)
	return GPG_ERR_INV_STATE;

    buf = pwmd_malloc(ASSUAN_LINELENGTH+1);

    if (!buf)
	return gpg_error_from_errno(ENOMEM);

    va_start(ap, cmd);
    len = vsnprintf(buf, ASSUAN_LINELENGTH+1, cmd, ap);
    va_end(ap);

    if (len >= ASSUAN_LINELENGTH+1) {
	pwmd_free(buf);
	return GPG_ERR_LINE_TOO_LONG;
    }

    rc = assuan_write_line(pwm->ctx, buf);
    pwmd_free(buf);

    if (!rc)
	pwm->state = ASYNC_PROCESS;

    return gpg_err_code(rc);
}

gpg_error_t pwmd_open_async(pwm_t *pwm, const char *filename)
{
    char *p = NULL;
    const char *f = NULL;
    gpg_error_t rc;

    if (!pwm || !filename)
	return GPG_ERR_INV_ARG;

    if (!pwm->ctx)
	return GPG_ERR_INV_STATE;

    if (pwm->cmd != ASYNC_CMD_NONE)
	return GPG_ERR_ASS_NESTED_COMMANDS;

    if (pwm->lastcmd == ASYNC_CMD_NONE) {
	pwm->pin_try = 0;

	if (pwm->filename)
	    pwmd_free(pwm->filename);

	pwm->filename = pwmd_strdup(filename);

	if (!pwm->filename)
	    return gpg_error_from_errno(ENOMEM);

	gpg_error_t rc = send_pinentry_options(pwm);

	if (rc)
	    return rc;

	rc = get_custom_passphrase(pwm, &p);

	if (rc && rc != GPG_ERR_NO_DATA)
	    return rc;

	f = filename;
    }
#ifdef WITH_PINENTRY
    else if (pwm->lastcmd == ASYNC_CMD_OPEN2) {
	p = pwm->_password;
	f = pwm->filename;
    }
#endif
    else if (pwm->lastcmd == ASYNC_CMD_OPEN) {
	rc = set_pinentry_retry(pwm);

	if (rc)
	    return rc;

	p = pwm->password;
	f = filename;
    }
    else
	return GPG_ERR_INV_STATE;

    pwm->cmd = ASYNC_CMD_OPEN;
    return do_nb_command(pwm, "OPEN %s %s", f, p ? p : "");
}

gpg_error_t pwmd_save_async(pwm_t *pwm)
{
    char *p = NULL;

    if (!pwm)
	return GPG_ERR_INV_ARG;

    if (!pwm->ctx)
	return GPG_ERR_INV_STATE;

    if (pwm->cmd != ASYNC_CMD_NONE)
	return GPG_ERR_ASS_NESTED_COMMANDS;

    if (pwm->lastcmd != ASYNC_CMD_SAVE2) {
	gpg_error_t rc = send_pinentry_options(pwm);

	if (rc)
	    return rc;

	rc = get_custom_passphrase(pwm, &p);

	if (rc && rc != GPG_ERR_NO_DATA)
	    return rc;
    }
#ifdef WITH_PINENTRY
    else
	p = pwm->_password;
#endif

    pwm->cmd = ASYNC_CMD_SAVE;
    return do_nb_command(pwm, "SAVE %s", p ? p : "");
}

static gpg_error_t parse_assuan_line(pwm_t *pwm)
{
    gpg_error_t rc;
    char *line;
    size_t len;

    rc = assuan_read_line(pwm->ctx, &line, &len);
    
    if (!rc) {
	if (line[0] == 'O' && line[1] == 'K' &&
		(line[2] == 0 || line[2] == ' ')) {
	    pwm->state = ASYNC_DONE;
	}
	else if (line[0] == '#') {
	}
	else if (line[0] == 'S' && (line[1] == 0 || line[1] == ' ')) {
	    if (pwm->status_func) {
		rc = pwm->status_func(pwm->status_data,
			line[1] == 0 ? line+1 : line+2);
	    }
	}
	else if (line[0] == 'E' && line[1] == 'R' && line[2] == 'R' &&
		(line[3] == 0 || line[3] == ' ')) {
	    line += 4;
	    rc = atoi(line);
	    pwm->state = ASYNC_DONE;
	}
    }

    return gpg_err_code(rc);
}

gpg_error_t pwmd_pending_line(pwm_t *pwm)
{
    if (!pwm)
	return GPG_ERR_INV_ARG;

    if (!pwm->ctx)
	return GPG_ERR_INV_STATE;

    return assuan_pending_line(pwm->ctx) ? 0 : GPG_ERR_NO_DATA;
}

static pwmd_async_t reset_async(pwm_t *pwm, int done)
{
    pwm->state = ASYNC_INIT;
    pwm->cmd = pwm->lastcmd = ASYNC_CMD_NONE;
    
#ifdef WITH_PINENTRY
    if (pwm->nb_fd != -1) {
	close(pwm->nb_fd);
	pwm->nb_fd = -1;
    }

    if (pwm->_password) {
	pwmd_free(pwm->_password);
	pwm->_password = NULL;
    }
#endif
#ifdef WITH_TCP
    if (done && pwm->tcp_conn) {
	_free_ssh_conn(pwm->tcp_conn);
	pwm->tcp_conn = NULL;
    }
#endif

    return ASYNC_DONE;
}

/*
 * Used for processing status messages when not in an async command and for
 * waiting for the result from pwmd_open_async() and pwmd_save_async().
 */
static gpg_error_t _pwmd_process(pwm_t *pwm)
{
    gpg_error_t rc = 0;
    fd_set fds;
    struct timeval tv = {0, 0};
    int n;

    FD_ZERO(&fds);
    FD_SET(pwm->fd, &fds);
#ifdef WITH_LIBPTH
    n = pth_select(pwm->fd+1, &fds, NULL, NULL, &tv);
#else
    n = select(pwm->fd+1, &fds, NULL, NULL, &tv);
#endif

    if (n == -1)
	return gpg_error_from_syserror();

    if (n > 0) {
	if (FD_ISSET(pwm->fd, &fds))
	    rc = parse_assuan_line(pwm);
    }

    while (!rc && assuan_pending_line(pwm->ctx))
	rc = parse_assuan_line(pwm);

    return gpg_err_code(rc);
}

static void reset_handle(pwm_t *h)
{
    h->fd = -1;
#ifdef WITH_PINENTRY
    if (h->pctx)
	_pinentry_disconnect(h);

    h->nb_fd = -1;
#endif
    h->pin_try = 0;
    reset_async(h, 0);
}

gpg_error_t pwmd_disconnect(pwm_t *pwm)
{
    if (!pwm)
	return GPG_ERR_INV_ARG;

#ifdef WITH_TCP
    if (pwm->fd == -1 && pwm->tcp_conn && pwm->tcp_conn->fd == -1)
#else
    if (pwm->fd == -1)
#endif
	return GPG_ERR_INV_STATE;

    if (pwm->fd != 1)
	disconnect(pwm);
#ifdef WITH_TCP
    else
	_ssh_disconnect(pwm);
#endif

    reset_handle(pwm);
    return 0;
}

pwmd_async_t pwmd_process(pwm_t *pwm, gpg_error_t *rc, char **result)
{
#if defined(WITH_PINENTRY) || defined(WITH_TCP)
    fd_set fds;
    int n;
    struct timeval tv = {0, 0};
#endif

    if (result)
	*result = NULL;

    if (!rc)
	return GPG_ERR_INV_ARG;

    *rc = 0;

    if (!pwm) {
	*rc = GPG_ERR_INV_ARG;
	return ASYNC_DONE;
    }
    else if (!pwm->ctx) {
	switch (pwm->cmd) {
	    default:
		*rc = GPG_ERR_INV_STATE;
		return ASYNC_DONE;
#ifdef WITH_TCP
	    case ASYNC_CMD_DNS:
	    case ASYNC_CMD_CONNECT:
	    case ASYNC_CMD_HOSTKEY:
		break;
#endif
	}
    }

    /* When not in a command, this will let libassuan process status messages
     * by calling PWMD_OPTION_STATUS_FUNC. The client can poll the file
     * descriptor returned by pwmd_get_fd() to determine when this should be
     * called or call pwmd_pending_line() to determine whether a buffered line
     * needs to be processed. */
    if (pwm->cmd == ASYNC_CMD_NONE) {
	*rc = _pwmd_process(pwm);
	return ASYNC_DONE;
    }

    /* Fixes pwmd_open/save_async2() when there is a cached or new file. */
    if (pwm->state == ASYNC_DONE) {
	*rc = _pwmd_process(pwm);
	return reset_async(pwm, 0);
    }

    if (pwm->state != ASYNC_PROCESS) {
	*rc = GPG_ERR_INV_STATE;
	return ASYNC_DONE;
    }

#ifdef WITH_TCP
    if (pwm->cmd == ASYNC_CMD_DNS) {
	fd_set rfds, wfds;

	if (pwm->tcp_conn->rc) {
	    *rc = pwm->tcp_conn->rc;
	    return reset_async(pwm, 1);
	}

	FD_ZERO(&rfds);
	FD_ZERO(&wfds);
	n = ares_fds(pwm->tcp_conn->chan, &rfds, &wfds);

	/* Shouldn't happen. */
	if (!n)
	    return pwm->state;

#ifdef WITH_LIBPTH
	n = pth_select(n, &rfds, &wfds, NULL, &tv);
#else
	n = select(n, &rfds, &wfds, NULL, &tv);
#endif

	if (n < 0) {
	    *rc = gpg_error_from_syserror();
	    return reset_async(pwm, 1);
	}

	if (n > 0)
	    ares_process(pwm->tcp_conn->chan, &rfds, &wfds);

	return pwm->state;
    }
    else if (pwm->cmd == ASYNC_CMD_CONNECT) {
	if (pwm->tcp_conn->rc == GPG_ERR_EINPROGRESS) {
	    int ret;
	    socklen_t len = sizeof(int);

	    FD_ZERO(&fds);
	    FD_SET(pwm->tcp_conn->fd, &fds);
#ifdef WITH_LIBPTH
	    n = pth_select(pwm->tcp_conn->fd+1, NULL, &fds, NULL, &tv);
#else
	    n = select(pwm->tcp_conn->fd+1, NULL, &fds, NULL, &tv);
#endif

	    if (!n || !FD_ISSET(pwm->tcp_conn->fd, &fds))
		return pwm->state;
	    else if (n == -1) {
		*rc = gpg_error_from_syserror();
		return reset_async(pwm, 1);
	    }

	    ret = getsockopt(pwm->tcp_conn->fd, SOL_SOCKET, SO_ERROR, &n, &len);

	    if (ret || n) {
		*rc = ret ? gpg_error_from_syserror() : gpg_error_from_errno(n);
		return reset_async(pwm, 1);
	    }
	    
	    pwm->tcp_conn->state = SSH_INIT;
	    pwm->tcp_conn->rc = 0;
	    *rc = _setup_ssh_session(pwm);

	    if (*rc && *rc != GPG_ERR_EAGAIN)
		return reset_async(pwm, 1);
	}
	else if (pwm->tcp_conn->rc) {
	    *rc = pwm->tcp_conn->rc;
	    return reset_async(pwm, 1);
	}

	switch (pwm->tcp_conn->state) {
	    case SSH_INIT:
		*rc = _setup_ssh_init(pwm);
		break;
	    case SSH_AUTHLIST:
		*rc = _setup_ssh_authlist(pwm);
		break;
	    case SSH_AUTH:
		*rc = _setup_ssh_auth(pwm);
		break;
	    case SSH_CHANNEL:
		*rc = _setup_ssh_channel(pwm);
		break;
	    case SSH_SHELL:
		*rc = _setup_ssh_shell(pwm);
		break;
	    default:
		break;
	}

	if (*rc == GPG_ERR_EAGAIN) {
	    *rc = 0;
	    return ASYNC_PROCESS;
	}

	if (!*rc) {
	    switch (pwm->tcp_conn->cmd) {
		case ASYNC_CMD_HOSTKEY:
		    *result = pwm->result;
		    break;
		default:
		    break;
	    }
	}

	return reset_async(pwm, *rc ? 1 : 0);
    }
#endif

#ifdef WITH_PINENTRY
    if (pwm->cmd == ASYNC_CMD_OPEN2 || pwm->cmd == ASYNC_CMD_SAVE2) {
	int status;

	if (pwm->nb_fd == -1) {
	    *rc = GPG_ERR_INV_STATE;
	    return reset_async(pwm, 0);
	}

	FD_ZERO(&fds);
	FD_SET(pwm->nb_fd, &fds);
	FD_SET(pwm->fd, &fds);
#ifdef WITH_LIBPTH
	n = pth_select(pwm->nb_fd+1, &fds, NULL, NULL, &tv);
#else
	n = select(pwm->nb_fd+1, &fds, NULL, NULL, &tv);
#endif
	if (n < 0) {
	    *rc = gpg_error_from_syserror();
	    return reset_async(pwm, 0);
	}

	if (n > 0 && FD_ISSET(pwm->nb_fd, &fds)) {
	    pwmd_nb_status_t nb;
#ifdef WITH_LIBPTH
	    size_t len = pth_read(pwm->nb_fd, &nb, sizeof(nb));
#else
	    size_t len = read(pwm->nb_fd, &nb, sizeof(nb));
#endif
	    waitpid(pwm->nb_pid, &status, WNOHANG);

	    if (len != sizeof(nb)) {
		memset(&nb, 0, sizeof(pwmd_nb_status_t));
		*rc = gpg_error_from_syserror();
		return reset_async(pwm, 0);
	    }

	    *rc = nb.error;

	    if (*rc)
		return reset_async(pwm, 0);

	    /* Since the non-blocking pinentry returned a success, do a
	     * non-blocking OPEN or SAVE. */
	    pwmd_async_cmd_t c = pwm->cmd;
	    reset_async(pwm, 0);
	    pwm->_password = pwmd_strdup(nb.password);
	    memset(&nb, 0, sizeof(pwmd_nb_status_t));
	    pwm->lastcmd = c;

	    if (!pwm->_password) {
		*rc = gpg_error_from_errno(ENOMEM);
		return reset_async(pwm, 0);
	    }

	    if (c == ASYNC_CMD_SAVE2)
		*rc = pwmd_save_async(pwm);
	    else
		*rc = pwmd_open_async(pwm, pwm->filename);

	    if (*rc) {
		reset_async(pwm, 0);
		return ASYNC_DONE;
	    }

	    return ASYNC_PROCESS;
	}

	/* Fall through so status messages can be processed during the
	 * pinentry. */
    }
#endif

    if (pwm->fd < 0) {
	*rc = GPG_ERR_INV_STATE;
	return reset_async(pwm, 0);
    }

    /* This is for pwmd_open_async() and pwmd_save_async(). For pinentry
     * retries. */
    *rc = _pwmd_process(pwm);

    if (*rc && *rc != GPG_ERR_INV_PASSPHRASE)
	return reset_async(pwm, 0);

    if (pwm->cmd == ASYNC_CMD_OPEN &&
	    *rc == GPG_ERR_INV_PASSPHRASE &&
#ifdef WITH_TCP
	    (!pwm->tcp_conn || (pwm->tcp_conn && pwm->lastcmd == ASYNC_CMD_OPEN2)) &&
#endif
	    ++pwm->pin_try < pwm->pinentry_tries) {
	if (!get_custom_passphrase(pwm, NULL))
	    goto done;

#ifdef WITH_PINENTRY
	if (pwm->_password) {
	    reset_async(pwm, 0);
	    pwm->lastcmd = ASYNC_CMD_OPEN2;
	    *rc = pwmd_open_async2(pwm, pwm->filename);
	}
	else {
#endif
	    reset_async(pwm, 0);
	    pwm->lastcmd = ASYNC_CMD_OPEN;
	    *rc = pwmd_open_async(pwm, pwm->filename);
#ifdef WITH_PINENTRY
	}
#endif
    }

done:
    if (*rc || pwm->state == ASYNC_DONE)
	return reset_async(pwm, 0);

    return pwm->state;
}

gpg_error_t _assuan_command(pwm_t *pwm, assuan_context_t ctx,
	char **result, const char *cmd)
{
    membuf_t data;
    gpg_error_t rc;

    if (!cmd || !*cmd)
	return GPG_ERR_INV_ARG;

    if (strlen(cmd) >= ASSUAN_LINELENGTH+1)
	return GPG_ERR_LINE_TOO_LONG;

    data.len = 0;
    data.buf = NULL;
    rc = assuan_transact(ctx, cmd, inquire_realloc_cb, &data,
#ifdef WITH_QUALITY
	    pwm->pctx == ctx ? pwm->_inquire_func : inquire_cb,
	    pwm->pctx == ctx ? pwm->_inquire_data : pwm,
#else
	    inquire_cb, pwm,
#endif
	    pwm->status_func, pwm->status_data);

    if (rc) {
	if (data.buf) {
	    pwmd_free(data.buf);
	    data.buf = NULL;
	}
    }
    else {
	if (data.buf) {
	    inquire_realloc_cb(&data, "", 1);

	    if (!result) {
		pwmd_free(data.buf);
		rc = GPG_ERR_INV_ARG;
	    }
	    else
		*result = (char *)data.buf;
	}
    }

    return gpg_err_code(rc);
}

gpg_error_t pwmd_inquire(pwm_t *pwm, const char *cmd, pwmd_inquire_cb_t fn,
	void *data)
{
    if (!pwm || !cmd || !fn)
	return GPG_ERR_INV_ARG;

    if (!pwm->ctx)
	return GPG_ERR_INV_STATE;

    pwm->inquire_func = fn;
    pwm->inquire_data = data;
    return _assuan_command(pwm, pwm->ctx, NULL, cmd);
}

gpg_error_t pwmd_command_ap(pwm_t *pwm, char **result, const char *cmd,
	va_list ap)
{
    char *buf;
    size_t len;
    va_list ap2;

    if (!pwm || !cmd)
	return GPG_ERR_INV_ARG;

    if (!pwm->ctx)
	return GPG_ERR_INV_STATE;

    /*
     * C99 allows the dst pointer to be null which will calculate the length
     * of the would-be result and return it.
     */
    va_copy(ap2, ap);
    len = vsnprintf(NULL, 0, cmd, ap)+1;
    buf = (char *)pwmd_malloc(len);

    if (!buf) {
	va_end(ap2);
	return gpg_error_from_errno(ENOMEM);
    }

    len = vsnprintf(buf, len, cmd, ap2);
    va_end(ap2);

    if (buf[strlen(buf)-1] == '\n')
	buf[strlen(buf)-1] = 0;

    if (buf[strlen(buf)-1] == '\r')
	buf[strlen(buf)-1] = 0;

    gpg_error_t rc = _assuan_command(pwm, pwm->ctx, result, buf);
    pwmd_free(buf);
    return rc;
}

gpg_error_t pwmd_command(pwm_t *pwm, char **result, const char *cmd, ...) 
{
    va_list ap;

    if (!pwm || !cmd)
	return GPG_ERR_INV_ARG;

    if (!pwm->ctx)
	return GPG_ERR_INV_STATE;

    if (result)
	*result = NULL;

    va_start(ap, cmd);
    gpg_error_t rc = pwmd_command_ap(pwm, result, cmd, ap);
    va_end(ap);
    return rc;
}

static gpg_error_t send_pinentry_options(pwm_t *pwm)
{
    gpg_error_t rc;

    if (pwm->pinentry_path) {
	rc = pwmd_command(pwm, NULL, "SET PINENTRY_PATH=%s",
		pwm->pinentry_path);

	if (rc)
	    return rc;
    }

    if (pwm->pinentry_tty) {
	rc = pwmd_command(pwm, NULL, "SET TTYNAME=%s", pwm->pinentry_tty);

	if (rc)
	    return rc;
    }

    if (pwm->pinentry_term) {
	rc = pwmd_command(pwm, NULL, "SET TTYTYPE=%s", pwm->pinentry_term);

	if (rc)
	    return rc;
    }

    if (pwm->pinentry_display) {
	rc = pwmd_command(pwm, NULL, "SET DISPLAY=%s",
		pwm->pinentry_display);

	if (rc)
	    return rc;
    }

    if (pwm->title) {
	rc = pwmd_command(pwm, NULL, "SET TITLE=%s", pwm->title);

	if (rc)
	    return rc;
    }

    if (pwm->desc) {
	rc = pwmd_command(pwm, NULL, "SET DESC=%s", pwm->desc);

	if (rc)
	    return rc;
    }

    if (pwm->prompt) {
	rc = pwmd_command(pwm, NULL, "SET PROMPT=%s", pwm->prompt);

	if (rc)
	    return rc;
    }

    if (pwm->lcctype) {
	rc = pwmd_command(pwm, NULL, "SET LC_CTYPE=%s", pwm->lcctype);

	if (rc)
	    return rc;
    }

    if (pwm->lcmessages) {
	rc = pwmd_command(pwm, NULL, "SET LC_MESSAGES=%s", pwm->lcmessages);

	if (rc)
	    return rc;
    }

    if (pwm->pinentry_timeout >= 0 && !pwm->pin_try) {
	rc = pwmd_command(pwm, NULL, "SET PINENTRY_TIMEOUT=%i",
		pwm->pinentry_timeout);

	if (rc)
	    return rc;
    }

    return 0;
}

gpg_error_t pwmd_socket_type(pwm_t *pwm, pwmd_socket_t *result)
{
    if (!pwm || !result)
	return GPG_ERR_INV_ARG;

#ifdef WITH_TCP
    if ((pwm->fd == -1 && !pwm->tcp_conn) ||
	    (pwm->fd == -1 && pwm->tcp_conn && pwm->tcp_conn->fd == -1))
#else
    if (pwm->fd == -1)
#endif
	return GPG_ERR_INV_STATE;

#ifdef WITH_TCP
    *result = pwm->tcp_conn ? PWMD_SOCKET_SSH : PWMD_SOCKET_LOCAL;
#else
    *result = PWMD_SOCKET_LOCAL;
#endif
    return 0;
}

static gpg_error_t set_pinentry_retry(pwm_t *pwm)
{
    gpg_error_t rc = 0;

    if (pwm->pin_try == 1) {
	rc = pwmd_command(pwm, NULL, "SET TITLE=%s",
		N_("Invalid passphrase, please try again."));

	if (rc)
	    return rc;

	rc = pwmd_command(pwm, NULL, "SET PINENTRY_TIMEOUT=0");
    }

    return rc;
}

static gpg_error_t get_custom_passphrase(pwm_t *pwm, char **result)
{
    gpg_error_t rc = GPG_ERR_NO_DATA;

    if (result)
	*result = NULL;
    else {
	if (pwm->password || pwm->passfunc)
	    return 0;
    }

    if (pwm->password) {
	rc = 0;
	*result = pwm->password;
    }
    else if (pwm->passfunc)
	rc = pwm->passfunc(pwm->passdata, result);

    return rc;
}

static gpg_error_t do_pwmd_open(pwm_t *pwm, const char *filename, int nb,
	int local_pinentry)
{
    char *result = NULL;
    char *password = NULL;
    gpg_error_t rc;

    if (pwm->lastcmd != ASYNC_CMD_OPEN2)
	pwm->pin_try = 0;

    if (!pwm || !filename || !*filename)
	return GPG_ERR_INV_ARG;

    if (!pwm->ctx)
	return GPG_ERR_INV_STATE;

    /*
     * Avoid calling pinentry if the password is cached on the server or if
     * this is a new file.
     */
    rc = pwmd_command(pwm, &result, "ISCACHED %s", filename);

    if (rc == GPG_ERR_ENOENT)
	goto gotpassword;

    if (rc && rc != GPG_ERR_NOT_FOUND)
	return rc;

    if (rc == GPG_ERR_NOT_FOUND) {
	rc = get_custom_passphrase(pwm, &password);

	if (rc && rc != GPG_ERR_NO_DATA)
	    return rc;
	else if (rc == GPG_ERR_NO_DATA)
	    rc = GPG_ERR_NOT_FOUND;
	else
	    goto gotpassword;
    }

#ifdef WITH_PINENTRY
    if (rc == GPG_ERR_NOT_FOUND && local_pinentry) {
	/* Prevent pwmd from using it's pinentry if the passphrase fails. */
	if (!pwm->pin_try) {
	    rc = pwmd_command(pwm, NULL, "SET ENABLE_PINENTRY=0");

	    if (rc)
		return rc;
	}

	rc = _pinentry_open(pwm, filename, &password, nb);

	/* pwmd_process() should be called if using a non-blocking local
	 * pinentry. */
	if (rc || (!rc && nb))
	    return rc;
    }
#endif

gotpassword:
    reset_async(pwm, 0);

#ifdef WITH_TCP
    if (!local_pinentry && !pwm->tcp_conn && !pwm->pin_try) {
#else
    if (!local_pinentry && !pwm->pin_try) {
#endif
	rc = send_pinentry_options(pwm);

	if (rc)
	    return rc;
    }

    rc = pwmd_command(pwm, NULL, "OPEN %s %s", filename,
	    password ? password : "");

    /*
     * Keep the user defined password set with pwmd_setopt(). The password may
     * be needed later (pwmd_save()) depending on the pwmd file cache settings.
     */
    if (!pwm->passfunc && password && password != pwm->password)
	pwmd_free(password);

    if (rc == GPG_ERR_INV_PASSPHRASE) {
	if (++pwm->pin_try < pwm->pinentry_tries) {
	    if (!get_custom_passphrase(pwm, NULL))
		goto done;

#ifdef WITH_PINENTRY
#ifdef WITH_TCP
	    if (pwm->tcp_conn && !local_pinentry)
		return rc;
	    else if (local_pinentry)
		rc = _getpin(pwm, &password, PWMD_PINENTRY_OPEN_FAILED);
	    else
#else
	    if (local_pinentry)
		rc = _getpin(pwm, &password, PWMD_PINENTRY_OPEN_FAILED);
	    else
#endif
#else
#ifdef WITH_TCP
	    if (pwm->tcp_conn)
		return rc;
	    else
#endif
#endif
		rc = set_pinentry_retry(pwm);

	    if (rc)
		return rc;

	    goto gotpassword;
	}
#ifdef WITH_PINENTRY
	else if (local_pinentry)
	    _pinentry_disconnect(pwm);
#endif

done:
	return rc;
    }
#ifdef WITH_PINENTRY
    else if (rc && local_pinentry)
	_pinentry_disconnect(pwm);
#endif

    if (!rc) {
	if (pwm->filename)
	    pwmd_free(pwm->filename);

	pwm->filename = pwmd_strdup(filename);
    }

    return rc;
}

gpg_error_t pwmd_open2(pwm_t *pwm, const char *filename)
{
#ifndef WITH_PINENTRY
    return GPG_ERR_NOT_IMPLEMENTED;
#else
    return do_pwmd_open(pwm, filename, 0, 1);
#endif
}

gpg_error_t pwmd_open(pwm_t *pwm, const char *filename)
{
    return do_pwmd_open(pwm, filename, 0, 0);
}

gpg_error_t pwmd_open_async2(pwm_t *pwm, const char *filename)
{
#ifndef WITH_PINENTRY
    return GPG_ERR_NOT_IMPLEMENTED;
#else
    if (!pwm || !filename)
	return GPG_ERR_INV_ARG;

    if (!pwm->ctx)
	return GPG_ERR_INV_STATE;

    if (pwm->cmd != ASYNC_CMD_NONE)
	return GPG_ERR_ASS_NESTED_COMMANDS;

    /* Initialize a new command since this is not a pinentry retry. */
    if (pwm->lastcmd != ASYNC_CMD_OPEN2)
	pwm->pin_try = 0;

    pwm->cmd = ASYNC_CMD_OPEN2;
    pwm->state = ASYNC_PROCESS;
    gpg_error_t rc = do_pwmd_open(pwm, filename, 1, 1);

    if (rc)
	reset_async(pwm, 0);

    return rc;
#endif
}

static gpg_error_t do_pwmd_save(pwm_t *pwm, int nb, int local_pinentry)
{
    char *result = NULL;
    char *password = NULL;
    gpg_error_t rc;

    if (!pwm)
	return GPG_ERR_INV_ARG;

    if (!pwm->ctx)
	return GPG_ERR_INV_STATE;

    rc = pwmd_command(pwm, &result, "ISCACHED %s", pwm->filename);
    
    if (rc == GPG_ERR_ENOENT)
	rc = GPG_ERR_NOT_FOUND;

    if (rc && rc != GPG_ERR_NOT_FOUND)
	return rc;

    if (rc == GPG_ERR_NOT_FOUND) {
	rc = get_custom_passphrase(pwm, &password);

	if (rc && rc != GPG_ERR_NO_DATA)
	    return rc;
	else if (rc == GPG_ERR_NO_DATA)
	    rc = GPG_ERR_NOT_FOUND;
	else
	    goto gotpassword;
    }

    if (rc == GPG_ERR_NOT_FOUND && local_pinentry) {
#ifdef WITH_PINENTRY
	/* Get the password using the LOCAL pinentry. */
	if (nb) {
	    int p[2];
	    pid_t pid;
	    pwmd_nb_status_t pw;

	    if (pipe(p) == -1)
		return gpg_error_from_syserror();

#ifdef WITH_LIBPTH
	    pid = pth_fork();
#else
	    pid = fork();
#endif

	    switch (pid) {
		case 0:
		    close(p[0]);
		    pw.fd = p[0];
		    password = NULL;
		    pw.error = _do_save_getpin(pwm, &password);

		    if (!pw.error) {
			snprintf(pw.password, sizeof(pw.password), "%s",
				password);
			pwmd_free(password);
		    }
#ifdef WITH_LIBPTH
		    pth_write(p[1], &pw, sizeof(pw));
#else
		    write(p[1], &pw, sizeof(pw));
#endif
		    memset(&pw, 0, sizeof(pw));
		    close(p[1]);
		    _exit(0);
		    break;
		case -1:
		    rc = gpg_error_from_syserror();
		    close(p[0]);
		    close(p[1]);
		    return rc;
		default:
		    break;
	    }

	    close(p[1]);
	    pwm->nb_fd = p[0];
	    pwm->nb_pid = pid;
	    return 0;
	}

	rc = _do_save_getpin(pwm, &password);

	if (rc)
	    return rc;
#endif
    }
    else
	pwm->state = ASYNC_DONE;

gotpassword:
    reset_async(pwm, 0);

#ifdef WITH_TCP
    if (!local_pinentry && !pwm->tcp_conn) {
#else
    if (!local_pinentry) {
#endif
	rc = send_pinentry_options(pwm);

	if (rc)
	    return rc;
    }

    rc = pwmd_command(pwm, NULL, "SAVE %s", password ? password : "");

    if (!pwm->passfunc && password && password != pwm->password)
	pwmd_free(password);

    return rc;
}

gpg_error_t pwmd_save_async2(pwm_t *pwm)
{
#ifndef WITH_PINENTRY
    return GPG_ERR_NOT_IMPLEMENTED;
#else
    if (!pwm)
	return GPG_ERR_INV_ARG;

    if (!pwm->ctx)
	return GPG_ERR_INV_STATE;

    if (pwm->cmd != ASYNC_CMD_NONE)
	return GPG_ERR_ASS_NESTED_COMMANDS;

    pwm->cmd = ASYNC_CMD_SAVE2;
    pwm->state = ASYNC_PROCESS;
    gpg_error_t rc = do_pwmd_save(pwm, 1, 1);
    
    if (rc)
	reset_async(pwm, 0);

    return rc;
#endif
}

gpg_error_t pwmd_save2(pwm_t *pwm)
{
#ifndef WITH_PINENTRY
    return GPG_ERR_NOT_IMPLEMENTED;
#else
    return do_pwmd_save(pwm, 0, 1);
#endif
}

gpg_error_t pwmd_save(pwm_t *pwm)
{
    return do_pwmd_save(pwm, 0, 0);
}

gpg_error_t pwmd_setopt(pwm_t *pwm, pwmd_option_t opt, ...)
{
    va_list ap;
    int n = va_arg(ap, int);
    char *arg1;
    gpg_error_t rc = 0;

    if (!pwm)
	return GPG_ERR_INV_ARG;

    va_start(ap, opt);

    switch (opt) {
	case PWMD_OPTION_STATUS_CB:
	    pwm->status_func = va_arg(ap, pwmd_status_cb_t);
	    break;
	case PWMD_OPTION_STATUS_DATA:
	    pwm->status_data = va_arg(ap, void *);
	    break;
	case PWMD_OPTION_PASSPHRASE_CB:
	    pwm->passfunc = va_arg(ap, pwmd_passphrase_cb_t);
	    break;
	case PWMD_OPTION_PASSPHRASE_DATA:
	    pwm->passdata = va_arg(ap, void *);
	    break;
	case PWMD_OPTION_PASSPHRASE:
	    arg1 = va_arg(ap, char *);

	    if (pwm->password)
		pwmd_free(pwm->password);

	    pwm->password = arg1 ? pwmd_strdup(arg1) : NULL;
	    break;
	case PWMD_OPTION_PINENTRY_TRIES:
	    n = va_arg(ap, int);

	    if (n <= 0) {
		va_end(ap);
		rc = GPG_ERR_INV_VALUE;
	    }
	    else
		pwm->pinentry_tries = n;
	    break;
	case PWMD_OPTION_PINENTRY_TIMEOUT:
	    n = va_arg(ap, int);

	    if (n < 0) {
		va_end(ap);
		rc = GPG_ERR_INV_VALUE;
	    }
	    else
		pwm->pinentry_timeout = n;
	    break;
	case PWMD_OPTION_PINENTRY_PATH:
	    if (pwm->pinentry_path)
		pwmd_free(pwm->pinentry_path);

	    pwm->pinentry_path = _expand_homedir(va_arg(ap, char *), NULL);
	    break;
	case PWMD_OPTION_PINENTRY_TTY:
	    arg1 = va_arg(ap, char *);

	    if (pwm->pinentry_tty)
		pwmd_free(pwm->pinentry_tty);

	    pwm->pinentry_tty = arg1 ? pwmd_strdup(arg1) : NULL;
	    break;
	case PWMD_OPTION_PINENTRY_DISPLAY:
	    if (pwm->pinentry_display)
		pwmd_free(pwm->pinentry_display);

	    pwm->pinentry_display = pwmd_strdup(va_arg(ap, char *));
	    break;
	case PWMD_OPTION_PINENTRY_TERM:
	    arg1 = va_arg(ap, char *);

	    if (pwm->pinentry_term)
		pwmd_free(pwm->pinentry_term);

	    pwm->pinentry_term = arg1 ? pwmd_strdup(arg1) : NULL;
	    break;
	case PWMD_OPTION_PINENTRY_TITLE:
	    if (pwm->title)
		pwmd_free(pwm->title);

	    pwm->title = _percent_escape(va_arg(ap, char *));
	    break;
	case PWMD_OPTION_PINENTRY_PROMPT:
	    if (pwm->prompt)
		pwmd_free(pwm->prompt);

	    pwm->prompt = _percent_escape(va_arg(ap, char *));
	    break;
	case PWMD_OPTION_PINENTRY_DESC:
	    if (pwm->desc)
		pwmd_free(pwm->desc);

	    pwm->desc = _percent_escape(va_arg(ap, char *));
	    break;
	case PWMD_OPTION_PINENTRY_LC_CTYPE:
	    arg1 = va_arg(ap, char *);

	    if (pwm->lcctype)
		pwmd_free(pwm->lcctype);

	    pwm->lcctype = arg1 ? pwmd_strdup(arg1) : NULL;
	    break;
	case PWMD_OPTION_PINENTRY_LC_MESSAGES:
	    arg1 = va_arg(ap, char *);

	    if (pwm->lcmessages)
		pwmd_free(pwm->lcmessages);

	    pwm->lcmessages = arg1 ? pwmd_strdup(arg1) : NULL;
	    break;
	case PWMD_OPTION_IP_VERSION:
#ifdef WITH_TCP
	    n = va_arg(ap, int);

	    switch (n) {
		case PWMD_IP_ANY:
		case PWMD_IPV4:
		case PWMD_IPV6:
		    pwm->prot = n;
		    break;
		default:
		    rc = GPG_ERR_INV_VALUE;
		    break;
	    }

	    va_end(ap);
#else
	    rc = GPG_ERR_NOT_IMPLEMENTED;
#endif
	    break;
	default:
	    rc = GPG_ERR_UNKNOWN_OPTION;
	    break;
    }

    va_end(ap);
    return rc;
}

gpg_error_t pwmd_get_fds(pwm_t *pwm, pwmd_fd_t *fds, int *n_fds)
{
    int in_total;
    int fd = 0;
#ifdef WITH_TCP
    int afds[ARES_GETSOCK_MAXNUM];
    int got_sock = 0;
    int n, i;
#endif

    if (!pwm || !fds || !n_fds || *n_fds <= 0)
	return GPG_ERR_INV_ARG;

    in_total = *n_fds;
#ifdef WITH_TCP
    memset(afds, 0, sizeof(int)*ARES_GETSOCK_MAXNUM);
#endif
    memset(fds, 0, sizeof(pwmd_fd_t)*in_total);
    *n_fds = 0;

    switch (pwm->cmd) {
	default:
	case ASYNC_CMD_NONE:
	case ASYNC_CMD_OPEN:
	case ASYNC_CMD_SAVE:
#ifdef WITH_PINENTRY
async1:
#endif
	    if (pwm->fd == -1)
		return GPG_ERR_INV_STATE;

	    (*n_fds)++;
	    fds[fd].fd = pwm->fd;
	    fds[fd++].flags = PWMD_FD_READABLE;
	    return 0;
#ifdef WITH_PINENTRY
	case ASYNC_CMD_OPEN2:
	case ASYNC_CMD_SAVE2:
	    /* The command has already completed (cached or new). */
	    if (pwm->state == ASYNC_DONE)
		return 0;

	    if (pwm->nb_fd == -1)
		return GPG_ERR_INV_STATE;

	    (*n_fds)++;
	    fds[fd].fd = pwm->nb_fd;
	    fds[fd++].flags = PWMD_FD_READABLE;
	    goto async1;
#endif
#ifdef WITH_TCP
	case ASYNC_CMD_DNS:
	    if (!pwm->tcp_conn || !pwm->tcp_conn->chan)
		return GPG_ERR_INV_STATE;

	    n = ares_getsock(pwm->tcp_conn->chan, afds, ARES_GETSOCK_MAXNUM);

	    for (i = 0; i < ARES_GETSOCK_MAXNUM; i++) {
		got_sock = 0;

		if (fd > in_total) {
		    *n_fds = fd;
		    return GPG_ERR_ERANGE;
		}

		if (ARES_GETSOCK_READABLE(n, i)) {
		    got_sock++;
		    fds[fd].flags |= PWMD_FD_READABLE;
		}

		if (ARES_GETSOCK_WRITABLE(n, i)) {
		    got_sock++;
		    fds[fd].flags |= PWMD_FD_WRITABLE;
		}

		if (got_sock)
		    fds[fd++].fd = afds[i];
	    }

	    *n_fds = fd;
	    return 0;
	case ASYNC_CMD_CONNECT:
	case ASYNC_CMD_HOSTKEY:
	    if (!pwm->tcp_conn || pwm->tcp_conn->fd == -1)
		return GPG_ERR_INV_STATE;

	    (*n_fds)++;
	    fds[fd].fd = pwm->tcp_conn->fd;
	    fds[fd++].flags = PWMD_FD_READABLE;
	    return 0;
#endif
    }

    return GPG_ERR_INV_STATE;
}

pwm_t *pwmd_new(const char *name)
{
    pwm_t *h = pwmd_calloc(1, sizeof(pwm_t));

    if (!h)
	return NULL;

    if (name) {
	h->name = pwmd_strdup(name);

	if (!h->name) {
	    pwmd_free(h);
	    return NULL;
	}
    }

    reset_handle(h);
    h->pinentry_timeout = -30;
    h->pinentry_tries = 3;
#ifdef WITH_TCP
    h->prot = PWMD_IP_ANY;
#endif

    if (ttyname(STDOUT_FILENO)) {
	char buf[256];

	ttyname_r(STDOUT_FILENO, buf, sizeof(buf));
	h->pinentry_tty = pwmd_strdup(buf);
	
	if (!h->pinentry_tty)
	    goto fail;
    }

    if (getenv("TERM") && h->pinentry_tty) {
	h->pinentry_term = pwmd_strdup(getenv("TERM"));

	if (!h->pinentry_term)
	    goto fail;
    }

    if (getenv("DISPLAY")) {
	h->pinentry_display = pwmd_strdup(getenv("DISPLAY"));

	if (!h->pinentry_display)
	    goto fail;
    }

    return h;

fail:
    pwmd_close(h);
    return NULL;
}

void pwmd_free(void *ptr)
{
    _xfree(ptr);
}

void *pwmd_malloc(size_t size)
{
    return _xmalloc(size);
}

void *pwmd_calloc(size_t nmemb, size_t size)
{
    return _xcalloc(nmemb, size);
}

void *pwmd_realloc(void *ptr, size_t size)
{
    return _xrealloc(ptr, size);
}

char *pwmd_strdup(const char *str)
{
    return _xstrdup(str);
}

char *pwmd_strdup_printf(const char *fmt, ...)
{
    va_list ap, ap2;
    int len;
    char *buf;

    if (!fmt)
	return NULL;

    va_start(ap, fmt);
    va_copy(ap2, ap);
    len = vsnprintf(NULL, 0, fmt, ap);
    va_end(ap);
    buf = pwmd_malloc(++len);

    if (buf)
	vsnprintf(buf, len, fmt, ap2);

    va_end(ap2);
    return buf;
}

gpg_error_t pwmd_getpin(pwm_t *pwm, const char *filename, char **result,
	pwmd_pinentry_t which)
{
#ifndef WITH_PINENTRY
    return GPG_ERR_NOT_IMPLEMENTED;
#else
    return _pwmd_getpin(pwm, filename, result, which);
#endif
}
