/*
                               STRING.C

                           String utilities
                    
                   Copyright (C) Laszlo Menczel, 2005
                         menczel@invitel.hu

        This is free software without warranty. See 'licence.txt'.
*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "mutil.h"
#include "mutlib.h"

char *__mut_white_sp = " \t\n";

#define BACKSP   8
#define ESC      27

    
//==========================================================================

int mut_strip_ext(char *s)
{
  int i, dotpos;

  if (s == NULL || s[0] == 0)
    MUTERR(MUTERR_BAD_ARG)

  i = dotpos = 0;
  while (i < MUT_MAX_LINE_LEN && s[i] != 0)
  {
    if (s[i] == '.')
      dotpos = i;
    i++;
  }

  if (i == MUT_MAX_LINE_LEN)
    MUTERR(MUTERR_TOO_MANY)
  
  if (dotpos != 0)
    s[dotpos] = 0;
    
  RETURN(1)
}

//==========================================================================

int mut_strip_eol(char *s)
{
  if (s == NULL)
  {
    __mut_errcode = MUTERR_BAD_ARG;
    return MUTERR_BAD_ARG;
  }
    
  while (*s != 0)
  {
    if (*s == '\n')
    {
      *s = 0;
      break;
    }
    s++;
  }
  
  RETURN(1)
}  

//==========================================================================

int mut_stripn_eol(char *s, int max)
{
  int i;
  
  if (s == NULL || max <= 0)
    MUTERR(MUTERR_BAD_ARG)
  
  i = 0;    
  while (i < max && s[i] != 0)
  {
    if (s[i] == '\n')
    {
      s[i] = 0;
      break;
    }
    i++;
  }
  
  if (i == max)
    MUTERR(MUTERR_TOO_MANY)
  else
    RETURN(i)
}  

//==========================================================================

char *mut_str_end(char *s)
{
  if (s == NULL)
    MUTERR_P(MUTERR_BAD_ARG)

  while (*s != 0)
    s++;

  RETURN(s)
}

//==========================================================================

char *mut_strn_end(char *s, int max)
{
  if (s == NULL || max <= 0)
    MUTERR_P(MUTERR_BAD_ARG)

  while (max > 0)
  {
    if (*s == 0)
      RETURN(s)
    s++;
    max--;
  }
  
  MUTERR_P(MUTERR_TOO_MANY)
}

//==========================================================================

char *mut_skip_delimiter(char *s, char *templ, int len)
{
  int n;
    
  n = 0;
  while (1)
  {
    if (n == len)
      MUTERR_P(MUTERR_TOO_MANY)
    
    if (*s == '\n' || *s == 0)
      MUTERR_P(MUTERR_END_OF_STR)

    if (! __mut_is_delimiter(*s, templ))
      RETURN(s)

    s++;
    n++;
  }
}

//==========================================================================

char *mut_skip_space(char *s, int len)
{
  return mut_skip_delimiter(s, " \t", len);
}

//==========================================================================

int mut_strip_quote(char *s, int len)
{
  int count;
  
  if (s == NULL || s[0] == 0 || len < 2)
    MUTERR(MUTERR_TOO_FEW)
      
  if (strlen(s) > len)
    MUTERR(MUTERR_TOO_MANY)

  s++;
  count = 1;
  while (1)
  {
    if (*s == 0 || *s == '\"')
    {
      *(s - 1) = 0;
      RETURN(1)
    }
    
    *(s - 1) = *s;
    s++;
    count++;
  }
}

//==========================================================================

char *mut_str_first(char *s, char c, int len)
{
  int i;
  
  if (s == NULL || len < 1)
    MUTERR_P(MUTERR_BAD_ARG)

  i = 0;
  while (1)
  {
    if (i == len)
      MUTERR_P(MUTERR_TOO_MANY)
      
    if (*s == 0)
      MUTERR_P(MUTERR_END_OF_STR)
    
    if (*s == c)
      RETURN(s)

    i++;
    s++;
  }
}

//==========================================================================

char *mut_str_last(char *s, char c, int len)
{
  int i;
  char *pos;
  
  if (s == NULL || len < 1)
    MUTERR_P(MUTERR_BAD_ARG)

  i = 0;
  pos = NULL;
  while (1)
  {
    if (i == len)
      MUTERR_P(MUTERR_TOO_MANY)
      
    if (*s == 0)
    {
      if (pos == NULL)
        MUTERR_P(MUTERR_END_OF_STR)
      else
        RETURN(pos)
    }
    
    if (*s == c)
      pos = s;
      
    i++;
    s++;
  }
}

//==========================================================================

char *mut_str_find(char *s, char c, int len, int dir)
{
  if (dir == MUT_FIND_FIRST)
    return mut_str_first(s, c, len);
  else
    return mut_str_last(s, c, len);
}

//==========================================================================

int str_repl(char *s, char old, char new)
{
  if (s == NULL || old == 0)
    MUTERR(MUTERR_BAD_ARG)
    
  while (*s != 0)
  {
    if (*s == old)
      *s = new;
    s++;
  }
  
  RETURN(1)
}

//==========================================================================

char **mut_str_split(char *s, char *delim, int *res)
{
  int i, next;
  char *start, *end, *frag[MUT_MAX_SUB_STRING], **array;

  if (s == NULL || s[0] == 0 || res == NULL)
  {
    *res = MUTERR_BAD_ARG;
    MUTERR_P(MUTERR_BAD_ARG)
  }

  *res = 0;
  next = 0;
  start = end = s;

  if (delim == NULL)
    delim = __mut_white_sp;

  while (1)
  {
    if (next == MUT_MAX_SUB_STRING || *start == 0)
      break;

    if (__mut_is_delimiter(*start, delim))              // skip delimiter
    {
      start++;
      end++;
      continue;
    }
    
    while (*end != 0 && ! __mut_is_delimiter(*end, delim))    // find next delimiter
      end++;
          
    frag[next] = (char *) malloc(end - start + 1);

    if (frag[next] == NULL)
    {
      next--;
      while (next >= 0)
      {
        free(frag[next]);
        next--;
      }
      *res = MUTERR_ALLOC;
      MUTERR_P(MUTERR_ALLOC)
    }
    
    for (i = 0; i < end - start; i++)
      frag[next][i] = start[i];

    frag[next][i] = 0;

    start = end;
    next++;
  }

  if (next == 0)
  {
    *res = MUTERR_EMPTY;
    MUTERR_P(MUTERR_EMPTY)
  }

  array = (char **) malloc(next * sizeof(char *));

  if (array == NULL)
  {
    for (i = 0; i < next; i++)
      free(frag[i]);
      *res = MUTERR_ALLOC;
      MUTERR_P(MUTERR_ALLOC)
  }      

  for (i = 0; i < next; i++)
    array[i] = frag[i];
    
  if (next == MUT_MAX_SUB_STRING)
  {
    *res = MUTERR_TOO_MANY;
    MUTERR_P(MUTERR_TOO_MANY)
  }
  else
    *res = next;
  
  RETURN(array)
}

//==========================================================================

char **mut_str_substr_ptr(char *buf, char delim, char end)
{
  char **ptr, *pos;
  int i, count;

  if (buf == NULL)
    MUTERR_P(MUTERR_BAD_ARG)

  i = count = 0;

  pos = buf;
  while (*pos != end)
  {
    if (*pos == delim && *(pos + 1) != end)
      count++;
    pos++;
  }

  if (count == 0)
    MUTERR_P(MUTERR_EMPTY)

  ptr = (char **) malloc((count + 1) * sizeof(char *));

  if (ptr == NULL)
    MUTERR_P(MUTERR_ALLOC)

  for (i = 0, pos = buf; i < count; i++)
  {
    ptr[i] = pos;
    while (*pos != end && *pos != delim)
      pos++;
  }

  ptr[i] = NULL;

  RETURN(ptr)
}

//==========================================================================

int mut_str_match(char *s, char *buf)
{
  int i;

  if (s == NULL || s[0] == 0 || buf == NULL)
    MUTERR(MUTERR_BAD_ARG)

  i = 0;
  while (*s != 0)
  {
    if (i == MUT_MAX_STR_LEN)
      MUTERR(MUTERR_TOO_MANY)

    if (*s != *buf)
      MUTERR(MUTERR_NOT_FOUND)

    s++;
    buf++;
    i++;
  }

  RETURN(i)
}

//==========================================================================
// Helper functions
//==========================================================================

int __mut_is_delimiter(char c, char *templ)
{
  while (*templ != 0)
  {
    if (c == *templ)
      return 1;
    else
      templ++;
  }
  
  return 0;
}
