/*
                               GLOB.C

          A comprehensive file & directory name globbbing module
                    
                   Copyright (C) Laszlo Menczel, 2005
                         menczel@invitel.hu

                This is free software without warranty.
     Distributed under the Library General Public Licence version 2.
*/

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <malloc.h>

#if defined _LINUX
  #include <glob.h>
#endif

#if defined _WIN32
  #include <windows.h>
#endif

#include "mutil.h"
#include "mutlib.h"

#if defined _WIN32
  static void copy_finfo(finfo_t *s, finfo_t *d);
#endif

//=============================================================================

flist_t *mut_new_glob(void)
{
  flist_t *new;

  new = (flist_t *) malloc(sizeof(flist_t));
  if (new == NULL)
    MUTERR_P(MUTERR_ALLOC)

  new->flag    = 0;
  new->count   = 0;
  new->filenum = 0;
  new->dirnum  = 0;
  new->dirlen  = 0;
  new->dir[0]  = 0;
  new->files   = NULL;
  new->next    = NULL;

  RETURN(new)
}

//=============================================================================

int mut_glob_append(flist_t *f, flist_t *list)
{
  flist_t *curr;

  if (f == NULL || list == NULL)
    MUTERR(MUTERR_BAD_ARG)

  curr = list;
  while (curr->next != NULL)
    curr = curr->next;
  curr->next = f;

  RETURN(1)
}

//=============================================================================

int mut_glob_insert(flist_t *f, flist_t *list, flist_t *pos)
{
  flist_t *tmp;

  if (f == NULL || list == NULL || pos == NULL)
    MUTERR(MUTERR_BAD_ARG)

  tmp = pos->next;
  pos->next = f;
  f->next = tmp;

  RETURN(1)
} 

//=============================================================================

#define MAX_GLOB_LIST_COUNT  100000

int glob_list_count(flist_t *list)
{
  flist_t *curr;
  int i;

  if (list == NULL)
    MUTERR(MUTERR_BAD_ARG)

  for (curr = list, i = 0; curr != NULL; curr = curr->next, i++)
    if (i == MAX_GLOB_LIST_COUNT)
      MUTERR(MUTERR_OVERFLOW)

  RETURN(i)
}

//=============================================================================

#define MAX_GLOB_ITEM_COUNT  100000

int glob_item_count(flist_t *list, int mode)
{
  flist_t *curr;
  int count;

  if (list == NULL)
    MUTERR(MUTERR_BAD_ARG)

  count = 0;
  for (curr = list; curr != NULL; curr = curr->next)
  {
    if (count == MAX_GLOB_ITEM_COUNT)
      MUTERR(MUTERR_OVERFLOW)

    if (mode != MUT_FLIST_DIRS)
      count += curr->filenum;

    if (mode != MUT_FLIST_FILES)
      count += curr->dirnum;
  }

  RETURN(count)
}

//=============================================================================

int mut_glob_discard(flist_t **list)
{
  flist_t *curr, *next;

  if (list == NULL)
    MUTERR(MUTERR_BAD_ARG)

  if (*list != NULL)
  {
    curr = *list;
    while (curr != NULL)
    {
      next = curr->next;
      if (curr->files != NULL)
         free(curr->files);
      free(curr);
      curr = next;
    }
    *list = NULL;
  }

  RETURN(1)
}

//=============================================================================

flist_t *mut_glob_dir(char *dir, char *pattern, int mode)
{
  flist_t *list, *curr, *first_new, *insert_after, *new;
  int i;
  char dirname[MUT_MAX_PATH_LEN];

  /*==================================================*/
  /* get initial filelist for the directory specified */

  list = mut_new_glob();
  if (list == NULL)
    MUTERR_P(MUTERR_ALLOC)

  if (! mut_glob_fill(dir, pattern, list))
  {
    mut_glob_discard(&list);
    return NULL;
  }

  if (mode == MUT_FLIST_SIMPLE || ! (list->flag & MUT_FLIST_HAS_DIR))
    RETURN(list)

  /*====================================*/
  /* process subdirectories recursively */

  curr = list;

  while (curr != NULL)
  {
    first_new = NULL;		// assume there are no directory entries in 'curr'
    insert_after = curr;	// first new entry is inserted after the current one
    
    for (i = curr->filenum; i < curr->count; i++)
    {
      if (! (curr->files[i].flag & MUT_FINFO_IS_DIR))		// skip files
        continue;

      if (curr->dirlen + curr->files[i].len > MUT_MAX_PATH_LEN - 1)
      {
        mut_glob_discard(&list);
        MUTERR_P(MUTERR_OVERFLOW)
      }

      new = mut_new_glob();
      if (new == NULL)
      {
        mut_glob_discard(&list);
        return NULL;
      }

      strcpy(dirname, curr->dir);
      strcat(dirname, curr->files[i].name);

      if (! mut_glob_fill(dirname, pattern, new))
      {
        mut_glob_discard(&new);
        mut_glob_discard(&list);
        return NULL;
      }

      if (first_new == NULL)	// first new directory found
        first_new = new; 	// resume processing from here in the next round

      mut_glob_insert(new, list, insert_after);
      insert_after = new;
    }

    if (first_new != NULL)			// new entry found, process it
      curr = first_new;
    else
      curr = curr->next;			// normal sequential processing
  }

  RETURN(list)
}

//=============================================================================

#if defined _LINUX

static int do_glob(char *pattern, glob_t *globdata)
{
  int ret;

  ret = glob((const char *) pattern, GLOB_MARK, NULL, globdata);

  if (ret != 0)
  {
    if (ret == GLOB_ABORTED)
      MUTERR(MUTERR_ACCESS_DIR)
    else if (ret == GLOB_NOMATCH)
     MUTERR(MUTERR_NOT_FOUND)
    else if (ret == GLOB_NOSPACE)
     MUTERR(MUTERR_ALLOC)
  }

  RETURN(1)
}

//=============================================================================

int mut_glob_fill(char *dir, char *pattern, flist_t *f)
{
  glob_t filedata, dirdata;
  finfo_t *info;
  int i, len, ret, arr_size, next, changed_dir;
  char cwd[MUT_MAX_PATH_LEN + 1];

  /*==========================*/
  /* check & adjust arguments */

  if (f == NULL)
    MUTERR(MUTERR_BAD_ARG)

  if (pattern == NULL || pattern[0] == 0)
    pattern = "*";

  if (dir != NULL && dir[0] != 0)
  {
    len = strlen(dir);
    if (len > MUT_MAX_PATH_LEN - 1)
      MUTERR(MUTERR_BAD_ARG)
    else
    {
      strcpy(f->dir, dir);
      f->dirlen = len;
    }
  }
  else
  {
    f->dir[0] = 0;
    f->dirlen = 0;
  }

  /*===========================*/
  /* enter specified directory */

  changed_dir = 0;

  if (dir != NULL && dir[0] != 0)
  {
    if (getcwd(cwd, MUT_MAX_PATH_LEN) == NULL)
      MUTERR(MUTERR_SYSCALL)

    ret = chdir(dir);
    if (ret != 0)
      MUTERR(MUTERR_ACCESS_DIR)
    else
      changed_dir = 1;
  }

  /*============================================*/
  /* obtain glob data for files and directories */

  f->flag = 0;

  if (! do_glob("*", &dirdata))
  {
    if (changed_dir)
      chdir(cwd);

    if (__mut_errcode == MUTERR_NOT_FOUND)
    {
      f->flag = MUT_FLIST_EMPTY;
      f->dirnum = 0;
      f->filenum = 0;
      f->count = 0;
      globfree(&dirdata);
      RETURN(1)
    }
    else
      return 0;
  }

  /*===============================================*/
  /* obtain glob data for files matching 'pattern' */

  if (! do_glob(pattern, &filedata))
  {
    if (__mut_errcode == MUTERR_NOT_FOUND)
      f->flag |= MUT_FLIST_NO_MATCH;
    else
    {
      globfree(&dirdata);
      globfree(&filedata);
      if (changed_dir)
        chdir(cwd);
      return 0;
    }
  }

  /*=======================================*/
  /* check and count files and directories */

  f->dirnum = 0;

  for (i = 0; i < dirdata.gl_pathc; i++)
  {
    len = strlen(dirdata.gl_pathv[i]);
    if (len > MUT_MAX_NAME_LEN - 1)
    {
      globfree(&filedata);
      globfree(&dirdata);
      if (changed_dir)
        chdir(cwd);
      MUTERR(MUTERR_OVERFLOW)
    }

    if (dirdata.gl_pathv[i][len - 1] == '/')
      f->dirnum++;
  }

  if (f->dirnum > 0)
    f->flag |= MUT_FLIST_HAS_DIR;

  f->filenum = 0;
  if (! (f->flag & MUT_FLIST_NO_MATCH))
    for (i = 0; i < filedata.gl_pathc; i++)
    {
      len = strlen(filedata.gl_pathv[i]);
      if (len > MUT_MAX_NAME_LEN - 1)
      {
        globfree(&filedata);
        globfree(&dirdata);
        if (changed_dir)
          chdir(cwd);
        MUTERR(MUTERR_OVERFLOW)
      }

      if (filedata.gl_pathv[i][len - 1] != '/')
        f->filenum++;
    }
  
  /*=================================*/
  /* allocate info array for results */

  arr_size = f->filenum + f->dirnum;

  info = (finfo_t *) malloc(arr_size * sizeof(finfo_t)); 
  if (info == NULL)
  {
    globfree(&filedata);
    globfree(&dirdata);
    if (changed_dir)
      chdir(cwd);
    MUTERR(MUTERR_ALLOC)
  }

  /*==============================*/
  /* copy items to the info array */

  next = 0;

  if (! (f->flag & MUT_FLIST_NO_MATCH))
    for (i = 0; i < filedata.gl_pathc; i++)
    {
      len = strlen(filedata.gl_pathv[i]);
      if (filedata.gl_pathv[i][len - 1] != '/')
      {
        strcpy(info[next].name, filedata.gl_pathv[i]);
        info[next].len = strlen(filedata.gl_pathv[i]);
        info[next].flag = 0;
        if (filedata.gl_pathv[i][0] == '.')
          info[next].flag |= MUT_FINFO_HIDDEN;
        next++;
      }
    }

  if (f->flag & MUT_FLIST_HAS_DIR)
    for (i = 0; i < dirdata.gl_pathc; i++)
    {
      len = strlen(dirdata.gl_pathv[i]);
      if (dirdata.gl_pathv[i][len - 1] == '/')
      {
        strcpy(info[next].name, dirdata.gl_pathv[i]);
        info[next].len = strlen(dirdata.gl_pathv[i]);
        info[next].flag = MUT_FINFO_IS_DIR;
        if (dirdata.gl_pathv[i][0] == '.')
          info[next].flag |= MUT_FINFO_HIDDEN;
        next++;
      }
    }

  /*==================================*/
  /* complete the glob list structure */

  f->count = arr_size;
  f->files = info;

  /*===================*/
  /* clean up & return */

  globfree(&filedata);
  globfree(&dirdata);
  if (changed_dir)
    chdir(cwd);

  RETURN(1)
}

#endif  /* _LINUX */

//=============================================================================

#if defined _WIN32

static void copy_finfo(finfo_t *s, finfo_t *d)
{
  d->flag = s->flag;
  d->len = s->len;
  strcpy(d->name, s->name);
}

//=============================================================================

static void sort_finfo(finfo_t *list, int num)
{
  finfo_t tmp;
  int i, swapped;
  
  do
  {
    swapped = 0;

    for (i = 0; i < num - 1; i++)
      if (strcmp(list[i].name, list[i + 1].name) > 0)
      {
        copy_finfo(&list[i], &tmp);
        copy_finfo(&list[i + 1], &list[i]);
        copy_finfo(&tmp, &list[i + 1]);
        swapped = 1;
      }
  }
  while (swapped);
}      

//=============================================================================

/*
   mut_glob_fill
   =========
   This is quite hard to do under Win32. The Win API function
   'FindFirstFile' is horrible. When it returns an invalid handle
   you don't know if the function just failed miserably, or it
   did not find any files matching the pattern. The API help file
   suggests using 'GetLastError' to find out what happened, but does
   not describe the possible error codes (from the description of
   'FindNextFile' I guess that it may be ERROR_NO_MORE_FILES for normal
   termination). Argh!

   Another complication is that the functions 'FindFirstFile" and
   'FindNextFile' return info for a single file only at each call, so
   we don't know in advance how many info structures to reserve for
   storing the data. So we have to allocate a hopefully large enough
   info array (and reallocate it if full), fill it randomly as data are
   returned by the functions, and then make another info array and copy
   the data using the proper order.
   
   The search must be carried out twice, first for directories, then for files
   matching the specified pattern. When searching for directories the pattern
   is set to "*.*" and file entries are ignored.
*/

#define FINFO_COUNT  100

int mut_glob_fill(char *dir, char *pattern, flist_t *f)
{
  HANDLE h;
  WIN32_FIND_DATA data;
  finfo_t *tmp, *info, *result;
  int i, k, limit, dirlen, patlen, found, get_all;
  char name[MUT_MAX_PATH_LEN + 6];

  /****************************/
  /* check & adjust arguments */
  /****************************/

  if (f == NULL)
    MUTERR(MUTERR_BAD_ARG)

  if (pattern == NULL || pattern[0] == 0 || strcmp(pattern, "*.*") == 0)
  {
    get_all = 1;
    patlen = 3;
  }
  else
  {
    get_all = 0;
    patlen = strlen(pattern);
  }
    
  if (dir == NULL || dir[0] == 0)			// use current directory
    dirlen = strlen(getcwd(name, MUT_MAX_PATH_LEN));
  else
    dirlen = strlen(dir);

  if (dirlen + patlen > MUT_MAX_PATH_LEN - 2)
    MUTERR(MUTERR_BAD_ARG)

  /********************************************/
  /* allocate temporary storage for file info */
  /********************************************/
  
  info = (finfo_t *) malloc(FINFO_COUNT * sizeof(finfo_t));
  if (info == NULL)
    MUTERR(MUTERR_ALLOC)

  limit = FINFO_COUNT;
  
  /******************************/
  /* collect subdirectory names */
  /******************************/

  if (dir == NULL || dir[0] == 0)			// use current directory
    getcwd(name, MUT_MAX_PATH_LEN);
  else
    strcpy(name, dir);

  strcat(name,"\\");
  strcat(name, "*.*");
  mut_fname_fix_delim(name);

  found = 0;

  h = FindFirstFile(name, &data);

  if (h == INVALID_HANDLE_VALUE)	// error or directory is empty
  {
    free(info);

    if (GetLastError() == ERROR_NO_MORE_FILES)
    {
      f->flag = MUT_FLIST_NO_MATCH;
      f->files = NULL;
      f->count = 0;
      RETURN(1)
    }
    else
      MUTERR(MUTERR_SYSCALL)
  }

  if (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
  {
    if (strcmp(data.cFileName, ".") != 0 && strcmp(data.cFileName, "..") != 0)
    {
      info[found].len = strlen(data.cFileName);

      if (info[found].len > MUT_MAX_STR_LEN - 2)
        MUTERR(MUTERR_TOO_MANY)

      strcpy(info[found].name, data.cFileName);

      // add trailing backslash
      info[found].name[info[found].len] = '\\';
      info[found].len++;
      info[found].name[info[found].len] = 0;

      info[found].flag = MUT_FINFO_IS_DIR;

      if (data.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN)
        info[found].flag |= MUT_FINFO_HIDDEN;

      found++;
    }
  }

  while (1)
  {
    if (! FindNextFile(h, &data))
      break;
      
    if (strcmp(data.cFileName, ".") == 0 || strcmp(data.cFileName, "..") == 0)
      continue;
      
    if (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
    {
      info[found].len = strlen(data.cFileName);

      if (info[found].len > MUT_MAX_STR_LEN - 2)
        MUTERR(MUTERR_TOO_MANY)

      strcpy(info[found].name, data.cFileName);

      // add trailing backslash
      info[found].name[info[found].len] = '\\';
      info[found].len++;
      info[found].name[info[found].len] = 0;

      info[found].flag = MUT_FINFO_IS_DIR;

      if (data.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN)
        info[found].flag |= MUT_FINFO_HIDDEN;

      found++;
    }
    
    if (found == limit)
    {
      limit += FINFO_COUNT;
      tmp = (finfo_t *) realloc(info, limit);
      if (tmp == NULL)
      {
        free(info);
        __mut_errcode = MUTERR_ALLOC;
        return __mut_errcode;
      }
      else
        info = tmp;
    }
  }
  
  if (GetLastError() != ERROR_NO_MORE_FILES)
  {
    free(info);
    MUTERR(MUTERR_SYSCALL)
  }
  
  f->dirnum = found;

  /*******************************************/
  /* collect file names matching the pattern */
  /*******************************************/

  if (dir == NULL || dir[0] == 0)			// use current directory
    getcwd(name, MUT_MAX_PATH_LEN);
  else
    strcpy(name, dir);

  strcat(name,"\\");

  if (get_all)
    strcat(name, "*.*");
  else
    strcat(name, pattern);

  mut_fname_fix_delim(name);

  h = FindFirstFile(name, &data);

  if (h == INVALID_HANDLE_VALUE)
  {
    if (GetLastError() == ERROR_NO_MORE_FILES)
      goto skip_files;
    else
    {
      free(info);
      MUTERR(MUTERR_SYSCALL)
    }
  }

  if (! (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
  {
    info[found].len = strlen(data.cFileName);

    if (info[found].len > MUT_MAX_STR_LEN - 1)
      MUTERR(MUTERR_TOO_MANY)

    strcpy(info[found].name, data.cFileName);
    info[found].flag = 0;

    if (data.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN)
      info[found].flag |= MUT_FINFO_HIDDEN;

    found++;
  }

  while (1)
  {
    if (! FindNextFile(h, &data))
      break;
      
    if (! (data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
    {
      info[found].len = strlen(data.cFileName);

      if (info[found].len > MUT_MAX_STR_LEN - 1)
        MUTERR(MUTERR_TOO_MANY)

      strcpy(info[found].name, data.cFileName);
      info[found].flag = 0;

      if (data.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN)
        info[found].flag |= MUT_FINFO_HIDDEN;

      found++;
    }

    if (found == limit)
    {
      limit += FINFO_COUNT;
      tmp = (finfo_t *) realloc(info, limit);
      if (tmp == NULL)
      {
        free(info);
        MUTERR(MUTERR_ALLOC)
      }
      else
        info = tmp;
    }
  }
  
  if (GetLastError() != ERROR_NO_MORE_FILES)
  {
    free(info);
    MUTERR(MUTERR_SYSCALL)
  }
  
skip_files:

  /*************************************/
  /* check if we found anything at all */
  /*************************************/

  if (found == 0)
  {
    free(info);
    f->flag = MUT_FLIST_NO_MATCH;
    f->files = NULL;
    f->count = 0;
    RETURN(1)
  }
  else
    f->filenum = found - f->dirnum;

  /****************************************/
  /* allocate final storage for file info */
  /****************************************/

  result = (finfo_t *) malloc(found * sizeof(finfo_t));
  if (result == NULL)
  {
    free(info);
    MUTERR(MUTERR_ALLOC)
  }

  /*********************************************/
  /* set the fields of the 'flist_t' structure */
  /*********************************************/

  if (dir == NULL || dir[0] == 0)			// use current directory
    f->dir[0] = 0;
  else
    strcpy(f->dir, dir);

  f->dirlen = dirlen;

  if (f->dirnum > 0)
    f->flag = MUT_FLIST_HAS_DIR;
  else
    f->flag = 0;

  f->count = found;
  f->files = result;

  /**********************************************/
  /* copy 'finfo' data on files and directories */
  /**********************************************/

  for (i = 0, k = found - 1; i < found; i++, k--)
    copy_finfo(&info[k], &result[i]);

  free(info);

  /********************/
  /* sort the entries */
  /********************/

  if (f->filenum > 1)
    sort_finfo(result, f->filenum);

  if (f->dirnum > 1)
    sort_finfo(&result[f->filenum], f->dirnum);

  RETURN(1)
}

#endif  /* _WIN32 */

//=============================================================================

#define GLOB_BUF_INCR	1024

char *mut_glob_format(flist_t *list, char delim, int mode)
{
  flist_t *curr;
  int i, bufsize, slen, full_path, ofs;
  char *new, *buf;

  if (list == NULL)
    MUTERR_P(MUTERR_BAD_ARG)

  if (mode & MUT_FLIST_PATH)
    full_path = 1;
  else
    full_path = 0;
    
  mode &= ~MUT_FLIST_PATH;
  if (mode < 0 || mode > MUT_FLIST_DIRS)
    MUTERR_P(MUTERR_BAD_ARG)

  bufsize = GLOB_BUF_INCR;
  buf = (char *) malloc(bufsize);

  if (buf == NULL)
    MUTERR_P(MUTERR_ALLOC)
    
  *buf = 0;
  ofs = 0;
  curr = list;

  while (curr != NULL)
  {
    for (i = 0; i < curr->count; i++)
    {
      if (
           (mode == MUT_FLIST_FILES && (curr->files[i].flag & MUT_FINFO_IS_DIR)) ||
           (mode == MUT_FLIST_DIRS  && ! (curr->files[i].flag & MUT_FINFO_IS_DIR))
         )
         continue;
         
      slen = curr->files[i].len;

      if (full_path)
        slen += curr->dirlen;

      if (ofs + slen + 1 >= bufsize)
      {
        bufsize += GLOB_BUF_INCR;
        new = realloc(buf, bufsize);

        if (new == NULL)
        {
          free(buf);
          MUTERR_P(MUTERR_ALLOC)
        }
        else
          buf = new;
      }

      if (full_path)
      {
        strcpy(buf + ofs, curr->dir);
        strcat(buf + ofs, curr->files[i].name);
      }
      else
        strcpy(buf + ofs, curr->files[i].name);

      ofs += slen;
      *(buf + ofs) = delim;
      ofs++;
    }
    
    curr = curr->next;
  }

  if (ofs > 0)
    *(buf + ofs - 1) = 0;		// replace last delimiter with zero

  RETURN(buf)
}

//=============================================================================

int mut_finfo_hidden(finfo_t *f)
{
  if (f == NULL)
    MUTERR(MUTERR_BAD_ARG)

  __mut_errcode = NO_ERROR;

  if (f->flag & MUT_FINFO_HIDDEN)
    return 1;
  else
    return 0;
}

//=============================================================================

int mut_finfo_is_dir(finfo_t *f)
{
  if (f == NULL)
    MUTERR(MUTERR_BAD_ARG)

  __mut_errcode = NO_ERROR;

  if (f->flag & MUT_FINFO_IS_DIR)
    return 1;
  else
    return 0;
}
