/*
                              DEBUG.C
                        
                   Functions for debugging programs.
   
                  Copyright (C) Laszlo Menczel, 2005
                         menczel@invitel.hu

        This is free software without warranty. See 'licence.txt'.
*/                  

#include <stdio.h>
#include <stdarg.h>
#include <string.h>

#include "mutil.h"
#include "mutlib.h"

//================================================================================

#define MAX_LOG_COUNT 50000

#define PR_BUF_SIZE   2048

static char logname[MUT_MAX_STR_LEN + 8] =
  { 'd', 'e', 'f', 'a', 'u', 'l', 't', '.', 'l', 'o', 'g', 0 };

static int logging      = 0;
static int tracing      = 0;
static int loglimit_set = 0;
static int loglimit     = MAX_LOG_COUNT;
static int lognum       = 0;

//================================================================================

#define MAX_BRPT_NUM 1000

void mut_brkpt(int num)
{
  if (num < 0 || num > MAX_BRPT_NUM)
    printf("Breakpoint, press ENTER...");
  else
    printf("Breakpoint %d, press ENTER...", num);

  getchar();
}

//================================================================================

void mut_brkpts(char *msg)
{
  if (msg != NULL && msg[0] != 0)
    printf("Breakpoint '%s'. Press ENTER...", msg);
  else
    printf("Breakpoint (no message). Press ENTER...");

  getchar();
}

//================================================================================

void mut_set_logname(char *name)
{
  if (name == NULL || name[0] == 0)
    strcpy(logname, "default.log");
  else
  {
    if (strlen(name) > MUT_MAX_STR_LEN)
      return;
    strcpy(logname, name);
    mut_strip_ext(logname);
    strcat(logname, ".log");
  }
}

//================================================================================

void mut_set_loglimit(int count)
{
  if (! loglimit_set)
  {
    if (count < 1)
      loglimit = 1;
    else if (count > MAX_LOG_COUNT)
      loglimit = MAX_LOG_COUNT;
    else
      loglimit = count;

    loglimit_set = 1;
  }
}

//================================================================================

void mut_setlog(int on_off)
{
  if (on_off < 0)
    return; 
  logging = on_off;
}

//================================================================================

void mut_settrace(int level)
{
  if (level < 0 || level > 2)
    return;
  tracing = level;
}

//================================================================================

void mut_logputs(char *msg)
{
  FILE *f;
  
  if (logging && lognum <= loglimit)
  {
    f = fopen(logname, "at");
    if (f != NULL)
    {
      lognum++;
      fprintf(f, "%5d %s\n", lognum, msg);
      fclose(f);
    }
  }
}

//================================================================================

void mut_logprintf(char *fmt, ...)
{
  FILE *f;
  char buf[PR_BUF_SIZE + 1];
  va_list arglist;

  if (logging && lognum <= loglimit)
  {
    f = fopen(logname, "at");
    if (f != NULL)
    {
      va_start(arglist, fmt);
      vsnprintf(buf, PR_BUF_SIZE, fmt, arglist);
      buf[PR_BUF_SIZE] = '\0';

      lognum++;
      fprintf(f, "%5d %s\n", lognum, buf);
      fclose(f);
    }
  }
}

//================================================================================

void mut_logtrace(char *msg)
{
  FILE *f;

  if (tracing > 0 && lognum <= loglimit)
  {
    f = fopen(logname, "at");
    if (f != NULL)
    {
      lognum++;
      fprintf(f, "%5d >  %s\n", lognum, msg);
      fclose(f);
    }
  }
}

//================================================================================

void mut_logtrace2(char *msg)
{
  FILE *f;

  if (tracing == 2 && lognum <= loglimit)
  {
    f = fopen(logname, "at");
    if (f != NULL)
    {
      lognum++;
      fprintf(f, "%5d >> %s\n", lognum, msg);
      fclose(f);
    }
  }
}

/***************************************************************************/

void mut_show_flist(flist_t *f)
{
  if (f == NULL)
  {
    printf("flist is NULL\n");
    return;
  }
  
  printf("flag                = %x\n", f->flag);

  if (f->dir[0] == 0)
    printf("base directory      = [current]\n");
  else
    printf("base directory      = %s\n", f->dir);

  printf("directory length    = %d\n", f->dirlen);

  if (f->files == NULL)
    printf("info array          = NULL\n");
  else
    printf("info array          = valid\n");

  printf("directory count     = %d\n", f->dirnum);
  printf("matching file count = %d\n", f->filenum);
  printf("total item count    = %d\n", f->count);

  if (f->next == NULL)
    printf("link to next        = NULL\n");
  else
    printf("link to next        = valid\n");

  printf("\n");
}
