/*
                               CONFIG.C

          Functions for parsing INI files and the command line.

                   Copyright (C) Laszlo Menczel, 2005
                         menczel@invitel.hu

        This is free software without warranty. See 'licence.txt'.
*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <math.h>

#include "mutil.h"
#include "mutlib.h"

//=============================================================================

static char *get_rest_of_line(char *dest, char *buf);

static config_t *find_par_by_name(char *name, config_t *p);

static int arg_missing(arglist_t *arg);
static int extract_index(char *s);
static int is_argstr(char *s);
static int is_templ(char *s, arglist_t *arg);
static int update_parameter(config_t *p, char *val, int index);

//=============================================================================
// Public functions
//=============================================================================

int mut_parse_cmd_line(int argc, char *argv[], arglist_t *list, char *prog)
{
  int n, ind;

  if (argv == NULL || list == NULL)
    MUTERR(MUTERR_BAD_ARG)

  /************************/
  /* reset flags & values */
  /************************/

  n = 0;
  while (list[n].type != MUT_ARG_END)
  {
    if (list[n].type == MUT_ARG_SWITCH)
      *((int *) list[n].dst) = 0;

    else if (list[n].type == MUT_ARG_PAIRED)
      *((char *) list[n].dst) = 0;

    list[n].found = 0;
    n++;
  }

  /***************************************/
  /* extract & store the name of program */
  /***************************************/

  if (prog != NULL)
  {
    if (strlen(argv[0]) > MUT_ARG_MAXLEN - 1)
      MUTERR(MUTERR_ARG_PROGNAME)
    else
      strcpy(prog, argv[0]);
  }

  /********************************************************/
  /* scan command line arguments & identify valid entries */
  /********************************************************/

  n = 1;
  while (n < argc)
  {
    if (strlen(argv[n]) > MUT_ARG_MAXLEN - 1)              /* string too long */
      MUTERR(MUTERR_ARG_SIZE)

    ind = is_templ(argv[n], list);
    if (ind < 0)                           /* unknown or invalid template */
      MUTERR(MUTERR_ARG_NAME)
    
    if (list[ind].type == MUT_ARG_SWITCH)           /* command line switch */
    {
      *((int *) list[ind].dst) = 1;
      list[ind].found = 1;
    }

    else if (list[ind].type == MUT_ARG_PAIRED)         /* string parameter */
    {
      n++;

      if (n == argc)             /* end of arguments, parameter not found */
        MUTERR(MUTERR_ARG_PARAM)

      if (strlen(argv[n]) > MUT_ARG_MAXLEN - 1)  /* parameter string too long */
        MUTERR(MUTERR_ARG_PARSIZE)

      /* last argument sometimes contains newline which confuses 'mut_parse_cmd_line' */
      mut_strip_eol(argv[n]);

      if (! is_argstr(argv[n]))                     /* not a valid string */
        MUTERR(MUTERR_ARG_BAD_PARAM)

      strcpy((char *) list[ind].dst, argv[n]);
      list[ind].found = 1;
    }

    else if (list[ind].found)                    /* duplicate argument */
      MUTERR(MUTERR_ARG_DUPLIC)

    n++;
  }

  /*******************************/
  /* check for missing arguments */
  /*******************************/

  ind = arg_missing(list);
  if (ind >= 0)
    MUTERR(MUTERR_ARG_MISSING)

  RETURN(1)
}

/**************************************************************************/

int mut_load_config(char *name, config_t *par)
{
  FILE *f;
  config_t *p;
  int index;
  char buf[256], token[256], *pos;

  if (name == NULL || name[0] == 0 || par == NULL)
    MUTERR(MUTERR_BAD_ARG)

  f = fopen(name, "rt");
  if (f == NULL)
    MUTERR(MUTERR_PAR_FILE)

  /**************************************************/
  /* check the parameter structure array for errors */
  /**************************************************/

  p = par;
  while (p->type != MUT_INI_END)
  {
    p->loaded = 0;
    p->error = 0;

    if (p->name == NULL || p->name[0] == 0)
      p->error = MUTERR_PAR_NAME;

    if (p->type < 0 || p->type > MUT_INI_ASTR)
      p->error = MUTERR_PAR_TYPE;

    if (p->var == NULL)
      p->error = MUTERR_PAR_VAR;

    if ((p->type == MUT_INI_STR || p->type == MUT_INI_ASTR) && p->len < 1)
      p->error = MUTERR_PAR_SIZE;

    p++;
  }

  /***************************************************/
  /* scan the file line by line and update variables */
  /***************************************************/

  while (1)
  {
    if (fgets(buf, 256, f) == NULL)
      break;

    if (buf[0] == '#' || buf[0] == '\n')         /* empty or comment line */
      continue;

    pos = mut_get_token(token, buf, NULL, 256);
    if (pos == NULL)
      break;

    p = find_par_by_name(token, par);
    if (p == NULL || p->error < 0)
      continue;

    if (p->type > MUT_INI_STR)                    /* array type */
    {
      index = extract_index(token);
      if (index < 0)
      {
        p->error = MUTERR_PAR_INDEX;
        continue;
      }
    }

    if (p->type == MUT_INI_STR || p->type == MUT_INI_ASTR)
      pos = get_rest_of_line(token, pos);
    else
      pos = mut_get_token(token, pos, NULL, 256);

    if (pos == NULL)
    {
      p->error = MUTERR_PAR_VALUE;
      continue;
    }

    if (! update_parameter(p, token, index))
    {
      p->error = MUTERR_PAR_CONV;
      continue;
    }
    else
      (p->loaded)++;

  } /* END WHILE */

  /***********************************************/
  /* check for errors & return appropriate value */
  /***********************************************/

  if (! feof(f))
  {
    fclose(f);
    MUTERR(MUTERR_PAR_FILE);
  }

  fclose(f);
  p = par;

  while (p->type != MUT_INI_END)
  {
    if (! p->loaded || p->error)
      MUTERR(MUTERR_PAR_LOAD)
    p++;
  }

  RETURN(1)
}

/**************************************************************************/
/*************************** local functions ******************************/
/**************************************************************************/

static config_t *find_par_by_name(char *name, config_t *p)
{
  int len;

  len = 0;
  while (name[len] != 0 && name[len] != '[')
    len++;

  while (p->type != MUT_INI_END)
  {
    if (strncmp(name, p->name, len) == 0)
      return p;
    p++;
  }

  return NULL;
}

/**************************************************************************/

static int extract_index(char *s)
{
  char *num;
  int fact, sum;

  while (*s != '[')
  {
    if (*s == 0)
      return -1;
    s++;
  }

  s++;
  num = s;                 /* point to first digit of index */

  while (*s != ']')
  {
    if (*s == 0 || *s < '0' || *s > '9')
      return -1;
    s++;
  }

  if (*(s + 1) != 0)
    return -1;

  s--;
  if (s < num)
    return -1;

  sum = 0;
  fact = 1;
  while (s >= num)
  {
    sum += fact * (int) (*s - '0');
    fact *= 10;
    s--;
  }

  return sum;
}

/**************************************************************************/

/*
   Advances to the first significant (not whitespace) character in buffer
   <buf> and copies the rest (up to a zero character or a newline) to
   <dest>. Returns a poointer to the end of line if OK, or NULL if error.
*/

static char *get_rest_of_line(char *dest, char *buf)
{
  *dest = 0;
  while (1)
  {
    if (*buf == 0 || *buf == 13 || *buf == 10)
      return NULL;
    else if (*buf == ' ' || *buf == '\t')
      buf++;
    else
      break;
  }

  while (*buf != 0 && *buf != 13 && *buf != 10)
  {
    *dest = *buf;
    dest++;
    buf++;
  }

  *dest = 0;
  return buf;
}

/**************************************************************************/

static int update_parameter(config_t *p, char *val, int index)
{
  char **cp;
  int *ip;
  double dval, *dp;

  switch (p->type)
  {
    case MUT_INI_INT:
      ip = (int *) p->var;
      if (mut_stoi(val, MUT_BASE_DEC, ip) < 0)
        return 0;
    break;

    case MUT_INI_AINT:
      ip = (int *) p->var;
      if (mut_stoi(val, MUT_BASE_DEC, &ip[index]) < 0)
        return 0;
    break;

    case MUT_INI_FLT:
      if (mut_stof(val, &dval) < 0)
        return 0;
      dp = (double *) p->var;
      *dp = dval;
    break;

    case MUT_INI_AFLT:
      if (mut_stof(val, &dval) < 0)
        return 0;
      dp = (double *) p->var;
      dp[index] = dval;
    break;

    case MUT_INI_STR:
      strcpy((char *) p->var, val);
    break;

    case MUT_INI_ASTR:
      cp = (char **) p->var;
      strcpy(cp[index], val);
    break;

    default:
      return 0;
    break;
  }

  return 1;
}

/***************************************************************************/

/*
   Scans the argument list <arg> to find if any of the specified template
   strings matches <s>. Returns the index of corresponding element in <arg>
   if yes, otherwise return -1.
*/

static int is_templ(char *s, arglist_t *arg)
{
  int n;

  if (s[0] != '-') return 0;

  n = 0;
  while (arg[n].type != MUT_ARG_END)
  {
    if (strcmp(&(s[1]), arg[n].templ) == 0) return n;
    else n++;
  }

  return -1;
}

/***************************************************************************/

/*
   Check if the string <s> conforms to the format of parameter strings (i.e.
   contains only those characters which are allowed in command line arguments).
   Return 1 if OK, 0 otherwise.

   Note: No Unicode support :-(
*/

static int is_argstr(char *s)
{
  int n, x;

  n = 0;
  while (s[n] != 0)
  {
    x = (int) s[n];
    if (x < 33 || x > 127)

{
  printf("arg = '%s'\n", s);
      return 0;
}
    else n++;
  }

  return 1;
}

/***************************************************************************/

/*
   Scan the argument list <arg>. If an item is found so that:

   - its type is MUT_ARG_PAIRED
   - it is required to be present
   - its <found> flag is 0

   then return the index of the element found. Otherwise return -1.
*/

static int arg_missing(arglist_t *arg)
{
  int n;

  n = 0;
  while (arg[n].type != MUT_ARG_END)
  {
    if (arg[n].type == MUT_ARG_PAIRED && arg[n].req && ! arg[n].found)
      return n;
    else
      n++;
  }

  return -1;
}
