/*
 *  libmodem library, a modem control facility.
 *
 *  modems.c - /etc/modems database routines
 *
 *  Copyright (C) 1994,1995,1996,1997,1998,1999  Riccardo Facchetti
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  You can contact the author at this e-mail address:
 *
 *  fizban@tin.it
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include <dial/modems.h>
#include <dial/mdmerrno.h>
#include <termios.h>

#include "modemsPriv.h"

#define BUFSIZE		128

char *__modems = "modems.c (C) Riccardo Facchetti, version " LIBMODEM_VER;

static int mdmentry = 0;
static FILE *mdmfp;
struct mdmdata mdmdata = {0,0,0,0};

static int open_mdm(void);
static void close_mdm(void);
static int demangle_modem_entry(struct modems *, char *);


/*
 * open/close_mdm: open and close the /etc/modems file
 */
static int open_mdm(void) {
	/*
	 * If environment variable MDMFILE is set use it else use
	 * hard coded default.
	 */
	char *mdm_file = getenv("MDMFILE");

	if (mdm_file == NULL)
		mdm_file = MDMFILE;

	if ((mdmfp = fopen(mdm_file, "r")) == NULL)
		return FAILURE;

	return SUCCESS;
}

static void close_mdm(void) {
	fclose(mdmfp);
}

/*
 * getmdmnam: returns a pointer to the modem entry named
 *            (char *) line, NULL if line is not present in /etc/modems
 */
struct modems *getmdmnam(char *line) {
    static struct modems mdm;
    char *buffer;

    buffer = mdmalloc(BUFSIZE);

    if (!buffer) {
	mdmerrno = -EMDMEM;
	return NULL;
    }

    mdmerrno = 0;

    if (open_mdm() == FAILURE) {
	    mdmfree(buffer);
        mdmerrno = -ENOMDMFILE;
        return NULL;
    }

    while (fgets(buffer, BUFSIZE, mdmfp) != NULL) {
        buffer[strlen(buffer)-1] = '\0';

        if (buffer[0] == '#' || buffer[0] == '\0' || buffer[0] == '\n')
            continue;

        if (demangle_modem_entry(&mdm, buffer) == FAILURE) {
            mdmerrno = -EMDMBADCONF;
            continue;
        }
        if (!strcmp(mdm.line, line)) {
            close_mdm();
	    mdmfree(buffer);
            return(&mdm);
        }
    }
    close_mdm();
    mdmerrno = -ENOMDMLINE;
    return NULL;
}

/*
 * setmdms: set the library for modem sequential search from the first to the
 *          end. See getnextmdm().
 */
void setmdms(void) {
    mdmentry = 0;
}

/*
 * getnextmdm: get the next modem entry in /etc/modems file
 *             returns NULL if we are over the last modem.
 */
struct modems *getnextmdm(void) {
    static struct modems mdm;
    int i = 0;
    char *buffer;

    buffer = mdmalloc(BUFSIZE);

    if (!buffer) {
	mdmerrno = -EMDMEM;
	return NULL;
    }


    mdmerrno = 0;

    if (open_mdm() == FAILURE) {
	    mdmfree(buffer);
        mdmerrno = -ENOMDMFILE;
        return NULL;
    }

    while (fgets(buffer, BUFSIZE, mdmfp) != NULL) {
        buffer[strlen(buffer)-1] = '\0';

        if (buffer[0] == '#' || buffer[0] == '\0' || buffer[0] == '\n')
            continue;

        if (i == mdmentry) {
            if (demangle_modem_entry(&mdm, buffer) == FAILURE) {
                mdmerrno = -EMDMBADCONF;
                continue;
            }
            mdmentry++;
            close_mdm();
	    mdmfree(buffer);
            return(&mdm);
        }
        i++;
    }
    close_mdm();
    mdmfree(buffer);
    return NULL;
}

/*
 * num2baud: convert baud rate to B* definition.
 */
int num2baud (int baud_num) {
      switch (baud_num) {
      case 0:
              return B0;
      case 50:
              return B50;
      case 75:
              return B75;
      case 110:
              return B110;
      case 134:
              return B134;
      case 150:
              return B150;
      case 200:
              return B200;
      case 300:
              return B300;
      case 600:
              return B600;
      case 1200:
              return B1200;
      case 1800:
              return B1800;
      case 2400:
              return B2400;
      case 4800:
              return B4800;
      case 7200:
      case 9600:
              return B9600;
      case 12200:
      case 14400:
      case 16800:
      case 19200:
              return B19200;
      case 21600:
      case 24000:
      case 26400:
      case 28800:
      case 31200:
      case 33600:
      case 38400:
              return B38400;
      }
      return -1;
}

/*
 * baud2num: convert B* definition to baud rate.
 */
int baud2num (int baud_num) {
      switch (baud_num) {
      case B0:
	      return 0;
      case B50:
	      return 50;
      case B75:
              return 75;
      case B110:
              return 110;
      case B134:
              return 134;
      case B150:
              return 150;
      case B200:
              return 200;
      case B300:
              return 300;
      case B600:
              return 600;
      case B1200:
              return 1200;
      case B1800:
              return 1800;
      case B2400:
              return 2400;
      case B4800:
              return 4800;
      case B9600:
              return 9600;
      case B19200:
              return 19200;
      case B38400:
              return 38400;
      }
      return -1;
}

/*
 * demangle_modem_entry: get a modems pointer that point to an existing
 * modems structure and a string to demangle. Returns FAILURE or SUCCESS
 */
static int demangle_modem_entry(struct modems *mdm, char *mdmstr) {
    char *ptr, *pptr;

/*
 * mdm->line (char *)
 */
    if ((ptr = strchr(mdmstr, ':')) == NULL)
        return FAILURE;
    *ptr = '\0';
    strcpy(mdm->line, mdmstr);
    ptr++;

/*
 * mdm->cp (char *)
 */
     if (*ptr == ':') {
        mdm->cp[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        strcpy(mdm->cp, ptr);
        ptr = pptr+1;
    }

/*
 * mdm->ce (char *)
 */
     if (*ptr == ':') {
        mdm->ce[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        strcpy(mdm->ce, ptr);
        ptr = pptr+1;
    }

/*
 * mdm->at (char *)
 */
     if (*ptr == ':') {
        mdm->at[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        strcpy(mdm->at, ptr);
        ptr = pptr+1;
    }

/*
 * mdm->dl (time_t)
 */
     if (*ptr == ':') {
        mdm->dl = 0;
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        mdm->dl = atol(ptr);
        ptr = pptr+1;
    }

/*
 * mdm->bd (int)
 */
     if (*ptr == ':') {
        mdm->bd = 0;
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        mdm->bd = num2baud( atoi(ptr) );
	mdm->bdi = baud2num(mdm->bd);
        ptr = pptr+1;
    }

/*
 * mdm->cs (char *)
 */
     if (*ptr == ':') {
        mdm->cs[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        strcpy(mdm->cs, ptr);
        ptr = pptr+1;
    }

/*
 * mdm->ds (char *)
 */
     if (*ptr == ':') {
        mdm->ds[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        strcpy(mdm->ds, ptr);
        ptr = pptr+1;
    }

/*
 * mdm->dp (char *)
 */
     if (*ptr == ':') {
        mdm->dp[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        strcpy(mdm->dp, ptr);
        ptr = pptr+1;
    }

/*
 * mdm->de (char *)
 */
     if (*ptr == ':') {
        mdm->de[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        strcpy(mdm->de, ptr);
        ptr = pptr+1;
    }

/*
 * mdm->im (char *)
 */
     if (*ptr == ':') {
        mdm->im[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        strcpy(mdm->im, ptr);
        ptr = pptr+1;
    }

/*
 * mdm->hu (char *)
 *
 * the last field.
 */
     if (*ptr == ':') {
        mdm->hu[0] = '\0';
        *ptr = '\0';
        ptr++;
    }
    else {
        if ((pptr = strchr(ptr, ':')) == NULL)
            return FAILURE;
        *pptr = '\0';
        strcpy(mdm->hu, ptr);
        ptr = pptr+1;
    }

     if (*ptr == '\n' || *ptr == '\0' || *ptr == ':') {
        mdm->fc = 0;
        return SUCCESS;
    }
    else
        mdm->fc = atoi(ptr);

    mdm->connected = 0;
    /*
     * Default connection parameters
     */
    mdm->bits = '8';
    mdm->parity = 'N';
    mdm->stop = '1';
    mdm->rings = 0;

    return SUCCESS;
}

void setmdmdata(char bits, char parity, char stop, int rings) {
	if (bits >= '5' && bits <= '8')
		mdmdata.bits = bits;
	if (parity == 'E' || parity == 'O' || parity == 'N')
		mdmdata.parity = parity;
	if (stop == '1' || stop == '2')
		mdmdata.stop = stop;
	mdmdata.rings = 0;
	if (rings > 0)
		mdmdata.rings = rings;
}
