/*
 *  libmodem library, a modem control facility.
 *
 *  mdmlog.c - syslog facility for library debugging
 *
 *  Copyright (C) 1994,1995,1996,1997,1998,1999  Riccardo Facchetti
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  You can contact the author at this e-mail address:
 *
 *  fizban@tin.it
 */

#if defined(LOGFILE)

#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>
#include <errno.h>
#include <time.h>
#include <sys/file.h>

#include <dial/modems.h>

#define BUFSIZE		1024

static int log_opened = 0;

void mdm_log(int logid, char *fmt, ...) {
	va_list args;
	char *buffer;
	char *tmpbuffer;
	FILE *fp;
	time_t tt;
	char *ct;

	tt = time(NULL);
	ct = ctime(&tt);
/*
 * ct points to a statically allocated string overwritten by subsequent calls
 * to ctime()
 */
	ct[strlen(ct)-1] = '\0';

	buffer = mdmalloc(BUFSIZE);

	if (!buffer)
		return;

	tmpbuffer = mdmalloc(BUFSIZE);

	if (!tmpbuffer) {
		mdmfree(buffer);
		return;
	}

#if !defined(DEBUG)
	if (logid == LOG_DEBUG) {
		mdmfree(buffer);
		mdmfree(tmpbuffer);
		return;
	}
#endif

	va_start(args, fmt);

	vsprintf(tmpbuffer, fmt, args);

	va_end(args);

	sprintf(buffer, "LibModem(%s) [%s] {%d} %s (errno: %s, mdmerrno: %s)",
		ct,
		(mdmopendevice ? mdmopendevice->line : "No Line"),
		getpid(),
		tmpbuffer,
		strerror(errno),
		mdmstrerror(mdmerrno));

	if ((fp = fopen(LOGFILE, "a")) == NULL) {
		mdmfree(buffer);
		mdmfree(tmpbuffer);
		return;
	}

	if (flock(fileno(fp), LOCK_EX) != 0) {
		fclose(fp);
		mdmfree(buffer);
		mdmfree(tmpbuffer);
		return;
	}

	if (fprintf(fp, "%s\n", buffer) != (strlen(buffer)+1)) {
		fclose(fp);
		mdmfree(buffer);
		mdmfree(tmpbuffer);
		return;
	}

	fclose(fp);
	mdmfree(buffer);
	mdmfree(tmpbuffer);
}

void open_mdm_log (void) {
    if (!log_opened) {
        log_opened = 1;
    }
}

void close_mdm_log (void) {
    if (log_opened) {
        log_opened = 0;
    }
}

#else

/*
 * No LOGFILE defined: dummy log functions.
 */
void mdm_log(int logid, char *fmt, ...) {
	return;
}
void open_mdm_log (void) {
	return;
}
void close_mdm_log (void) {
	return;
}

#endif
