/*
 *  libmodem library, a modem control facility.
 *
 *  line_manage.c - modem line control
 *
 *  Copyright (C) 1994,1995,1996,1997,1998,1999  Riccardo Facchetti
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  You can contact the author at this e-mail address:
 *
 *  fizban@tin.it
 */

/*
 * Some termios stuff is stolen from Gert Doering's mgetty-0.21 code.
 * Here is his (C) for mgetty-0.21 package:
 * --------------------------------------------------------------------------
 * The mgetty+sendfax package is Copyright (c) 1993 Gert Doering. You are
 * permitted to do nearly anything you want with this program - redistribute
 * it, use parts of the code in your own programs, ..., but you have to give
 * me credit - do not remove my name.
 * 
 * If you make money by selling mgetty, I want a share.
 * 
 * For more details, see the chapter "copying conditions" in mgetty.texi
 * --------------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <signal.h>
#include <setjmp.h>
#include <syslog.h>
#include <ctype.h>
/*
 * Use POSIX termios
 */
#include <termios.h>

#ifdef linux
#include <linux/fs.h>
#include <linux/major.h>
#endif

#include <dial/modems.h>
#include <dial/mdmerrno.h>

#include "line_managePriv.h"
#include "modemsPriv.h"
#include "termios_cc.h"

#define BUFSIZE	1024

char *__line_manage = "line_manage.c (C) Riccardo Facchetti, version " LIBMODEM_VER;

static jmp_buf not_opened;
#if 1
struct termios savetty;
#endif

/*
 * Device opened when dial return a valid fd
 */
struct modems *mdmopendevice;
/*
 * If set, cycle all the modems when an error is encountered
 * This don't work if you use the bldial() call
 */
int mdm_cycle = 0;

struct mdm_reply_s mdm_reply[] = {
/*
 * The "UNKNOWN MODEM REPLY" message MUST be the first one:
 * check_modem_reply rely on this to return "unknown error"
 */
	{ "[LIBMODEM] UNKNOWN MODEM REPLY",	EUNMDMERR	},
	{ "OK",					ENOMDMERR	},
	{ "CONNECT",				ENOMDMERR	},
	{ "ERROR",				EMDMERROR	},
	{ "NO CARRIER",				EMDMCARR	},
	{ "BUSY",				EMDMBUSY	},
	{ "NO DIALTONE",			EMDMTONE	},
	{ "RING",				EMDMRING	},
	{ "NO ANSWER",				EMDMANSW	},
	{ "RINGING",				EMDMRINGING	},
	{ NULL,					0		}
};

static void set_mode_sane(struct termios *, struct modems *, int);

static void alarm_jump(int sig) {
	longjmp(not_opened, 1);
}

int dial(char *phonenum)  {
    struct modems *mdl;
    int ret = 0;

    open_mdm_log();
    mdmerrno = 0;
    setmdms();

    while ((mdl = getnextmdm()) != NULL) {
/*
 * if locked, skip to the next modem
 */
        if (mdm_chklock(mdl) == SUCCESS)
		continue;

/*
 * Try to dial...
 */
        ret = bldial(phonenum, mdl->line, baud2num(mdl->bd));

/*
 * If the user ask it, cycle all the modems on error
 */
	if ((ret < 0) && mdm_cycle) {
/*
 * If cycle is chosen, the meaning of mdm_cycle is not only an on/off, it is
 * the max number of times we cycle. At 0, end of the story: no more retries.
 */
		--mdm_cycle;
/*
 * Log the error and continue (we force the error log because this modem can
 * be in a fault state while the sysadmin don't know the faulty state).
 */
		mdm_log(LOG_INFO,
			"Error encountered while opening %s (%s)\n",
			mdl->line,
			mdmstrerror(ret));

		switch (ret) {
/*
 * If config file don't exist or unknown error is returned,  or line is busy
 * don't cycle the modems.
 */
			case -EUNMDMERR:
			case -ENOMDMFILE:
			case -EMDMBUSY:
				return ret;
			default:
		    		continue;
		}
	}
/*
 * Return the error or the file descriptor.
 */
	return ret;
    }

    return (mdmerrno = -ENOMDMDEV);
}

int ldial(char *phonenum, char *line) {
    struct modems *mdl;

    open_mdm_log();
    mdmerrno = 0;

    if ((mdl = getmdmnam(line)) == NULL)
        return mdmerrno;

/*
 * dial at the modem baud rate
 */
    return bldial(phonenum, line, baud2num(mdl->bd));
}

int bdial(char *phonenum, int baud)  {
    struct modems *mdl;
    int ret = 0;

    /*
     * Transform baud number in B*
     */
    baud = num2baud(baud);

    open_mdm_log();
    mdmerrno = 0;
    setmdms();

    while ((mdl = getnextmdm()) != NULL) {
        if (baud <= mdl->bd) {
/*
 * if locked, skip this modem
 */
            if (mdm_chklock(mdl) == SUCCESS)
	    	continue;
/*
 * Try to dial...
 */
            ret = bldial(phonenum, mdl->line, baud2num(mdl->bd));
/*
 * If the user ask it, cycle all the modems on error
 */
	    if ((ret < 0) && mdm_cycle) {
/*
 * If cycle is chosen, the meaning of mdm_cycle is not only an on/off, it is
 * the max number of times we cycle. At 0, end of the story: no more retries.
 */
		--mdm_cycle;
/*
 * Log the error and continue
 */
			mdm_log(LOG_INFO,
				"Error encountered while opening %s (%s)\n",
				mdl->line,
				mdmstrerror(ret));

		    	switch (ret) {
/*
 * If config file don't exist or unknown error is returned, or line is busy
 * don't cycle the modems.
 */
				case -EUNMDMERR:
				case -ENOMDMFILE:
				case -EMDMBUSY:
					return ret;
				default:
		    			continue;
		    	}
	    }
/*
 * Return the error or the file descriptor.
 */
            return ret;
        }
    }

    return (mdmerrno = -ENOMDMDEV);
}

int bldial(char *phonenum, char *line, int baud) {
    struct modems *mdl;
    struct termios trm;
    int fd, ret = 0;

    baud = num2baud(baud);

    open_mdm_log();
    mdmerrno = 0;

    if ((mdl = getmdmnam(line)) == NULL)
        return mdmerrno;

    if (mdm_chklock(mdl) == SUCCESS)
	    return (mdmerrno = -EMDMLOCK);

    if (mdmdata.bits) {
    	mdl->bits = mdmdata.bits;
    	mdl->parity = mdmdata.parity;
    	mdl->stop = mdmdata.stop;
	mdl->rings = mdmdata.rings;
    }

/*
 * Here sleep 1 sec just to let the modem settle down before starting
 * writing on it.
 */
    sleep(1);

/*
 * lock and open the modem line and set it up for communications
 * remember to unlock the mdl device when returning an error.
 */
    if ((fd = open_modem_setup_line(mdl, baud)) < 0)
        return fd;

/*
 * Here sleep 1 sec just to let the modem settle down before starting
 * writing on it.
 */
    sleep(1);

/*
 * Now we must save the actual device to have it in hands when hangup
 */
    mdmopendevice = mdl;

/*
 * set up the modem: init string et al
 */
    if ((ret = initialize_modem(fd, mdl)) < 0) {
        hangup(fd);
        return ret;
    }

/*
 * finally dial to phonenum
 */
    if ((ret = dial_to_num(fd, mdl, phonenum)) < 0) {
        hangup(fd);
        return ret;
    }

/*
 * Set line mode to sane.
 */
    
    if (tcgetattr(fd, &trm) != -1) {
	    set_mode_sane(&trm, mdl, 0);
	    tcsetattr(fd, TCSADRAIN, &trm);
    }

/*
 * dial sequence terminated with success, returning fd
 */
    close_mdm_log();
    return fd;
}

int open_mdm_line()  {
    struct modems *mdl;
    int ret = 0;

    open_mdm_log();
    mdmerrno = 0;
    setmdms();

    while ((mdl = getnextmdm()) != NULL) {
/*
 * if locked, skip to the next modem
 */
        if (mdm_chklock(mdl) == SUCCESS)
		continue;

/*
 * Try to dial...
 */
        ret = blopen_mdm_line(mdl->line, mdl->bd);

/*
 * If the user ask it, cycle all the modems on error
 */
	if ((ret < 0) && mdm_cycle) {
/*
 * If cycle is chosen, the meaning of mdm_cycle is not only an on/off, it is
 * the max number of times we cycle. At 0, end of the story: no more retries.
 */
		--mdm_cycle;
/*
 * Log the error and continue (we force the error log because this modem can
 * be in a fault state while the sysadmin don't know the faulty state).
 */
		mdm_log(LOG_INFO,
			"Error encountered while opening %s (%s)\n",
			mdl->line,
			mdmstrerror(ret));

		switch (ret) {
			case -EUNMDMERR:
			case -ENOMDMFILE:
			case -EMDMBUSY:
/*
 * If config file don't exist or unknown error is returned,  or line is busy
 * don't cycle the modems.
 */
				break;
			default:
		    		continue;
		}
	}
/*
 * If the modem is alredy used, skip to the next modem else return the error
 */
	if (ret != EMDMOPEN)
		return ret;
    }

    return (mdmerrno = -ENOMDMDEV);
}

int lopen_mdm_line(char *line) {
    struct modems *mdl;

    open_mdm_log();
    mdmerrno = 0;

    if ((mdl = getmdmnam(line)) == NULL)
        return mdmerrno;

/*
 * dial at the maximum baud rate
 */
    return blopen_mdm_line(line, mdl->bd);
}

int bopen_mdm_line(int baud)  {
    struct modems *mdl;
    int ret = 0;

    open_mdm_log();
    mdmerrno = 0;
    setmdms();

    while ((mdl = getnextmdm()) != NULL) {
        if (baud <= mdl->bd) {
/*
 * if locked, skip this modem
 */
            if (mdm_chklock(mdl) == SUCCESS)
	    	continue;
/*
 * Try to dial...
 */
            ret = blopen_mdm_line(mdl->line, baud);
/*
 * If the user ask it, cycle all the modems on error
 */
	    if ((ret < 0) && mdm_cycle) {
/*
 * If cycle is chosen, the meaning of mdm_cycle is not only an on/off, it is
 * the max number of times we cycle. At 0, end of the story: no more retries.
 */
		--mdm_cycle;
/*
 * Log the error and continue
 */
			mdm_log(LOG_INFO,
				"Error encountered while opening %s (%s)\n",
				mdl->line,
				mdmstrerror(ret));

		    	switch (ret) {
				case -EUNMDMERR:
				case -ENOMDMFILE:
				case -EMDMBUSY:
/*
 * If config file don't exist or unknown error is returned, or line is busy
 * don't cycle the modems.
 */
					break;
				default:
		    			continue;
		    	}
	    }
/*
 * If the modem is alredy used, skip to the next modem else return the error
 */
            if (ret != EMDMOPEN)
                    return ret;
        }
    }

    return (mdmerrno = -ENOMDMDEV);
}

int blopen_mdm_line(char *line, int baud) {
    struct modems *mdl;
    int fd, ret = 0;

    open_mdm_log();
    mdmerrno = 0;

    if ((mdl = getmdmnam(line)) == NULL)
        return mdmerrno;

    if (mdm_chklock(mdl) == SUCCESS)
	    return (mdmerrno = -EMDMLOCK);
/*
 * lock and open the modem line and set it up for communications
 * remember to unlock the mdl device when returning an error.
 */
    if ((fd = open_modem_setup_line(mdl, baud)) < 0)
        return fd;

/*
 * Now we must save the actual device to have it in hands when hangup
 */
    mdmopendevice = mdl;

/*
 * set up the modem: init string et al
 */
    if ((ret = initialize_modem(fd, mdl)) < 0) {
        mdm_unlock(mdl);
        hangup(fd);
        return ret;
    }

/*
 * Comm link now open with modem, returning fd
 */
    close_mdm_log();
    return fd;
}

static int dial_to_num(int fd, struct modems *device, char *num) {
	char *buffer;
	int ret = -ENOMDMERR;

	buffer = mdmalloc(BUFSIZE);

	if (!buffer)
		return -EMDMEM;
/*
 * DIAL string
 */
	sprintf(buffer, "%s%s%s%s%s%s%c",
		    device->cp,
		    device->ds,
		    device->dp,
		    num,
                    device->de,
		    device->ce,
		    MODEM_RETURN);
	ret = talk_to_modem(fd, device, buffer);

	mdmfree(buffer);

	if (ret == ENOMDMERR)
		device->connected = 1;

	return ret;
}

static int initialize_modem(int fd, struct modems *device) {
	char *buffer;
	int ret;

	buffer = mdmalloc(BUFSIZE);

	if (!buffer)
		return -EMDMEM;
/*
 * INIT string
 */
	sprintf(buffer,
		"%s%s%s%c", device->cp, device->im, device->ce, MODEM_RETURN);
	ret = talk_to_modem(fd, device, buffer);

	mdmfree(buffer);

	return ret;
}

static void set_default_cc(struct termios *trm) {
/*
 * set default c_cc flags
 */
	trm->c_cc[VQUIT]  = CQUIT;
	trm->c_cc[VKILL]  = CKILL;
	trm->c_cc[VEOF]   = CEOF;
#if defined(VEOL) && VEOL < TIONCC
	trm->c_cc[VEOL] = CEOL;
#endif
#if defined(VSTART) && VSTART < TIONCC
	trm->c_cc[VSTART] = CSTART;
#endif
#if defined(VSTOP) && VSTOP < TIONCC
	trm->c_cc[VSTOP] = CSTOP;
#endif
#if defined(VSUSP) && VSUSP < TIONCC
	trm->c_cc[VSUSP] = CSUSP;
#endif
#if defined(VSWTCH) && VSWTCH < TIONCC
	trm->c_cc[VSWTCH] = CSWTCH;
#endif
    /* the following are for SVR4.2 (and higher) */
#if defined(VDSUSP) && VDSUSP < TIONCC
	trm->c_cc[VDSUSP] = CDSUSP;
#endif
#if defined(VREPRINT) && VREPRINT < TIONCC
	trm->c_cc[VREPRINT] = CRPRNT;
#endif
#if defined(VDISCARD) && VDISCARD < TIONCC
	trm->c_cc[VDISCARD] = CFLUSH;
#endif
#if defined(VWERASE) && VWERASE < TIONCC
	trm->c_cc[VWERASE] = CWERASE;
#endif
#if defined(VLNEXT) && VLNEXT < TIONCC
	trm->c_cc[VLNEXT] = CLNEXT;
#endif
	trm->c_cc[VMIN]  = 1;                /* 1 char buffering */
}

static void set_flow_control(struct modems *device, struct termios *trm) {
/*
 * set flow control
 */
	trm->c_cflag &= ~CRTSCTS;
	trm->c_iflag &= ~(IXON|IXOFF|IXANY);
	if ( device->fc == FLOW_HARD )
		trm->c_cflag |= CRTSCTS;
	if ( device->fc == FLOW_XON_XOFF )
		trm->c_iflag |= IXOFF|IXON|IXANY;
}

static void set_mode_sane(struct termios *trm, struct modems *device, int local) {
	trm->c_iflag |= BRKINT|IGNPAR;

	trm->c_oflag = 0;
	trm->c_lflag = 0;
	trm->c_cflag &= ~(CSIZE|CSTOPB|PARENB|PARODD|CLOCAL
#ifdef LOBLK
		                   | LOBLK
#endif
					);
	trm->c_cflag |= CREAD|HUPCL|(local ? CLOCAL:0);

	mdm_log(LOG_INFO,
		"Set up device to be %c/%c/%c\n",
		device->bits,
		device->parity,
		device->stop);

	/*
	 * Set up data bits
	 */
	switch (device->bits) {
		case '5':
			trm->c_cflag = (trm->c_cflag & ~CSIZE) | CS5;
			break;
        	case '6':
                	trm->c_cflag = (trm->c_cflag & ~CSIZE) | CS6;
                	break;
        	case '7':
                	trm->c_cflag = (trm->c_cflag & ~CSIZE) | CS7;
                	break;
        	case '8':
        	default:
                	trm->c_cflag = (trm->c_cflag & ~CSIZE) | CS8;
                	break;
	}

	/*
	 * Set up parity
	 */
	switch (device->parity) {
		case 'E':
			trm->c_cflag |= PARENB;
			break;
		case 'O':
			trm->c_cflag |= PARODD;
			break;
		default:
			break;
	}

	/*
	 * Set up stop bits
	 */
	switch (device->stop) {
		case '2':
			trm->c_cflag |= CSTOPB;
			break;
		default:
			break;
	}

}

static int open_modem_setup_line(struct modems *device, int baud) {
	struct termios trm, old;
	struct stat st;
	void (*old_alarm)(int);
	int old_alarm_value = 0;
	int fd = -1;
	int ret = 0;
	char *buffer;

	buffer = mdmalloc(BUFSIZE);

	if (!buffer)
		return -EMDMEM;

#if defined(MDM_CHECK_DEVICES)
reopen_line:
#endif

	if ((ret = mdm_lock(device)) < 0) {
		mdmfree(buffer);
		return ret;
	}

	if (setjmp(not_opened) == 0) {
/*
 * build the modem device filename
 */
		sprintf(buffer, "/dev/%s", device->line);

/*
 * set up the alarm for jump if timeout
 */
		old_alarm_value = alarm(0);
		old_alarm = signal(SIGALRM, alarm_jump);
		alarm(2);

		fd = open(buffer, O_RDWR|O_NOCTTY|O_NDELAY);
		if (fd >= 0) {
			int n = 0;
/*
 * Cancel the O_NDELAY flag.
 */
#ifdef SBAGLIATO
			n = fcntl(fd, F_GETFL, 0);
#else
			n = fcntl(fd, F_GETFL);
#endif
			n = fcntl(fd, F_SETFL, n & ~O_NDELAY);
		}

	}
/*
 * Restore the old alarm handler and scheduled timeout
 */
	alarm(0);
	signal(SIGALRM, old_alarm);
	alarm(old_alarm_value);
	if (fd < 0) {
		mdmfree(buffer);
		return (mdmerrno = -EMDMOPEN);
	}

/*
 * fstat it to get informations
 */
	if (fstat(fd, &st) == -1) {
		mdm_unlock(device);
		close(fd);
		mdmfree(buffer);
		return (mdmerrno = -EMDMSTAT);
	}
/*
 * Now check if it is a character device.
 */
	if (!S_ISCHR(st.st_mode)) {
		mdm_unlock(device);
		close(fd);
		mdmfree(buffer);
		return (mdmerrno = -EMDMNOCHR);
	}

#if defined(MDM_CHECK_DEVICES)
/*
 * Check if we are using a callout device. If not, close the ttyS device and
 * change it with the cua one.
 */

/*
 * Now check the minor device (is better to check the major one after minor)
 *
 * Warning: Linux specific.
 */
#define MDM_MINOR_START		64
#define MDM_MINOR_END		127

	if ((MINOR(st.st_rdev) < MDM_MINOR_START) ||
	    (MINOR(st.st_rdev) > MDM_MINOR_END)) {
		mdm_unlock(device);
		close(fd);
		mdmfree(buffer);
		return (mdmerrno = -EMDMBADMIN);
	}

/*
 * Now check if major is okay:
 * major == TTYAUX_MAJOR (callout device). If == TTY_MAJOR, reopen callout one.
 */
	 if ((MAJOR(st.st_rdev) != TTY_MAJOR) &&
	     (MAJOR(st.st_rdev) != TTYAUX_MAJOR)) {
		mdm_unlock(device);
		close(fd);
		mdmfree(buffer);
		return (mdmerrno = -EMDMBADMAJ);
	}

/*
 * Now check if major == TTY_MAJOR, if so, close, change device and reopen.
 */
	if (MAJOR(st.st_rdev) == TTY_MAJOR) {
		mdm_unlock(device);
		close(fd);
/*
 * Now we are going to change the device from ttyS one to cua one.
 * The problem is that we are going to lock /dev/cua?? so if another person
 * will attempt to open the same /dev/ttyS?? line, we will have a timeout
 * error on open, and only on the second (right) open, (open of /dev/cua??)
 * we will find the lock of the line.
 */
		sprintf(device->line, "cua%d", MINOR(st.st_rdev));
		goto reopen_line;
	}
#endif /* MDM_CHECK_DEVICES */
/*
 * Set up the line for working with modem (POSIX termios)
 * Retry 60 times (1 sec sleep) while tcgetattr succedes.
 */
	{
		int retr=0, result=0;

		while (retr < 60) {
			if (tcgetattr(fd, &trm) != -1) {
				result = 1;
				break;
			}
			sleep(1);
			retr++;
		}
		if (result == 0) {
			mdm_unlock(device);
			close(fd);
#if defined(DEBUG)
			mdm_log(LOG_DEBUG,
				"tcgetattr retries: too many, dropping line\n");
#endif /* DEBUG */
			mdmfree(buffer);
			return (mdmerrno = -EMDMGETA);
		}
#if defined(DEBUG)
		mdm_log(LOG_DEBUG, "tcgetattr retries: %d\n", retr);
#endif /* DEBUG */
	}

/*
 * save tty state
 */
	memcpy(&savetty, &trm, sizeof(struct termios));
	memcpy(&old, &trm, sizeof(struct termios));

	set_mode_sane(&trm, device, 1);
	set_default_cc(&trm);
	set_flow_control(device, &trm);

/*
 * Toggle DTR
 */
	cfsetospeed(&trm, B0);
	tcsetattr(fd, TCSANOW, &trm);
	sleep(1);
	tcsetattr(fd, TCSANOW, &old);

/*
 * set speed
 */
	if (device->bd < baud) {
		mdm_unlock(device);
		close(fd);
		mdmfree(buffer);
		return (mdmerrno = -EMDMBAUD);
	}
	cfsetospeed(&trm, device->bd);
	cfsetispeed(&trm, device->bd);

/*
 * set the new attributes, flush output and discard all not read input
 */
	if (tcsetattr(fd, TCSANOW, &trm) == -1) {
		mdm_unlock(device);
		close(fd);
		mdmfree(buffer);
		return (mdmerrno = -EMDMSETA);
	}

	clean_line(fd, 1);

	mdmfree(buffer);

	return fd;
}

#define ISCRLF(x) ((x == '\n') || (x == '\r'))

static int get_command(char *buffer, char *cmd) {
	char *nbuffer;
	char *p = buffer;

	nbuffer = mdmalloc(BUFSIZE);

	if (!nbuffer)
		return -EMDMEM;

	/*
	 * Strip leading spaces.
	 */
	while (ISCRLF(*p))
		++p;

	/*
	 * Copy the stripped string in local buffer
	 */
	strcpy(nbuffer, p);

	p = nbuffer;

	while (*p) {
		if (ISCRLF(*p)) {
			/*
			 * fill cmd with the string found.
			 */
			*p = '\0';
			strcpy(cmd, nbuffer);

			p++;
			
			/*
			 * Copy the remaining string into the original
			 * modem (read) buffer updating it.
			 */
			strcpy(buffer, p);
			mdmfree(nbuffer);
			return SUCCESS;
		}
		++p;
	}

	mdmfree(nbuffer);
	return FAILURE;
}


/* Angelo: 
 *
 * Problem with this routine:
 * When we send command strings to the modem
 * We are expecting return values from the modem in the form
 * of simple strings.
 * 
 * When we send the ATDT command we expect a CONNECT
 * string to be returned, if we write the ATDT command and read
 * the result into a buffer - reading 1000 charaters we get the
 * result and potentially any other data that may follow. It is 
 * a very bad idea to throw away this data as an application
 * will never see it!!!
 * 
 * The solution is to read the result a character at a time 
 * stopping as soon as we get the response string we are looking
 * for.
 */
static
int talk_to_modem(int fd, struct modems *device, char *command) {
	fd_set readfds;
	struct timeval tmout;
	struct mdm_reply_s *mr;
	int ret;
	int rings;
	int cmd_found = FAILURE;
	int mdm_read_retries = READ_RTRY;
	char cfm;
	char *cmdbuffer;
	char *cmdmodem;

	if (device->rings == 0) {
		/*
		 * If this is the case, rings should not be counted.
		 * Since we do rings--, setting it to -1 means that it is
		 * infinite and that's what we want.
		 */
		rings = -1;
	}

	cmdbuffer = mdmalloc(BUFSIZE);

	if (!cmdbuffer)
		return -EMDMEM;

	cmdmodem = mdmalloc(BUFSIZE);

	if (!cmdmodem) {
		mdmfree(cmdbuffer);
		return -EMDMEM;
	}

	mdm_log(LOG_INFO, "writing >> %s << to modem\n", command);

/*
 * write the command to modem
 */

	if (write(fd, command, strlen(command)) != strlen(command)) {
		mdm_log(LOG_INFO, "Error [%s] while writing to modem\n",
				strerror(errno));
		mdmfree(cmdbuffer);
		mdmfree(cmdmodem);
		return (mdmerrno = -EMDMWRITE);
	}

#if defined(DEBUG)
	mdm_log(LOG_DEBUG, "written >> %s << to modem\n", command);
#endif /* DEBUG */

    /*
     * Seems that we have to sleep here (but should be checked).
     */
	sleep (1);

	while (!cmd_found) {

		/*
		 * If number of RINGING messages reached the maximum allowed,
		 * return now with error EMDMRINGING.
		 */
		if (rings == 0)
			return -EMDMRINGING;

		cfm = '\0';
		FD_ZERO(&readfds);
		FD_SET(fd, &readfds);
		tmout.tv_sec = device->dl;
		tmout.tv_usec = 0;

		/*
		 * We have to sleep here to sync with modem.
		sleep(1);
		 */


		ret = select(FD_SETSIZE, &readfds, NULL, NULL, &tmout);

		if (ret < 0) {
			mdmfree(cmdbuffer);
			mdmfree(cmdmodem);
			return (mdmerrno = -EMDMSEL);
		}

		if (!FD_ISSET(fd, &readfds)) {
			mdmfree(cmdbuffer);
			mdmfree(cmdmodem);
			return (mdmerrno = -EMDMNOFDSET);
		}
        
		/* Angelo:
		 *
		 * Data is waiting to be read from fd
		 * read only 1 character and append to buf. 
		 * Reading more than 1 character will potentially
		 * eat into data being sent from remote machine
		 */
		ret = read(fd, &cfm, 1);
		
		if (ret < 0) {
			mdmfree(cmdbuffer);
			mdmfree(cmdmodem);
			return (mdmerrno = -EMDMREAD);
		}
        
#if defined(DEBUG)
		mdm_log(LOG_DEBUG, "read >> %c << from modem\n",
			(isspace(cfm) ? ' ' : cfm));
#endif /* DEBUG */

		if (ret == 0) {
			mdm_log(LOG_INFO, "read returned 0 bytes (EOF)\n");
			mdmfree(cmdbuffer);
			mdmfree(cmdmodem);
			return (mdmerrno = -EMDMRDRT);
		}
		
		strncat(cmdbuffer, &cfm, 1);

		while (get_command(cmdbuffer, cmdmodem) == SUCCESS) {
/*
 * Reset read retries: we have found something.
 */
			mdm_read_retries = READ_RTRY;

			mdm_log(LOG_INFO, "found >> %s << from modem\n",
					cmdmodem);

			/*
			 * If we have read back the command just written,
			 * ignore it and search again for the modem response
			 * The strlen(command)-1 is caused by the fact that
			 * get_command returns a string without heading or
			 * trailing spaces while command is a string without
			 * leading spaces, but with a trailing '\r' so in the
			 * compare we have to remove the '\r'.
			 */
			if (!strncmp(command, cmdmodem, strlen(command)-1))
				continue;

			if ((mr = check_for_string(cmdmodem)) != NULL) {
				cmd_found = 1;

				/*
				 * Now handle 'RINGING' response.
				 */
				if (mr->r_err == EMDMRINGING)
					rings--;
			}

		}

		/*
		 * If we have found a cmd, don't exit with error if we are at
		 * the last retry.
		 */
		if (!cmd_found) {
			if (!(mdm_read_retries--)) {
				mdmfree(cmdbuffer);
				mdmfree(cmdmodem);
				return (mdmerrno = -EMDMRDRT);
			}
		}
	}

/*
 * The modem returned a good reply. Return the right error code.
 */
	mdmfree(cmdbuffer);
	mdmfree(cmdmodem);
	return (mdmerrno = -(mr->r_err));
}

static struct mdm_reply_s * check_for_string(char *from) {
	int i;

	/*
	 * Sanity check
	 */
	if ((from == NULL) || (*from == '\0'))
		return NULL;

	/*
	 * First of all search for a known modem reply starting with
	 * second mdm_reply entry.
	 */
	for (i=1; mdm_reply[i].r_str; i++)
		if (!strncasecmp(from, mdm_reply[i].r_str,
				strlen(mdm_reply[i].r_str))) {
			return & mdm_reply[i];
		}

	/*
	 * No known modem reply found. Return "Unknown error":
	 * the mdm_reply[0] entry.
	 */
	return & mdm_reply[0];
}

int close_nohangup(int fd) {
	struct termios trm;

	tcgetattr(fd, &trm);
	trm.c_cflag &= ~HUPCL;
	tcsetattr(fd, TCSANOW, &trm);

	close(fd);
/*
 * fd closed, but modem still on-line
 */
	return (mdmerrno = -ENOMDMERR);
}

int hangup(int fd) {
	char *buffer;
	int n, mdmerr;

	if (fd < 0)
		return (mdmerrno = -EMDMBADFD);

	if (mdmopendevice == NULL)
		return hangup_nodev(fd);

	if (mdmopendevice->connected) {
		buffer = malloc(BUFSIZE);

		if (!buffer)
			return -EMDMEM;

/*
 * ATTENTION+HANGUP string with nodelay, because we don't want block here
 * and sleep for allow modem to hangup
 */
/*
 * Set the O_NDELAY flag.
 */
		n = fcntl(fd, F_GETFL, 0);
		n = fcntl(fd, F_SETFL, n | O_NDELAY);

		sprintf(buffer, "%s%s%s%s%c", mdmopendevice->at,
				mdmopendevice->cp,
				mdmopendevice->hu, mdmopendevice->ce,
				MODEM_RETURN);

/*
 * Write to modem the +++ATH0, but save the mdmerrno because this is the end
 * and we are not interested in the error condition aftre talk.
 */
		mdmerr = mdmerrno;
		talk_to_modem(fd, mdmopendevice, buffer);
/*
 * Restore original modem error
 */
		mdmerrno = mdmerr;

/*
 * Cancel the O_NDELAY flag.
 */
		n = fcntl(fd, F_GETFL, 0);
		n = fcntl(fd, F_SETFL, n & ~O_NDELAY);

		mdmfree(buffer);
	}

	sleep(2);

/*
 * Last part of hangup procedure in another function.
 */
	return hangup_nodev(fd);
}

int hangup_nodev(int fd) {
	struct termios trm;

/*
 * hangup with termios (speed to B0). just to be sure :)
 */
	if (tcgetattr(fd, &trm) != -1) {
		struct termios old_trm = trm;

		cfsetospeed(&trm, B0);
/*
 * hangup NOW lowering DTR (0 speed) and sleep 2 seconds to allow modem
 * to hangup
 */
		tcsetattr(fd, TCSANOW, &trm);
		sleep(2);
		tcsetattr(fd, TCSANOW, &old_trm);
	}

#if 1
/*
 * reset line stat if called from hangup()
 */
	if (mdmopendevice != NULL)
		tcsetattr(fd, TCSANOW, &savetty);
#endif

/*
 * close and unlock
 */
	close(fd);

	if (mdmerrno < 0)
		return mdmerrno;
	return (mdmerrno = -ENOMDMERR);
}

struct modems *getopenmodem(void) {
	static struct modems openmdm;

	if (mdmopendevice != NULL) {
		memcpy(&openmdm, mdmopendevice, sizeof(struct modems));
		return &openmdm;
	}
	else
		return NULL;
}

int clean_line(int fd, int waittime) {
	char buffer[2];
	int bytes = 0;
	struct termios tty, oldtty;

	if (tcgetattr(fd, &tty) == -1) {
#if defined(DEBUG)
		mdm_log(LOG_DEBUG, "tcgetattr: cannot clean line\n");
#endif /* DEBUG */
		return (mdmerrno = -EMDMGETA);
	}
   
	memcpy(&oldtty, &tty, sizeof(struct termios));

	tty.c_lflag &= ~ICANON;
	tty.c_cc[VMIN] = 0;
	tty.c_cc[VTIME] = waittime;

	tcsetattr(fd, TCSANOW, &tty);

	while ( read( fd, buffer, 1 ) > 0 && bytes < 10000 )
		bytes++;

	if (bytes && bytes < 50)
		mdm_log(LOG_INFO, "clean_line: got %d bytes [%s]\n",
				bytes, buffer);
	else
		mdm_log(LOG_INFO, "clean_line: got %d bytes [not logged]\n",
				bytes);
/*
 * reset terminal settings
 */
	tcsetattr(fd, TCSANOW, &oldtty);
    
	return SUCCESS;
}
