/* $Id: launch.c,v 1.3 2001/02/04 21:22:30 rsmith Exp $
 * -*- c -*-
 * This file implements functions to launch and kill a child process.
 * Copyright (C) 2000-2001 Roland F. Smith <rsmith@xs4all.nl>
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this code; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * $Log: launch.c,v $
 * Revision 1.3  2001/02/04 21:22:30  rsmith
 * Do not modify pprogram->name in kill_program.
 *
 * Revision 1.2  2001/01/14 11:17:42  rsmith
 * Added debugging statments.
 *
 * Revision 1.1  2001/01/10 20:05:30  rsmith
 * Initial revision
 *
 * 
 */

#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/wait.h>
#include <unistd.h>		/* for pipe(), fork() and pid_t */
#include <errno.h>		/* for errno */
#include <fcntl.h>		/* for fcntl() */
#include <signal.h>		/* for signal() */
#include <stdlib.h>		/* for malloc() */
#include "launch.h"

#ifndef NULL
#define NULL (void*)0
#endif

#ifndef RETURN_IF_FAIL
#define RETURN_IF_FAIL(test) if (!(test)) return
#endif

#ifndef RETURN_VAL_IF_FAIL
#define RETURN_VAL_IF_FAIL(test,val) if (!(test)) return ((val))
#endif

#ifndef NDEBUG
/* __FILE__ is a predefined cpp macro, __FUNCTION__ is a GCC feature. */
#undef debug
#define debug(TXT) fprintf(stderr, "file %s, function %s: %s\n", __FILE__, __FUNCTION__, TXT)
#else
#define debug(TXT) (void)0
#endif /* NDEBUG */


static int array_size;
static int array_used;
const static int array_grow = 100;
static program_t **ptr_array = NULL;

static int store_program_ptr (program_t *pp);
static int grow_array (void);
static program_t *free_program_ptr (pid_t pid);

static void sigchild_handler (int num, siginfo_t *info, void *ptr);

int
launch_program (program_t *pprogram)
{
  int rv;
  static int hreg;
  struct stat namestat;
  mode_t othermode = S_IROTH | S_IXOTH;
  int parent_read[2];
  int parent_write[2];
  pid_t pid;
  struct sigaction newact;
  char *args[2];

  if (hreg == 0)
    {
      /* register our signal handler once */
      hreg = 1;
      newact.sa_handler = NULL;
      newact.sa_sigaction = sigchild_handler;
      sigemptyset (&newact.sa_mask);
      sigaddset (&newact.sa_mask, SIGCHLD);
      newact.sa_flags = SA_NOCLDSTOP | SA_SIGINFO;
      newact.sa_restorer = NULL;
      if (sigaction (SIGCHLD, &newact, NULL) == -1)
        {
          debug ("installing sigchild handler failed.");
          return -1;		/* can't install signal handler */
        }
    }
  if (pprogram == 0)
    {
      debug ("invalid (NULL) pprogram parameter");
      return -2;			/* invalid parameter */
    }
  if (pprogram->name == 0)
    {
      debug ("pprogram->name is NULL");
      return -3;			/* name is NULL */
    }
  /* see if the file can be run, before we fork */
  if (stat (pprogram->name, &namestat))
    {
      debug ("can't stat the file.");
      return -4;		/* can't stat the file */
    }
  /* Usually executables are owned by root, and we're executing them as an
   * other user, so this check suffices. FIXME for root? */
  if ((namestat.st_mode & othermode) != othermode)
    {
      debug ("can't execute the file.");
      return -5;		/* can't execute the file */
    }
  if (store_program_ptr (pprogram))
    {
      debug ("memory allocation failed.");
      return -6;		/* memory allocation failed */
    }
  /* create pipes */
  rv = pipe (parent_read);
  if (rv)
    {
      debug ("can't create parent_read pipes");
      return -7;		/* can't create pipes */
    }
  rv = pipe (parent_write);
  if (rv)
    {
      debug ("can't create parent_write pipes");
      close (parent_read[0]);
      close (parent_read[1]);
      return -7;
    }
  /* create new process */
  errno = 0;
  pid = fork ();
  if (pid == -1)
    {
      close (parent_read[0]);
      close (parent_read[1]);
      close (parent_write[0]);
      close (parent_write[1]);
      debug ("fork failed");
      return -8;		/* the fork failed */
    }
  else if (pid == 0)
    {
      /* we're in the forked process now. */
      args[0] = pprogram->name;
      args[1] = 0;
      /* close the descriptors we don't need; they're for the parent */
      close (parent_read[0]);
      close (parent_write[1]);
      /* move the others to the server's stdin/stdout */
      if (dup2 (parent_write[0], fileno (stdin)) == -1)
        {
          debug ("can't dup parent_write[0] to stdin");
          _exit (1);
        }
      if (dup2 (parent_read[1], fileno (stdout)) == -1)
        {
          debug ("can't dup parent_read[1] to stdout");
          _exit (1);
        }
      /* make sure that close-on-exec isn't set for these descriptors */
      if (fcntl (parent_write[0], F_SETFD, 0) == -1)
        {
          debug ("can't unset close-on-exec flag for parent_write[0]");
          _exit (1);
        }
      if (fcntl (parent_read[1], F_SETFD, 0) == -1)
        {
          debug ("can't unset close-on-exec flag for parent_read[1]");
          _exit (1);
        }
      /* execute the program in the forked process */
      execve (pprogram->name, args, 0);
      debug ("execve failed.");
      _exit (1);		/* this should not be reached */
    }
  else
    {
      /* still in the parent process */
      /* return the program's pid */
      pprogram->pid = pid;
      /* return the file descriptors that are linked to the program's
       * stdin and stdout */
      pprogram->fd_from = parent_read[0];
      pprogram->fd_to = parent_write[1];
      /* close the other descriptors in the parent process */
      close (parent_read[1]);
      close (parent_write[0]);
    }
  return 0;
}

void 
kill_program (program_t *pprogram)
{
  if (pprogram)
    {
      pprogram = free_program_ptr (pprogram->pid);
      if (pprogram)
        {
          close (pprogram->fd_from);
          close (pprogram->fd_to);
          kill (pprogram->pid, SIGTERM);
          waitpid (pprogram->pid, NULL, WNOHANG);
          pprogram->pid = 0;
          /* the server is killed by the parent process, so we don't invoke
           * the notify callback here. */
        }
      else
        {
          debug ("invalid pprogram pointer.");
        }
    }
}

static int 
store_program_ptr (program_t *pp)
{
  int n = 0;
  program_t **pptr;

  if (pp == NULL)
    {
      return -1;        /* invalid parameter */
    }
  if (array_used == array_size)
    {
      if (grow_array ())
        {
          return -2;      /* memory allocation failed */
        }      
    }
  for (n = 0, pptr = ptr_array; n < array_size; n++, pptr++)
    {
      if (*pptr == NULL)
        {
          *pptr = pp;
          array_used++;
        }
    }
  return 0;
}

static int 
grow_array (void)
{
  program_t **ppnew;

  if (ptr_array == NULL)
    {
      ptr_array = calloc (array_grow, sizeof(program_t *));
      if (ptr_array)
        {
          array_size = array_grow;
          array_used = 0;
        }
      else
        {
          return -1;    /* memory allocation failed */
        }
    }
  else
    {
      ppnew = realloc (ptr_array, 
                       (array_size+array_grow)*sizeof(program_t *));
      if (ppnew)
        {
          ptr_array = ppnew;
          array_size += array_grow;
        }
      else
        {
          return -1;    /* memory allocation failed */
        }
    }
  return 0;             /* normal return */
}

static program_t *
free_program_ptr (pid_t pid)
{
  int n = 0;
  program_t **pptr;
  program_t *ret = NULL;
  
  for (n = 0, pptr = ptr_array; n < array_size; n++, pptr++)
    {
      if (*pptr)
        {
          if ((*pptr)->pid == pid)
            {
              ret = *pptr;
              *pptr = NULL;
              array_used--;
            }
        }
    }
  if (array_used == 0)
    {
      free (ptr_array);
      ptr_array = NULL;
      array_size = 0;
    }
  return ret;
}

static void
sigchild_handler (int num, siginfo_t * info, void *ptr)
{
  program_t *pprogram;

  if (num != SIGCHLD)
    {
      return;
    }
  pprogram = free_program_ptr (info->si_pid);
  if (pprogram)
    {
      close (pprogram->fd_from); pprogram->fd_from = -1;
      close (pprogram->fd_to); pprogram->fd_to = -1;
      waitpid (pprogram->pid, NULL, WNOHANG);
      pprogram->pid = 0;
    }
}
