///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998, 1999, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#ifndef _THREADHEAP_H_
#define _THREADHEAP_H_

#include "config.h"

#include <string.h>

#include "heap.h"

class processHeap; // forward declaration

//
// We use one threadHeap for each thread (processor).
//

class threadHeap : public hoardHeap {

public:

  threadHeap (void);

  // Memory allocation and deallocation routines.
  void * malloc (const size_t sz);
  inline void * memalign (size_t alignment, size_t sz);
  void free (void * ptr);

  // Find out how large an allocated object is.
  inline static size_t objectSize (void * ptr);

  // Set our process heap.
  inline void setpHeap (processHeap * p);

private:

  // Prevent copying and assignment.
  threadHeap (const threadHeap&);
  const threadHeap& operator= (const threadHeap&);

  processHeap *	pHeap;				// Our process heap.
  double _pad[CACHE_LINE / sizeof(double)];	// Cache pad.
};


void * threadHeap::memalign (size_t alignment,
			     size_t size)
{
  // Calculate the amount of space we need
  // to satisfy the alignment requirements.

  while (alignment != align (alignment))
    alignment <<= 1;

  size = size + alignment - 1;

  // Now malloc the space up with a little extra
  // (we'll put the block pointer in right behind
  // the allocated space).

  void * ptr = malloc (size + sizeof(block));

  void * newptr = (void *)
    ((((unsigned int) ptr + alignment - 1) / alignment) * alignment);

  // Copy the block from the start of the allocated memory.

  block * b = ((block *) ptr - 1);
  block * p = ((block *) newptr - 1);
  memcpy (p, b, sizeof(block));

  return newptr;
}


size_t threadHeap::objectSize (void * ptr) 
{
  // Find the superblock pointer.
  
  block * b = ((block *) ptr - 1);
  assert (b->isValid());
  superblock * sb = b->getSuperblock ();
  assert (sb);
  
  // Return the size.
  return sizeFromClass (sb->getBlockSizeClass());
}


void threadHeap::setpHeap (processHeap * p) 
{
  pHeap = p; 
}


#endif // _THREADHEAP_H_

